# coding: utf-8

from __future__ import unicode_literals

import travel.rasp.admin.scripts.load_project  # noqa

import argparse
import codecs
import logging
import re
import os
from datetime import datetime
from traceback import format_exc

from django.utils.encoding import force_text

from common.models.schedule import Supplier
from common.utils.fileutils import smart_split_ext
from common.utils.metrics import task_progress_report
from travel.rasp.admin.lib.exceptions import FormattingException
from travel.rasp.admin.lib.logs import create_current_file_run_log, get_collector_context, print_log_to_stdout, get_script_log_context, ylog_context
from travel.rasp.admin.lib.mail import mail_train_import
from travel.rasp.admin.scripts.schedule.tis_train.check_deluxe_trains import get_deluxe_trains_errors
from travel.rasp.admin.scripts.schedule.tis_train.importer import TisImporter
from travel.rasp.admin.scripts.schedule.tis_train.parser import parse_train_file
from travel.rasp.admin.scripts.schedule.tis_train.utils import get_tis_filelist, get_tis_file
from travel.rasp.admin.scripts.utils.file_wrapper.uploaders import upload_tmp_schedule
from travel.rasp.admin.scripts.utils.import_file_storage import get_all_temporary_date_files, get_schedule_temporary_date_filepath


log = logging.getLogger(__name__)


TRAIN_URL = 'ftp://yandex:yandex456@213.156.129.26/TRAIN'
TRAIN_OUT_FILENAME = 'train.txt'


def good_filepath(filename):
    name, ext = smart_split_ext(filename)

    return '{}_good{}'.format(name, ext)


def is_good_file(filepath):
    no_ext_path, ext = smart_split_ext(filepath)

    return no_ext_path.endswith('_good')


def mark_as_good_file(filepath):
    if is_good_file(filepath):
        return

    os.rename(filepath, good_filepath(filepath))


def get_last_train_file_with_date():
    train_files = []
    train_re = re.compile(r'TRAIN\.\d{8}')
    for name, url in get_tis_filelist():
        if train_re.match(name):
            d = datetime.strptime(name, 'TRAIN.%Y%m%d').date()
            train_files.append((d, url))

    if not train_files:
        raise FormattingException('Ни одного файла типа TRAIN.YYYYmmdd')

    train_files.sort(key=lambda f: f[0])
    return train_files[-1]


def download_train_file():
    last_date, last_train_url = get_last_train_file_with_date()

    out_file = get_schedule_temporary_date_filepath(TRAIN_OUT_FILENAME, Supplier.objects.get(code='tis'),
                                                    today=last_date)
    if os.path.exists(out_file):
        log.info('Файл уже скачан %s', out_file)
        return out_file

    if os.path.exists(good_filepath(out_file)):
        out_file = good_filepath(out_file)
        log.info('Файл уже скачан и был однажны успешно импортирован %s', out_file)
        return out_file

    log.info('Качаем файл %s', out_file)
    wget, stdout, stderr = get_tis_file(last_train_url, out_file=out_file)

    if wget.returncode != 0:
        raise FormattingException('Не удалось скачать TRAIN от ТИС: %s', stderr)

    log.info('Успешно скачали файл %s', out_file)
    upload_tmp_schedule(Supplier.objects.get(code='tis'), today=last_date)

    return out_file


def get_tis_file_variants():
    tis_files = get_all_temporary_date_files(TRAIN_OUT_FILENAME, Supplier.objects.get(code='tis'))
    good_tis_files = get_all_temporary_date_files(good_filepath(TRAIN_OUT_FILENAME), Supplier.objects.get(code='tis'))
    result_tis_files = []

    for date_, filepath in tis_files.items():
        result_tis_files.append(dict(good=False, path=filepath, date_=date_))

    for date_, filepath in good_tis_files.items():
        result_tis_files.append(dict(good=True, path=filepath, date_=date_))

    result_tis_files.sort(key=lambda x: x['date_'], reverse=True)

    if not result_tis_files:
        return None, None

    fresh_file = result_tis_files[0]['path']
    backup_file = None
    backup_files = [filedict for filedict in result_tis_files[1:] if filedict['good']]
    if backup_files:
        backup_file = backup_files[0]['path']

    return fresh_file, backup_file


def import_from_file(filepath):
    with codecs.open(filepath, 'r', '1251') as f:
        importer = TisImporter(parse_train_file(f))
        return importer.do_import()


def try_import(filepath):
    log.info('Импортируем файл %s', filepath)
    try:
        route_count = import_from_file(filepath)
        if route_count:
            mark_as_good_file(filepath)
        return route_count
    except Exception:
        log.exception('Ошибка импортирования ТИС из файла %s', filepath)
        mail_train_import(
            'schedule_tis_train.py: import error',
            'Ошибка импортирования из файла(попробуем взять предыдущий) %s:\n%s' %
            (filepath, force_text(format_exc(), 'utf8', errors='ignore'))
        )
        return 0


def do_import():
    with get_collector_context('', format=u'%(message)s', level=logging.ERROR) as collector:
        fresh_file, backup_file = get_tis_file_variants()

        if not fresh_file:
            mail_train_import('Rasp: schedule_tis_train NO FILE!!!', 'Ничего не импортировали')
            raise FormattingException('Нет файлов для импорта')

        route_count = try_import(fresh_file)
        if not route_count:
            log.error('Свежий файл пустой %s', fresh_file)
            mail_train_import(
                'Rasp: schedule_tis_train NO NEW FILE!!!',
                'Свежий файл пустой или невалидный {}'.format(fresh_file)
            )
            if backup_file:
                route_count = try_import(backup_file)
            else:
                mail_train_import(
                    'schedule_tis_train.py: import error',
                    'Не нашли предыдущего хорошего файла для импорта'
                )
                raise FormattingException('Не нашли бекап файла для импорта')

        if not route_count:
            mail_train_import('Rasp: schedule_tis_train NO DATA!!!', 'Ничего не импортировали')
            raise FormattingException('Не импортировали ни одного маршрута')

        log.info('Загрузили %s маршрутов', route_count)

        message = '\n'.join(list(collector.get_collected().splitlines()))

        deluxe_message = get_deluxe_trains_errors()
        if deluxe_message:
            message = '{}\n{}'.format(deluxe_message, message)

        try:
            mail_train_import('Ошибки при импорте от ТИС', message)
        except Exception:
            log.exception('Ошибка при отправке письма')

        # Обновляем дату последнего импорта от tis
        Supplier.objects.get(code='tis').save()
        log.info('Done')


def update_tis():
    download_train_file()
    do_import()


if __name__ == '__main__':
    with ylog_context(**get_script_log_context()), task_progress_report('import_tis'):
        create_current_file_run_log()

        parser = argparse.ArgumentParser()
        parser.add_argument('-v', '--verbose', action='store_true')
        parser.add_argument('action', choices=(
            'import', 'download', 'download-and-import', 'import-from-file'
        ))
        parser.add_argument('--file')

        args = parser.parse_args()
        if args.verbose:
            print_log_to_stdout()

        if args.verbose:
            print_log_to_stdout()

        if args.action == 'import':
            do_import()

        elif args.action == 'download':
            download_train_file()

        elif args.action == 'download-and-import':
            download_train_file()
            do_import()

        elif args.action == 'import-from-file':
            import_from_file(args.file)

        else:
            parser.error('Unrecognized action')
