# coding: utf-8

from __future__ import unicode_literals

import logging
from collections import OrderedDict
from datetime import time, date, timedelta

from django.utils.functional import cached_property

from common.models.schedule import RThreadType, TrainSchedulePlan
from common.models.transport import TransportType
from travel.rasp.library.python.common23.date import environment
from travel.rasp.admin.lib.exceptions import FormattingException
from travel.rasp.admin.lib.import_bounds import import_bounds
from travel.rasp.admin.lib.mask_builder.bounds import MaskBounds
from travel.rasp.admin.lib.mask_builder.standard_builders import empty_mask, mask_from_day_condition, daily_mask
from travel.rasp.admin.scripts.schedule.tis_train.utils import ExpressStationsGetter


log = logging.getLogger(__name__)


def tis_date_to_date(tis_date):
    """
    дата действия данного варианта поезда или ‘95999’ –до конца срока действия того расписания,
    к которому относится начальная дата действия. Две цифры – год, три цифры – номер дня в году.
    :param tis_date: str
    :rtype: date | None
    """

    if tis_date == '95999':
        return

    year = 2000 + int(tis_date[:2])
    day_number = int(tis_date[2:5])

    return date(year, 1, 1) + timedelta(days=day_number - 1)


LETTER_MAPPING = {
    'ZH': 'Ж',
    'CH': 'Ч',
    'JI': 'Й',
    'PC': 'П',
    'JA': 'Я',
    'KH': 'Х',
    'SH': 'Ш',
    'SZ': 'Щ',
    'MZ': 'Ь',
    'EI': 'Э',
    'AJ': 'А',
    'BJ': 'Б',
    'CJ': 'Ц',
    'DJ': 'Д',
    'EJ': 'Е',
    'FJ': 'Ф',
    'GJ': 'Г',
    'IJ': 'И',
    'KJ': 'К',
    'LJ': 'Л',
    'MJ': 'М',
    'NJ': 'Н',
    'OJ': 'О',
    'RJ': 'Р',
    'SJ': 'С',
    'TJ': 'Т',
    'VJ': 'В',
    'UJ': 'У',
    'YJ': 'Ы',
    'Y ': 'Ы',
    'ZJ': 'З',
}


class BadLengthReadError(FormattingException):
    pass


class BadRecordError(FormattingException):
    pass


class BaseRecord(object):
    raw_data = None
    data = None
    fields_mapping = None

    @classmethod
    def is_applicable(cls, line):
        raise NotImplementedError()

    @classmethod
    def unpack_string_using_mapping(cls, line, fields_mapping):
        data = OrderedDict()
        for (start_index, length), field_name in fields_mapping.items():
            value = line[start_index:start_index + length]

            if len(value) != length:
                log.error('Неверная длина записи, не влезло поле "%s", строка "%s"', field_name, line)
                value = None

            data[field_name] = value

        return data

    @classmethod
    def parse(cls, line):
        assert cls.is_applicable(line)

        record = cls()
        record.data = record.unpack_string_using_mapping(line, cls.fields_mapping)
        record.raw_data = line

        return record


class TrainRecord(BaseRecord):
    TRAIN_TYPE_BASIC = '0'
    TRAIN_TYPE_THROUGH_COACH = '1'
    TRAIN_TYPE_COMPLEX_AND_MIXED = '2'
    # start, length: field_name
    fields_mapping = OrderedDict([
        ((5, 3), 'number_digit'),
        ((8, 2), 'number_letter'),

        # 0 – поезд
        # 1 – беспересадочный вагон
        # 2 – вторая и последующая группа поезда, состоящего из нескольких групп вагонов, имеющих разные пункты
        #     назначения или отправления
        ((10, 1), 'train_type'),

        # Дата начала действия данного варианта поезда (две цифры год, три цифры – номер дня в году)
        ((11, 5), 'start_of_schedule'),

        ((16, 7), 'start_station_code'),
        ((23, 7), 'end_station_code'),

        # Категория поезда:
        # 0 – почтовый, багажный
        # 1 – грузо-пассажирский
        # 2 – пассажирский
        # 3 – скорый
        # 4 – скоростной
        ((30, 1), 'train_category'),

        # Комфортность поезда
        # 0 – пригородный
        # 1 – обычный
        # 2 – фирменный («brended»)
        # 3 - смешанные перевозки
        # 4 - автобус
        ((31, 1), 'train_comfort'),

        # Наличие вагона-ресторана (только для поездов отправлением от Московской ж.д., для остальных - 0)
        # 0 – нет, 1 – есть
        ((32, 1), 'has_restaurant'),

        # Наличие купе - буфета(только для поездов отправлением от Московской ж.д.)
        # 0 – нет, 1 – есть
        ((33, 1), 'has_compartment_buffet'),

        # Код железнодорожной администрации принадлежности поезда(UIC 920)
        ((34, 2), 'administrative_code'),

        # Наличие спальных вагонов 1-го класса (СВ) – в  первом байте (0-есть, 1-нет).
        # Во втором байте 1 – при наличии вагонов с продажей мужских-женских купе.
        # (только для поездов отправлением от Московской ж.д., для остальных - 00)
        ((36, 2), 'has_sleep_coach_with_gender'),

        # Аналогично – наличие спальных вагонов категории «Люкс» (купе с душем и туалетом).
        # Второй байт всегда ноль, в этих вагонах купе продается только полностью (оба места в одном билете)
        ((38, 2), 'has_deluxe_with_gender'),

        # Аналогично - наличие спальных вагонов 2-го класса с 3-х местными купе («РИЦ»)
        ((40, 2), 'has_sleep_coach_2_class_3_seater_with_gender'),

        # Аналогично – наличие спальных вагонов 2-го класса с 4-х местными купе («купейный»)
        ((42, 2), 'has_sleep_coach_2_class_4_seater_with_gender'),

        # Аналогично – наличие вагонов для сидения.
        # Первый байт – сидячие вагоны второго класса, второй байт – сидячие вагоны первого класса
        ((44, 2), 'has_seat_coach_2_class_or_1_class'),

        # Аналогично – наличие вагонов для лежания открытого типа («плацкартный»). Второй байт – всегда ноль.
        ((46, 2), 'has_platzkart_coach'),


        # Аналогично – наличие вагонов для сидения открытого типа («общий»)
        ((48, 1), 'has_common_coach'),

        # Резерв
        ((49, 1), 'reserved'),

        # Наличие багажного вагона
        ((50, 1), 'has_baggage_car'),

        # Ежедневный (365 дней в год,у)
        ((51, 1), 'is_every_day_train'),

        # Идентификатор группы вагонов внутри поезда («нитка»)
        ((52, 2), 'coach_group_code'),

        # КОД ПЕРЕВОЗЧИКА (ДВЕ ЦИФРЫ – ПРАВЫЕ ЦИФРЫ КОДА ПЕРЕВОЗЧИКА ИЗ ТАБЛИЦЫ ПЕРЕВОЗЧИКОВ,
        # ПРИМЕНЯТЬ ВМЕСТЕ С КОДОМ ГОСУДАРСТВА ПЕРЕВОЗЧИКА)
        ((54, 2), 'carrier_code')
    ])

    # не беспересадочный - так в документации :)
    not_through_coach_fields_mapping = OrderedDict([
        # Последняя дата действия данного варианта поезда или
        # ‘95999’ – до конца срока действия того расписания, к которому относится начальная дата действия.
        # Две цифры – год, три цифры – номер дня в году.
        ((56, 5), 'end_of_schedule'),

        # Резерв
        ((61, 7), 'reserved_2'),

        # Срок продажи (максимальная дата отправления)
        ((68, 3), 'max_departure'),

        # Резерв
        ((71, 9), 'reserved_3'),

        # список ниток маршрута поезда
        ((80, 16), 'threads_list')
    ])

    # беспересадочный вагон
    through_coach_fields_mapping = OrderedDict([
        # Код станции отцепки
        ((56, 7), 'detach_station_code'),

        # Номер поезда после переприцепки
        ((63, 3), 'after_attaching_number_digit'),
        ((66, 2), 'after_attaching_number_letter'),

        # Резерв
        ((68, 7), 'reserved_3'),

        # Последняя дата действия данного варианта поезда или
        # ‘95999’ –до конца срока действия того расписания, к которому относится начальная дата действия.
        # Две цифры – год, три цифры – номер дня в году.
        ((75, 5), 'end_of_schedule'),

        # Список ниток маршрута поезда
        ((80, 16), 'threads_list')
    ])

    start_day_shift = 0

    def __init__(self):
        self.calendar_records = []
        self.raw_stop_records = []

    @classmethod
    def is_applicable(cls, line):
        return line[:5] == 'TRAIN'

    @classmethod
    def parse(cls, line):
        record = super(TrainRecord, cls).parse(line)  # type: TrainRecord

        if record.data['train_type'] == cls.TRAIN_TYPE_THROUGH_COACH:
            record.data.update(record.unpack_string_using_mapping(line, cls.through_coach_fields_mapping))
        else:
            record.data.update(record.unpack_string_using_mapping(line, cls.not_through_coach_fields_mapping))

        return record

    @property
    def number(self):
        return self.data['number_digit'] + LETTER_MAPPING[self.data['number_letter']]

    @property
    def company_code(self):
        return '{}_{}'.format(self.data['administrative_code'], self.data['carrier_code'])

    @cached_property
    def thread_type(self):
        if self.data['train_type'] == self.TRAIN_TYPE_BASIC:
            return RThreadType.objects.get(pk=RThreadType.BASIC_ID)
        else:
            return RThreadType.objects.get(pk=RThreadType.THROUGH_TRAIN_ID)

    @property
    def t_type_id(self):
        return TransportType.BUS_ID if self.data['train_comfort'] == '4' else TransportType.TRAIN_ID

    @property
    def country_code(self):
        return self.data['administrative_code']

    @property
    def start_station_code(self):
        return self.data['start_station_code']

    @cached_property
    def start_station(self):
        return ExpressStationsGetter.get_station(self.start_station_code)

    @cached_property
    def end_station(self):
        return ExpressStationsGetter.get_station(self.data['end_station_code'])

    @cached_property
    def stop_records(self):
        stop_records = []

        for stop_record in self.raw_stop_records:
            if stop_record.is_city:
                continue

            if stop_record.is_technical and stop_record.stop_time == 0:
                continue

            stop_records.append(stop_record)
        return stop_records

    @property
    def is_every_day_train(self):
        return self.data['is_every_day_train'] == '1'

    @cached_property
    def run_mask(self):
        today = environment.today()

        bounds = MaskBounds(*import_bounds(today=today))

        start_date = tis_date_to_date(self.data['start_of_schedule']) or (today - timedelta(7))
        end_date = (
            tis_date_to_date(self.data['end_of_schedule']) or self.get_schedule_end_date(start_date, bounds)
        )

        if not end_date:
            log.error('Не нашли дату окончания графика для начальной даты %s', start_date)
            return empty_mask(today)

        bounds = bounds.get_narrowed(start_date, end_date)

        every_day_mask = daily_mask(bounds, today=today)

        if not self.calendar_records and self.is_every_day_train:
            return every_day_mask

        run_mask = empty_mask(today)

        for calendar_record in self.calendar_records:
            run_mask |= calendar_record.build_mask(today) & every_day_mask

        return run_mask

    @property
    def record_key(self):
        return '_'.join([self.number, self.raw_data])

    def get_schedule_end_date(self, start_date, bounds):
        for plan in TrainSchedulePlan.objects.all():
            if plan.start_date <= start_date <= plan.end_date:
                return plan.end_date

        if TrainSchedulePlan.objects.exists():
            max_plan_date = max([p.end_date for p in TrainSchedulePlan.objects.all()])
            if start_date > max_plan_date:
                return bounds.end_date


class CalendarRecord(BaseRecord):
    fields_mapping = OrderedDict([
        # Номер месяца (от 01 до 12)
        ((5, 2), 'month_number'),

        # Каждый байт соответствует числу месяца от 1 до 31.
        # Значение 0 – поезд (группа вагонов) в этот день не отправляется от начальной станции,
        #          1 – отправляется.
        # Календарь следует анализировать только в пределах интервала дат действия
        # данного варианта поезда в строке TRAIN.
        # Допускается отсутствие «1» на все даты интервала действия варианта – это означает,
        # что данный вариант поезда ни на одну из дат не назначен.
        ((7, 31), 'month_mask'),
    ])

    @classmethod
    def is_applicable(cls, line):
        return line[:5] == 'KALEN'

    def build_mask(self, today):
        bounds = MaskBounds(*import_bounds(today=today))

        month = int(self.data['month_number'])
        day_mask = self.data['month_mask']

        def day_condition(day):
            return day.month == month and day_mask[day.day - 1] == '1'

        return mask_from_day_condition(bounds, day_condition, today=today)


class StopRecord(BaseRecord):
    fields_mapping = OrderedDict([
        # Код станции (UIC 920)
        ((0, 7), 'station_code'),
        # Расстояние от начала маршрута, км
        ((7, 5), 'distance_from_start'),

        # Время отправления (HHMM), для конечной станции – время прибытия
        ((12, 4), 'departure_time'),

        # Константа времени – количество смен дат, прошедших с момента отправления с начальной станции.
        # Цифра от 0 до 9 или буквы: A – 10, B – 11, C- 12
        ((16, 1), 'date_switch_count'),

        # Время стоянки в минутах
        ((17, 3), 'stop_duration'),

        # 1 – станция является станцией отправления или начального участка для одной из
        # групп вагонов данного поезда («станция первого уровня одной из ниток»)
        ((20, 1), 'coach_group_first_level_station'),

        # 1 – данная строка соответствует коду многовокзального города
        # 2 – техстоянка. Если время техстоянки равно 0, то это означает, что поезд на станции не останавливается,
        # но проходит ее (для дополнительной идентификации маршрута)
        ((21, 1), 'station_type'),

        # 1 – станция является узлом линий (может быть как тарифной остановкой, так и технической)
        ((22, 1), 'is_node_station'),

        # 1 – при отправлении с этой станции инвертируется четность/нечетность номера поезда
        ((23, 1), 'evenness_switch'),

        # Номер поезда отправления со станции
        ((24, 4), 'departure_number'),

        # Номер поезда прибытия на станцию
        ((28, 4), 'arrival_number')
    ])

    @classmethod
    def is_applicable(cls, line):
        """
        У этой записи нет специально идентификатора, поэтому на нее нужно проверять в последнюю очередь.
        На всякий случай сравним длину строки, т.к. она у все записей в файле TRAINS.txt отличается.
        В старых версиях, длина 22, через некоторое время можно совсем выпилить.
        """
        return len(line) in (22, 24, 32)

    @property
    def station_code(self):
        return self.data['station_code']

    @cached_property
    def station(self):
        return ExpressStationsGetter.get_station(self.station_code)

    @property
    def departure_time(self):
        return time(int(self.data['departure_time'][:2]), int(self.data['departure_time'][2:4]))

    @property
    def is_city(self):
        return self.data['station_type'] == '1'

    @property
    def is_technical(self):
        return self.data['station_type'] == '2'

    @property
    def day_shift(self):
        return int(self.data['date_switch_count'], 16)

    @property
    def stop_time(self):
        return int(self.data['stop_duration'])

    @property
    def arrival_number(self):
        return self.data['arrival_number']

    @property
    def departure_number(self):
        return self.data['departure_number']

def parse_train_file(fileobj):
    current_train_record = None
    for line in fileobj:
        line = line.lstrip().rstrip('\n')
        if not line:
            continue
        if TrainRecord.is_applicable(line):
            if current_train_record:
                yield current_train_record
            current_train_record = TrainRecord.parse(line)
        elif CalendarRecord.is_applicable(line):
            current_train_record.calendar_records.append(CalendarRecord.parse(line))
        elif StopRecord.is_applicable(line):
            current_train_record.raw_stop_records.append(StopRecord.parse(line))
        else:
            raise BadRecordError('Can not parse record "%s"', line)

    if current_train_record:
        yield current_train_record
