# coding: utf-8

from __future__ import unicode_literals

import logging
from functools import wraps

from django.utils.encoding import force_text
from lxml import html

from common.models.geo import Station, StationCode
from common.utils.safe_xml_parser import get_html_safe_parser
from travel.rasp.admin.lib.exceptions import FormattingException
from travel.rasp.admin.scripts.schedule.utils.ftp import get_ftp_file


log = logging.getLogger(__name__)

FTP_URL = 'ftp://213.156.129.26/'
FTP_USER = 'yandex'
FTP_PASSWORD = 'yandex456'


class ExpressStationsGetter(object):
    global_caching_getter = None
    _stations_cache = None

    systems = ('express', 'express_ru', 'express_ua')

    def fill_cache(self):
        self._stations_cache = {}
        for system_code in self.systems:
            self._stations_cache[system_code] = {
                sc.code: sc.station
                for sc in StationCode.objects.filter(system__code=system_code).order_by().select_related('station')
            }

    @classmethod
    def get_station(cls, code):
        getter = cls.global_caching_getter or ExpressStationsGetter()

        return getter._get_station(code)

    def _get_station(self, code):
        if self._stations_cache:
            for system_code in self.systems:
                try:
                    return self._stations_cache[system_code][code]
                except KeyError:
                    pass
        else:
            for system_code in self.systems:
                try:
                    return Station.get_by_code(system_code, code)
                except Station.DoesNotExist:
                    pass

        raise Station.DoesNotExist()

    def __enter__(self):
        self.fill_cache()

        ExpressStationsGetter.global_caching_getter = self

    def __exit__(self, exc_type, exc_val, exc_tb):
        ExpressStationsGetter.global_caching_getter = None

    @classmethod
    def cache(cls, func):
        @wraps(func)
        def decorator(*args, **kwargs):
            with ExpressStationsGetter():
                return func(*args, **kwargs)

        return decorator


class TisDownloadError(FormattingException):
    pass


def get_tis_filelist():
    log.info(u"Получаем листинг от ТИС")
    wget, stdout, stderr = get_ftp_file(FTP_URL, ftp_user=FTP_USER, ftp_password=FTP_PASSWORD)

    if wget.returncode != 0:
        raise TisDownloadError(
            u'Не удалось скачать листинг от ТИС - wget return code = %d, stderr: %s', wget.returncode,
            force_text(stderr)
        )

    tree = html.fromstring(stdout, parser=get_html_safe_parser())

    for a in tree.findall('.//a'):
        filename = a.text
        yield filename, FTP_URL + filename


def get_tis_file(url, out_file=None):
    wget, stdout, stderr = get_ftp_file(url, ftp_user=FTP_USER, ftp_password=FTP_PASSWORD, out_file=out_file)
    return wget, stdout, stderr
