# coding: utf8
from __future__ import unicode_literals

import travel.rasp.admin.scripts.load_project  # noqa

import logging
import sys
import os.path
import getopt
from cStringIO import StringIO
from xml.etree import ElementTree

from common.apps.suburban_events import dynamic_params
from common.models.geo import ExternalDirection
from common.models.schedule import Route, Supplier
from common.models.transport import TransportType
from travel.rasp.library.python.common23.date import environment
from travel.rasp.admin.lib.logs import print_log_to_stdout, create_current_file_run_log, get_collector_context, get_script_log_context, ylog_context
from travel.rasp.admin.lib.xmlutils import get_sub_tag_text2
from travel.rasp.admin.importinfo.models.af import AFUpdateFile
from travel.rasp.admin.scripts.schedule.af_processors import suburban as af_suburban, common as af_common
from travel.rasp.admin.scripts.schedule.af_processors.af_thread_parser import AfThreadParser
from travel.rasp.admin.scripts.schedule.af_processors.utils import parse_thread, RZD_COMPANY_ID
from travel.rasp.admin.scripts.utils.to_python_parsers import parse_date
from suburban_change_notices.models import Notice


script_name = os.path.basename(__file__)

t_suburban = TransportType.objects.get(code='suburban')

log = logging.getLogger(__name__)


def update_threads(thread_el, filename, today, recount_schedule_on_the_fly=False, parser=None):
    # Проверяем changemode только в остальных видах траспорта, городской транспорт у нас особенный

    log.info(u"Обрабатываем изменение change_mode=%s, number=%s", thread_el.get('changemode'),
             thread_el.get('number'))

    if parser:
        thread = parser.parse_thread(thread_el)
    else:
        thread = parse_thread(thread_el, default_t_type_code='suburban')

    if thread is None:
        log.info(u"Не смогли разобрать thread %r", thread_el.attrib)
        return

    if thread.t_type == t_suburban:
        af_suburban.process_thread(thread, thread_el, filename, today,
                                   recount_schedule_on_the_fly=recount_schedule_on_the_fly)
    else:
        af_common.process_thread(thread, filename)


def update_from_file(data, filename, recount_schedule_on_the_fly=False):
    today = environment.today()

    thread_parser = AfThreadParser(
        default_t_type_code='suburban',
        default_company_id=RZD_COMPANY_ID,
        default_supplier=Supplier.objects.get(code='af')
    )
    with get_collector_context() as collector:
        log.info(u'Импортируем %s, %s', filename, environment.now())

        tree = ElementTree.parse(StringIO(data))
        update_notices(tree)

        for thread_el in tree.findall('.//thread'):
            try:
                update_threads(thread_el, filename, today,
                               recount_schedule_on_the_fly=recount_schedule_on_the_fly,
                               parser=thread_parser)
            except Exception:
                log.exception(u"Ошибка при импорте машрута %s", repr(thread_el.attrib))

        empty_routes = Route.objects.filter(rthread__isnull=True, t_type__code='suburban')
        if empty_routes:
            for route in empty_routes:
                log.info(u"Удаляем маршрут без ниток %s", route.route_uid)
                route.delete()
        try:
            dynamic_params.set_param('last_afupdatefile', environment.now())
        except Exception as ex:
            log.exception('Не смогли установить флаг в монге. %s', repr(ex))

        return collector.get_collected()


def update_notices(tree):
    for notice_el in tree.findall('./notices/notice'):
        try:
            update_notice(notice_el)
        except Exception:
            log.exception(u'Ошибка разбора объявления')


def update_notice(notice_el):
    title = get_sub_tag_text2(notice_el, 'title').strip()
    text = get_sub_tag_text2(notice_el, 'text').strip()
    mobile_text = get_sub_tag_text2(notice_el, 'mobile_text', u'').strip()
    start_date = parse_date(notice_el.get('start_date'))
    end_date = parse_date(notice_el.get('end_date'))

    log.info(u'Обрабатываем объявление %s от %s до %s',
             title, start_date, end_date)

    directions = []
    for direction_el in notice_el.findall('./directions/direction'):
        try:
            directions.append(ExternalDirection.objects.get(code=direction_el.text.strip()))
        except ExternalDirection.DoesNotExist:
            log.error(u'Не нашли направление по коду %s', direction_el.text)
            return

    if not directions:
        log.error(u'Не указано ни одного направления для объявления %s', title)
        return

    same_notices = Notice.objects.filter(
        title=title,
        text=text,
        mobile_text=mobile_text,
        start_date=start_date,
        end_date=end_date
    )

    if same_notices:
        log.error(u'Объявление %s уже загружено в базу %s',
                  title, u' ,'.join(u'id={} title={}'.format(n.id, n.title) for n in same_notices))
        return

    similar_notices = Notice.objects.filter(title=title, text=text)

    if similar_notices:
        log.error(u'Похожие объявления уже есть в базе %s',
                  title, u' ,'.join(u'id={} title={}'.format(n.id, n.title) for n in similar_notices))

    notice = Notice(
        title=title,
        text=text,
        mobile_text=mobile_text,
        start_date=start_date,
        end_date=end_date
    )

    notice.save()

    notice.directions = list(set(directions))

    log.info(u'Добавили объявление %s', title)


def run():
    for afuf in AFUpdateFile.objects.all():
        if not afuf.loaded:
            update_from_file(afuf.get_encoded_data(), afuf.update_file_name)
            afuf.save(True)


usage = "Usage: python {script_name} [-v]".format(script_name=script_name)

# Если файл запустили из консоли парсим параметры и запускаем импорт
if __name__ == '__main__':
    with ylog_context(**get_script_log_context()):
        create_current_file_run_log()

        error = False
        try:
            options, args = getopt.gnu_getopt(sys.argv[1:], 'v', ['verbose'])
            for option in options:
                key, value = option
                if key in ('-v', '--verbose'):
                    print_log_to_stdout(log)

        except getopt.GetoptError, e:
            error = unicode(e)

        if error:
            log.error(u'Проблема запуска скрипта: %s', error)
            print error
            print usage
            sys.exit(1)

        run()
