# -*- coding: utf-8 -*-
from __future__ import unicode_literals
import logging
from traceback import format_exc

from django.utils.translation import ugettext as _

from common.models.geo import Station
from common.models.schedule import Route, RThread, Company
from common.models.transport import TransportModel
from common.utils.caching import cache_method_result_with_exception, cache_method_result
from travel.rasp.admin.importinfo.models.mappings import MappingContext, TransportModelMapping
from travel.rasp.admin.scripts.schedule.utils.errors import RaspImportError


log = logging.getLogger(__name__)


def get_float(value, default=None):
    if isinstance(value, basestring):
        value = value.replace(',', '.')
        try:
            return float(value)
        except ValueError:
            return default
    return default


def gen_avia_number(company, number):
    if isinstance(company, Company):
        code = company.iata or company.icao or company.sirena_id
    else:
        code = company

    number = code + " " + number

    return number


class TariffMatrix(object):
    """
        A | B | C
    -------------------
    A   x |
    -------------------
    B  10 |
    ------------------
    C  30 | 20

    matrix size 2
    A -> B 10
    A -> C 30
    B -> C 20
    """
    def __init__(self, write_to_log=None):
        self.tariffs = []
        self.log = write_to_log or log

    def set_tariff(self, from_station_index, to_station_index, tariff):
        if len(self.tariffs) <= from_station_index:
            while len(self.tariffs) != (from_station_index + 1):
                self.tariffs.append([])

        from_station_tariffs = self.tariffs[from_station_index]

        if len(from_station_tariffs) <= to_station_index:
            while len(from_station_tariffs) != (to_station_index + 1):
                from_station_tariffs.append(None)

        self.tariffs[from_station_index][to_station_index] = tariff

    def get_tariff(self, from_station_index, to_station_index):
        try:
            return self.tariffs[from_station_index][to_station_index]
        except IndexError:
            return None


PATH_KEY_DELIMITER = "-->"


def get_supplier_station_path_key(path):
    return PATH_KEY_DELIMITER.join(supplier_station.key for supplier_station in path)


class SupplierPath(object):
    def __init__(self, supplier_stations):
        self.supplier_stations = supplier_stations

    def get_path(self):
        return self.supplier_stations

    def get_key(self):
        return get_supplier_station_path_key(self.get_path())

    def __iter__(self):
        return iter(self.get_path())

    def __len__(self):
        return len(self.supplier_stations)

    def __getitem__(self, item):
        return self.supplier_stations[item]


class SupplierRoute(object):
    """
    Представляеет:
    only 1 - route
    only 1 - thread
    """
    data_provider = None
    supplier_title = ""
    supplier_number = ""
    ROUTE_GEN_UID_PARAMS = {'use_stations': True}

    def get_supplier_title(self):
        return self.supplier_title

    def get_supplier_number(self):
        return self.supplier_number

    def get_description(self):
        if hasattr(self, 'route_entity') and hasattr(self.route_entity, 'get_description'):
            return self.route_entity.get_description()
        else:
            return ""

    def __unicode__(self):
        return self.get_description()

    def get_path(self):
        raise NotImplementedError()

    def get_path_group(self):
        return PathGroupAdapter(self)

    def get_path_key(self):
        return get_supplier_station_path_key(self.get_path())

    @cache_method_result_with_exception
    def get_route(self):
        try:
            return self.build_route()
        except RaspImportError:
            raise
        except Exception:
            raise RaspImportError("Ошибка построения маршурта: %s" % format_exc().decode('utf8'))

    def build_route(self):
        route = self.create_route()

        self.add_thread_to_route(route)

        return route

    def build_thread(self):
        raise NotImplementedError()

    def create_route(self):
        route = Route()
        route.supplier = self.data_provider.supplier
        route.two_stage_package = self.data_provider.two_stage_package
        route.t_type = self.data_provider.t_type

        self.route = route

        return route

    def create_thread(self):
        self.thread = thread = RThread()
        thread.type_id = 1
        self.thread.t_type = self.route.t_type
        self.thread.supplier = self.route.supplier

        return thread

    def add_thread_to_route(self, route, override_title=None):
        thread = self.build_thread()

        thread.route = route
        thread.t_type = route.t_type
        self.thread.supplier = self.route.supplier

        route.threads = [thread]

        thread.gen_title()

        if override_title:
            log.info(_('Переписываем название нитки с %s на %s'), thread.title, override_title)

            thread.title = override_title
            thread.is_manual_title = True

        route.route_uid = thread.gen_route_uid(**self.ROUTE_GEN_UID_PARAMS)

        return thread

    def get_tariffs_variants(self):
        return None


def get_start_station(supplier_station, finder):
    try:
        return finder.find_by_supplier_station(supplier_station)
    except Station.DoesNotExist:
        raise RaspImportError("Не нашли начальную станцию")
    except Station.MultipleObjectsReturned:
        raise RaspImportError("Несколько сответствий для начаной станции")


class PathGroupAdapter(SupplierPath):
    def __init__(self, simple_supplier_route):
        self.simple_supplier_route = simple_supplier_route

    def get_path(self):
        return self.simple_supplier_route.get_path()


class SupplierMappingContext(object):
    route_number = ""
    route_title = ""
    first_station_code = ""
    first_station_title = ""
    last_station_code = ""
    last_station_title = ""

    @property
    def key(self):
        return "#".join(getattr(self, f) for f in MappingContext.CONTEXT_FIELDS)

    def __hash__(self):
        return hash(self.key)

    def __eq__(self, other):
        return other.key == self.key

    def __unicode__(self):
        return "<%s: %s>" % (
            self.__class__.__name__,
            ", ".join(["%s=%s" % (f, getattr(self, f)) for f in MappingContext.CONTEXT_FIELDS
                      if getattr(self, f)])
        )


class SupplierTransportModel(object):
    t_type = None
    strange = False
    real_title = None
    title = None
    phone = ""
    code = ""

    def get_key(self):
        return self.title, self.code

    def __hash__(self):
        return hash(self.get_key())

    def __eq__(self, other):
        return other.get_key() == self.get_key()

    def __ne__(self, other):
        return not self.__eq__(other)


class TransportModelFinder(object):
    def __init__(self, supplier):
        self.supplier = supplier

    @cache_method_result
    def find(self, supplier_t_model):
        try:
            return self.find_in_mappings(supplier_t_model)
        except TransportModel.DoesNotExist:
            pass

        try:
            return self.extra_find(supplier_t_model)
        except TransportModel.DoesNotExist:
            pass

        try:
            return self.find_by_title(supplier_t_model)
        except TransportModel.DoesNotExist:
            pass

        if supplier_t_model.title:
            title = supplier_t_model.title
            if supplier_t_model.real_title:
                title = supplier_t_model.real_title

            t_model = TransportModel.objects.create(title=title, t_type=supplier_t_model.t_type)

            tmm = TransportModelMapping.objects.create(supplier=self.supplier, title=supplier_t_model.title,
                                                       code=supplier_t_model.code, t_model=t_model)
            log.info("Создали модель транспорта '%s' по названию '%s', создали соответствие '%s'",
                     t_model, title, tmm)

            return t_model
        else:
            self.log_not_found(supplier_t_model)

    def log_not_found(self, supplier_t_model):
        log.error("Не нашли и не смогли создать модель транспорта title='%s' code='%s'",
                  supplier_t_model.title, supplier_t_model.code)

    def find_in_mappings(self, supplier_t_model):
        try:
            tmms = TransportModelMapping.objects.filter(title=supplier_t_model.title, code=supplier_t_model.code,
                                                        supplier=self.supplier)
            tmm = tmms[0]
            log.info("Нашли модель транспорта '%s' по имеющемуся соответствию '%s'",
                     tmm.t_model, tmm)
            return tmm.t_model

        except IndexError:
            raise TransportModel.DoesNotExist

    def find_by_title(self, supplier_t_model):
        title = supplier_t_model.title
        if supplier_t_model.real_title:
            title = supplier_t_model.real_title

        # для "Аааа" может найтись модель "АААА"
        t_models = TransportModel.objects.filter(title=title, t_type=supplier_t_model.t_type)

        for t_model in t_models:
            if t_model.title == title:

                tmm = TransportModelMapping.objects.create(supplier=self.supplier, title=supplier_t_model.title,
                                                           code=supplier_t_model.code, t_model=t_model)

                log.info("Нашли модель транспорта '%s' по названию '%s', создали соответствие '%s'",
                         t_model, title, tmm)

                return t_model

        raise TransportModel.DoesNotExist

    def extra_find(self, supplier_t_model):
        raise TransportModel.DoesNotExist
