# coding: utf-8

import re
import logging
from datetime import timedelta

from django.conf import settings

from common.utils.caching import cache_until_switch, cache_method_result

from travel.rasp.admin.importinfo.models.af import AFMaskText
from travel.rasp.admin.lib.exceptions import SimpleUnicodeException
from travel.rasp.admin.scripts.schedule.utils.mask_builders import MaskBuilder
from common.utils.date import DateTimeFormatter
from common.xgettext.common import get_datetemplate_translation


log = logging.getLogger(__name__)


DIGITS_RE = re.compile(ur'[1-7]', re.U + re.I)
LETTERS_RE = re.compile(ur'[A-Z]', re.U + re.I)
DIGIT_RANGE_RE = re.compile(ur'<d (?:\[ (?P<from>\d) - (?P<to>\d) \])? >', re.U + re.I + re.X)
DAYS_SEPARATOR = u', '


def shift_and_build_days_texts(thread, template_timezone='Europe/Moscow', builder=None):
    start = thread.template_start
    end = thread.template_end
    template = (thread.template_code or u'').strip()

    if not template:
        return None

    builder = builder or AFTextBuilder()

    if thread.time_zone == template_timezone:
        template_shift = 0
    elif template_timezone == 'local':
        template_shift = 0
    else:
        template_shift = -thread.calc_mask_shift(timezone=template_timezone)

    if template_shift:
        template = builder.shift_template(template, template_shift)
        start = start and start + timedelta(template_shift)
        end = end and end + timedelta(template_shift)

    thread.template_start = start
    thread.template_end = end
    thread.template_code = template

    try:
        texts = builder.build_range_day_texts(template, start, end)
    except (AFTextBuildError, AFTextMatchError) as e:
        log.error(u'Ошибка построения шаблона: %s', unicode(e))
        return None

    return texts


class AFTextBuildError(SimpleUnicodeException):
    pass


translations = {
    'AFTextBuilder_': u'',
    'AFTextBuilder_positive': u'{positive}',
    'AFTextBuilder_negative': u'кроме {negative}',
    'AFTextBuilder_positive_negative': u'{positive} кроме {negative}',

    'AFTextBuilder_positive_start': u'{positive} с {start:%d %B}',
    'AFTextBuilder_positive_end': u'{positive} по {end:%d %B}',
    'AFTextBuilder_positive_start_end': u'{positive} с {start:%d %B} по {end:%d %B}',

    'AFTextBuilder_negative_start': u'кроме {negative} с {start:%d %B}',
    'AFTextBuilder_negative_end': u'кроме {negative} по {end:%d %B}',
    'AFTextBuilder_negative_start_end': u'кроме {negative} с {start:%d %B} по {end:%d %B}',

    'AFTextBuilder_positive_negative_start': u'{positive} кроме {negative} с {start:%d %B}',
    'AFTextBuilder_positive_negative_end': u'{positive} кроме {negative} по {end:%d %B}',
    'AFTextBuilder_positive_negative_start_end': u'{positive} кроме {negative} с {start:%d %B} по {end:%d %B}',
}


@cache_until_switch
def get_af_texts():
    return list(AFMaskText.objects.all())


@cache_until_switch
def get_af_matcher(mask_codes):
    return AFTextMatcher(mask_codes)


class OldAfMaskBuilder(MaskBuilder):
    def __init__(self, lower_bound, upper_bound, today=None, country=None, **kwargs):
        super(OldAfMaskBuilder, self).__init__(lower_bound, upper_bound, today=today, country=country,
                                               **kwargs)

        from travel.rasp.admin.lib.mask_builder.afmask_builders import AfMaskBuilder
        self.af_mask_builder = AfMaskBuilder()

    def is_af_mask_text(self, text):
        return self.af_mask_builder.is_af_mask_text(text)

    def parse_af_text(self, text, start_date=None, end_date=None):
        return self.af_mask_builder.build(self.get_bounds(start_date, end_date), text, self.country, today=self.today)


class AFTextBuilder(object):
    def __init__(self):
        self.af_mask_texts = {
            mt.code: mt
            for mt in AFMaskText.objects.all()
        }

        self.mask_codes = list(self.af_mask_texts.keys())

        self.text_matcher = AFTextMatcher(self.mask_codes)

    def build_range_day_texts(self, template, start=None, end=None, range_=range(-1, 3)):
        texts = {}

        for shift in range_:
            day_texts = self.build_shifted_day_texts(template, start, end, shift)

            if day_texts:
                texts[shift] = day_texts

        return texts

    def build_day_texts(self, template, start, end):
        positive_matches, negative_matches = self.text_matcher.find_compound_match(template)

        positive_texts = self.build_texts_from_matches(positive_matches)
        negative_texts = self.build_texts_from_matches(negative_matches)

        params = {
            'positive': positive_texts,
            'negative': negative_texts,
            'start': start,
            'end': end
        }

        if not (positive_matches or negative_matches):
            raise AFTextBuildError(u'Пустой текст')

        key_parts = []
        for part in ['positive', 'negative', 'start', 'end']:
            if params[part]:
                key_parts.append(part)

        translation_key = 'AFTextBuilder_' + '_'.join(key_parts)

        texts = {}

        for lang in settings.FRONTEND_LANGUAGES:
            lang_params = dict()

            if params['positive']:
                lang_params['positive'] = params['positive'][lang]

            if params['negative']:
                lang_params['negative'] = params['negative'][lang]

            if start:
                lang_params['start'] = DateTimeFormatter(start, lang)

            if end:
                lang_params['end'] = DateTimeFormatter(end, lang)

            try:
                template = get_datetemplate_translation(lang, translation_key)
            except KeyError:
                template = translations[translation_key]

            texts[lang] = template.format(**lang_params)

        return texts

    def build_shifted_day_texts(self, template, start, end, shift):
        if shift:
            try:
                template = self.shift_template(template, shift)
            except AFTextBuildError as e:
                # Логируем только для этого случая, т.к. остальные не критичны
                # Т.к. электрички редко ходят больше суток
                if abs(shift) == 1:
                    log.error(u'Не смогли сдвинуть шаблон %s на %s: %s', template, shift,
                              unicode(e))
                return

            if start:
                start += timedelta(shift)

            if end:
                end += timedelta(shift)

        return self.build_day_texts(template, start, end)

    def build_texts_from_matches(self, matches):
        if not matches:
            return

        if len(matches) == 1:
            use_long = True
        else:
            use_long = False

        texts = {}

        for lang in settings.FRONTEND_LANGUAGES:
            parts = []
            for match in matches:
                parts.append(self.build_text_from_match(match, lang, use_long))

            texts[lang] = u', '.join(parts)

        return texts

    def build_text_from_match(self, match, lang, use_long):
        af_mask = self.get_aftext(match)

        if use_long:
            text = af_mask.L_long_text(lang=lang)
        else:
            text = af_mask.L_short_text(lang=lang)

        if not text:
            log.error(u'Не определено текста для %s %s %s',
                      af_mask.code,
                      'long' if use_long else 'short', lang)
            return u'<n/a>'

        if match.digits:
            digits = tuple(sorted(match.digits))
            digit_dict = self.get_digit_dict(digits, lang)

            text = text.format(**digit_dict)

        return text

    @cache_method_result
    def get_digit_dict(self, digits, lang):
        digit_dict = {}

        parts = []
        for d in digits:
            parts.append(self.get_digit_text(d, lang))

        digit_dict['days'] = DAYS_SEPARATOR.join(parts)

        inverted_digits = sorted({'1', '2', '3', '4', '5', '6', '7'} - set(digits))

        parts = []
        for d in inverted_digits:
            parts.append(self.get_digit_text(d, lang))

        digit_dict['inverted_days'] = DAYS_SEPARATOR.join(parts)

        return digit_dict

    def get_digit_text(self, d, lang):
        return self.af_mask_texts[d].L_short_text(lang=lang)

    @cache_method_result
    def shift_template(self, template, shift):
        if shift == 0:
            return template

        if shift < 0:
            shift_direction = -1
        else:
            shift_direction = 1

        shifted_template = template
        while shift:
            positive_matches, negative_matches = \
                self.text_matcher.find_compound_match(shifted_template)

            positive_parts, negative_parts = \
                self.shift_matches_on_day(positive_matches, shift_direction)
            new_positive_parts, new_negative_parts = \
                self.shift_matches_on_day(negative_matches, shift_direction)

            positive_parts.update(new_positive_parts)
            negative_parts.update(new_negative_parts)

            shifted_template = u''.join(sorted(positive_parts))
            if negative_parts:
                shifted_template += u'-' + u''.join(sorted(negative_parts))

            shift -= shift_direction

        return shifted_template

    def get_aftext(self, match):
        try:
            return self.af_mask_texts[match.match_template.code]
        except KeyError:
            raise AFTextBuildError(u'Не нашли шаблона с кодом {}'.format(match.match_template.code))

    def shift_matches_on_day(self, positive_matches, shift, _can_use_submatches=True):
        if shift == -1:
            attr = 'prev_code'
        elif shift == 1:
            attr = 'next_code'

        positive_parts = set()
        negative_parts = set()

        for match in positive_matches:
            af_text = self.get_aftext(match)

            shift_code = getattr(af_text, attr)

            if shift_code == 'none':
                raise AFTextBuildError(u'Не известен сдвиг шаблона {} на {}'
                                       .format(af_text.code, shift))

            if shift_code == 'auto':
                if not _can_use_submatches:
                    raise AFTextBuildError(u'Повторное автораскрытие шаблона {}'
                                           .format(af_text.code))

                submatches = self.text_matcher.get_submatches(match.mask)

                pos_sub_part_set, neg_sub_part_set = self.shift_matches_on_day(
                    submatches, shift, _can_use_submatches=False
                )

                positive_parts |= pos_sub_part_set
                negative_parts |= neg_sub_part_set

                continue

            if u'-' in shift_code:
                pos_part, neg_part = shift_code.split(u'-')
            else:
                pos_part = shift_code
                neg_part = u''

            positive_parts.update(list(pos_part))
            negative_parts.update(list(neg_part))

        return positive_parts, negative_parts


class AFTextMatchError(SimpleUnicodeException):
    pass


class AFTextMatcher(object):
    match_templates = None

    def __init__(self, mask_codes):
        self.mask_codes = mask_codes

        self.build_match_templates()

    def find_compound_match(self, template):
        if u'-' in template:
            positive, negative = template.split(u'-')
            negative = u'-' + negative

        else:
            positive = template
            negative = None

        positive_matches = self.find_matches(positive)

        if negative:
            negative_matches = self.find_matches(negative)
        else:
            negative_matches = []

        return positive_matches, negative_matches

    def find_matches(self, text):
        mask = AfMask(text)

        for template in self.match_templates:
            match = template.match(mask)
            if match:
                return [match]

        return self.get_submatches(mask)

    def get_submatches(self, mask):
        matches = []
        for mask in mask.get_submasks():
            for template in self.match_templates:
                if template.single:
                    match = template.match(mask)

                    if match:
                        matches.append(match)
                        break
            else:
                raise AFTextMatchError(u'Не нашли протейшего совпадения для {}'
                                       .format(mask.text))

        return sorted(matches, key=lambda m: m.match_template.code.lstrip(u'-'))

    def build_match_templates(self):
        match_templates = []

        for text in self.mask_codes:
            match_templates.append(MatchTemplate(text))

        # Шаблоны с диапазоном, должны рассматриваться позже
        self.match_templates = sorted(match_templates, key=lambda mt: mt.digit_range)


class AfMask(object):
    negative = False
    letters = set()
    digits = set()

    def __init__(self, text):
        self.text = text
        self._parse()

    def __unicode__(self):
        return u'<AfMask: {}>'.format(self.text)

    def _parse(self):
        text = self.text.replace(u'0', u'7')

        if text.startswith(u'-'):
            self.negative = True
            text = text.lstrip(u'-')

        digits = DIGITS_RE.findall(text)
        self.digits = set(digits)

        text = DIGITS_RE.sub(u'', text)

        letters = LETTERS_RE.findall(text)
        self.letters = set(letters)

        text = LETTERS_RE.sub(u'', text)

        if text:
            raise AFTextMatchError(u"Не смогли разобрать текст маски '{}'"
                                   .format(self.text))

    def get_submasks(self):
        submasks = []

        head = u''
        if self.negative:
            head = u'-'

        for part in self.text.replace(u'0', u'7').lstrip(u'-'):
            submasks.append(AfMask(head + part))

        return submasks


class MatchTemplate(object):
    single = False

    negative = False
    digits = set()
    letters = set()
    digit_range = None

    def __init__(self, code):
        self.code = code

        self._parse()

    def __unicode__(self):
        params = dict(**vars(self.__class__))
        params.update(vars(self))

        return u'<MatchTemplate: {}, single={single}, negative={negative}, digits={digits}' \
               u', letters={letters}, digit_range={digit_range}>'.format(self.code, **params)

    def _parse(self):
        text = self.code
        if text.startswith(u'-'):
            self.negative = True
            text = text.lstrip(u'-')

        if len(text) == 1:
            self.single = True

        if len(DIGIT_RANGE_RE.findall(text)) > 1:
            raise AFTextMatchError(u'Не смогли разобрать текст {}, несколько цифровых диапазонов'
                                   .format(self.text))

        match = DIGIT_RANGE_RE.search(text)
        if match:
            range_dict = match.groupdict()
            from_ = 1
            to = 7

            if range_dict.get('from', None):
                from_ = int(range_dict['from'])

            if range_dict.get('to', None):
                to = int(range_dict['to'])

            self.digit_range = (from_, to)

            text = DIGIT_RANGE_RE.sub(u'', text)

        digits = DIGITS_RE.findall(text)
        self.digits = set(digits)

        text = DIGITS_RE.sub(u'', text)

        letters = LETTERS_RE.findall(text)
        self.letters = set(letters)

        text = LETTERS_RE.sub(u'', text)

        if text:
            raise AFTextMatchError(u'Не смогли разобрать шаблон {}, {}'
                                   .format(self.code, text))

    def match(self, mask):
        match = Match(mask, self)

        if mask.negative != self.negative:
            return

        if not self.letters_match(mask):
            return

        if not self.digits_match(mask):
            return

        return match

    def letters_match(self, mask):
        if self.letters and not mask.letters:
            return False
        elif not self.letters and mask.letters:
            return False

        has_all_letters_from_template = mask.letters & self.letters == self.letters
        has_more_letters_then_template = bool(mask.letters - self.letters)

        return has_all_letters_from_template and not has_more_letters_then_template

    def digits_match(self, mask):
        if self.digits and not mask.digits:
            return False

        has_all_digits_from_template = mask.digits & self.digits == self.digits
        if not has_all_digits_from_template:
            return False

        digit_rest = mask.digits - self.digits

        if self.digit_range:
            return self.digit_range[0] <= len(digit_rest) <= self.digit_range[1]

        if digit_rest:
            return False
        else:
            return True

    def get_digit_rest(self, mask):
        return mask.digits - self.digits


class Match(object):
    def __init__(self, mask, template):
        self.mask = mask
        self.match_template = template

        self.digits = self.match_template.get_digit_rest(self.mask)

    def __unicode__(self):
        return u'<Match: {} to {}>'.format(
            self.mask, self.match_template
        )
