# -*- coding: utf-8 -*-

import os
import re
import os.path
import shutil
import logging
from urllib2 import urlopen

from django.conf import settings

from travel.rasp.admin.lib.tmpfiles import get_tmp_dir
from travel.rasp.admin.lib.unrar import unpack_rar_file
from travel.rasp.admin.lib.unzip import unpack_zip_file
from travel.rasp.admin.scripts.schedule.utils import RaspImportError
from travel.rasp.admin.scripts.utils.import_file_storage import get_schedule_temporary_date_filepath


log = logging.getLogger(__name__)


class FileProvider(object):
    CHUNK_SIZE = 1000000

    is_archive_file_re = re.compile(r'.*\.(?P<type>zip|rar)$', re.U + re.I)

    def __init__(self, supplier):
        self.supplier = supplier

        self._created_dirs = []
        self._create_files = []

    def unzip_files(self, zip_path):
        zip_tmp_dir = self.get_tmp_dir('unzip')

        file_map = unpack_zip_file(zip_path, temporary_directory=zip_tmp_dir)

        return file_map

    def unrar_files(self, rar_path):
        rar_tmp_dir = self.get_tmp_dir('unrar')

        file_map = unpack_rar_file(rar_path, temporary_directory=rar_tmp_dir)

        return file_map

    def store_fileobj(self, fileobj, filename, subpath=None):
        filepath = self.get_tmp_file_path(filename, subpath)

        fileobj.seek(0)

        with open(filepath, 'wb') as f:
            f.write(fileobj.read())

        return filepath

    def get_tmp_dir(self, subpath=None):
        base = "%s/%s" % (self.supplier.code, self.__class__.__name__)

        dir_path = os.path.join(base, subpath) if subpath else base

        tmp_dir = get_tmp_dir(dir_path)

        self._created_dirs.append(tmp_dir)

        return tmp_dir

    def get_tmp_file_path(self, filename, subpath):
        tmp_dir = self.get_tmp_dir(subpath)
        filepath = os.path.join(tmp_dir, filename)

        self._create_files = filepath

        return filepath

    def clean(self):
        for filepath in self._create_files:
            try:
                os.remove(filepath)
            except OSError:
                pass

        for dirpath in self._created_dirs:
            shutil.rmtree(dirpath, ignore_errors=True)

    def download_file(self, download_params, filepath):
        if os.path.exists(filepath):
            log.info(u"Файл %s уже скачан берем его", filepath)
            return filepath
        else:
            log.info(u"Скачиваем файл в %s", filepath)

        f = open(filepath, 'w')

        try:
            for data in self.retrieve_data(download_params):
                f.write(data)

            f.close()
        except Exception:
            os.remove(filepath)
            raise

        return filepath

    def retrieve_data(self, download_params):
        request = self.get_request(download_params)
        urlopen = self.get_opener()
        responce = urlopen(request, timeout=settings.SCHEDULE_IMPORT_TIMEOUT)

        data = responce.read(self.CHUNK_SIZE)
        while data:
            yield data
            data = responce.read(self.CHUNK_SIZE)

    def get_opener(self):
        return urlopen

    def get_request(self, download_params):
        return download_params

    @classmethod
    def is_archive_file(cls, filename):
        return cls.is_archive_file_re.match(filename)


class PackageFileProvider(FileProvider):
    def __init__(self, package, supplier=None):
        supplier = supplier or package.supplier
        super(PackageFileProvider, self).__init__(supplier)

        self.package = package

        self.fileobj = package.package_file

    def is_archive_package_file(self):
        return self.is_archive_file(self.fileobj.name)

    def unpack_package_archive(self):
        match = self.is_archive_file_re.match(self.fileobj.name)
        ext = match.groupdict()['type']
        filename = self.supplier.code + '.' + ext

        filepath = self.store_fileobj(self.fileobj, filename)

        return getattr(self, 'un%s_files' % ext)(filepath)

    def get_unpack_map_with_trimmed_dirs_from_package_archive(self):
        file_map = self.unpack_package_archive()

        new_file_map = dict()
        for filename, filepath in file_map.items():
            new_filename = os.path.basename(filename)
            new_file_map[new_filename] = filepath

        return new_file_map

    def get_files_from_archive_with_ext(self, ext):
        if not ext.startswith('.'):
            ext = '.' + ext

        file_map = self.unpack_package_archive()
        files = []

        for filename, filepath in file_map.items():
            if filename.endswith(ext):
                files.append(filepath)

        return files

    def store_package_file(self):
        return self.store_fileobj(self.fileobj, self.fileobj.name)

    def get_package_filepath(self, filename):
        return get_schedule_temporary_date_filepath(filename, self.package)

    def get_tmp_dir(self, subpath=None):
        base = "%s/%s/%s" % (self.supplier.code, self.package.id, self.__class__.__name__)

        dir_path = os.path.join(base, subpath) if subpath else base

        tmp_dir = get_tmp_dir(dir_path)

        self._created_dirs.append(tmp_dir)

        return tmp_dir


class XmlPackageFileProvider(PackageFileProvider):
    def get_schedule_files(self):
        if self.is_archive_package_file():
            files = self.get_files_from_archive_with_ext('.xml')

            if not files:
                raise RaspImportError(u"Не нашли файлов xml в архиве")

            log.info(u'Обрабатываем файлы %s', files)

            return files

        else:
            log.info(u'Обрабатываем файл %s', self.fileobj.name)

            return [self.store_package_file()]
