# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import os
import logging

from django.conf import settings
from django.utils.encoding import force_text
from retry import retry

from common.data_api.proxy.partners_proxy import PartnersProxy
from travel.rasp.library.python.common23.date import environment
from common.utils.fileutils import get_project_relative_path, is_project_path
from subprocess import Popen, PIPE
from travel.rasp.admin.lib import tmpfiles
from travel.rasp.admin.lib.exceptions import FormattingException


log = logging.getLogger(__name__)


class FtpDownloadError(FormattingException):
    pass


def get_script_runs_log_dir():
    import __main__ as main_module

    filepath = getattr(main_module, '__file__', None)

    if not filepath:
        return

    if not is_project_path(filepath):
        return

    return os.path.join(settings.LOG_PATH, 'special/script_runs/',
                        get_project_relative_path(os.path.splitext(filepath)[0]))


def get_wget_log_file():
    script_run_log_dir = get_script_runs_log_dir()
    if script_run_log_dir:
        if not os.path.exists(script_run_log_dir):
            os.makedirs(script_run_log_dir)

        return os.path.join(script_run_log_dir, environment.now().strftime('wget_%Y-%m-%d_%H%M%S.log'))
    else:
        return os.path.join(settings.LOG_PATH, 'wget.log')


@retry(tries=3, delay=60)
def get_ftp_file(url, ftp_user, ftp_password, out_file=None):
    partner_proxy_pool = PartnersProxy.get_proxy_pool()
    if not partner_proxy_pool:
        return ftp_wget(url, ftp_user, ftp_password, out_file)

    proxies = partner_proxy_pool.get_http_proxies()

    if len(proxies) == 0:
        raise FtpDownloadError('Не удалось скачать файл - не нашли доступный прокси сервер')

    wget, stdout, stderr = (None, None, None)
    for proxy in proxies:
        wget, stdout, stderr = ftp_wget(url, ftp_user, ftp_password, tries=2, out_file=out_file, proxy_url=proxy.url)

        if wget.returncode == 0:
            break

        log.warn('Не удалось скачать файл %s - wget return code = %d, stderr: %s',
                 url, wget.returncode, force_text(stderr))

    return wget, stdout, stderr


@tmpfiles.clean_temp
def ftp_wget(url, user, password, out_file=None, tries=3, proxy_url=None):
    args = ['wget', url, '-O', out_file or '-', '--dot-style=mega', '-a', get_wget_log_file(), '--connect-timeout=3',
            '--tries={}'.format(tries), '--user={}'.format(user), '--password={}'.format(password)]

    if proxy_url:
        args.extend(['-e', 'use_proxy=yes', '-e', 'ftp_proxy={}'.format(proxy_url)])

    wget = Popen(args, stdout=PIPE, stderr=PIPE, cwd=tmpfiles.get_tmp_dir('wget-working-dir'))

    stdout = wget.stdout.read()
    stderr = wget.stderr.read()
    wget.wait()

    return wget, stdout, stderr
