# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import requests
from retry import retry

from common.data_api.proxy.partners_proxy import PartnersProxy
from travel.rasp.admin.lib.exceptions import SimpleUnicodeException
from travel.rasp.admin.lib.logs import get_current_file_logger


log = get_current_file_logger()


class HttpProxyGetFileError(SimpleUnicodeException):
    pass


def get_response(method, url, proxies, **kwargs):
    try:
        response = requests.request(
            method=method,
            url=url,
            proxies=proxies,
            **kwargs
        )

    except Exception:
        msg = "Ошибка при получении данных"
        log.exception(msg)
        raise HttpProxyGetFileError(msg)

    try:
        response.raise_for_status()
    except requests.HTTPError:
        msg = "Неуспешный ответ. status: {} reason: {}", response.status_code, response.reason
        log.exception(msg)
        raise HttpProxyGetFileError(msg)

    return response.content


@retry(tries=3, delay=60)
def download_data(method, url, data=None, headers=None, timeout=None, params=None):
    partner_proxy_pool = PartnersProxy.get_proxy_pool()
    proxies = partner_proxy_pool.get_http_proxies()

    if len(proxies) == 0:
        raise HttpProxyGetFileError("Нет доступных прокси")

    for proxies_dict in proxies:
        try:
            data = get_response(method, url, proxies_dict, data=data, headers=headers, timeout=timeout, params=params)
            return data
        except Exception:
            log.error("Не смогли получить данные через прокси: {}".format(proxies_dict))

    raise HttpProxyGetFileError("Не смогли получить данные")
