# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import os.path
import time

from django.conf import settings

from travel.rasp.admin.lib.exceptions import SimpleUnicodeException
from travel.rasp.admin.lib.logs import get_current_file_logger
from travel.rasp.admin.scripts.schedule.utils.http_proxy import download_data
from travel.rasp.admin.scripts.utils.import_file_storage import get_schedule_temporary_date_filepath


log = get_current_file_logger()


class OlvenGetFileError(SimpleUnicodeException):
    pass


def get_olven_file(package, method, departure_day=None, **kwargs):
    """
    Достает данные по запросу method, если указан departure_day
    то кладет это в список параметров запроса( работает только для day_timetable ).
    Возвращает имя файла с данными в случае успеха, либо None в случае неудачи
    """

    start = time.time()

    post = make_olven_post_data(method, departure_day=departure_day, **kwargs)

    filename = "olven_{method}{departure_day}.xml".format(
        method=method,
        departure_day=departure_day and departure_day.strftime("_%d.%m.%Y") or ""
    )

    filepath = get_schedule_temporary_date_filepath(filename, package)

    log.info("Получаем данные от Олвена %s %s: %s", method, departure_day, filepath)

    if os.path.exists(filepath):
        log.info("Файл %s уже скачан", filepath)
        return filepath

    headers = {
        "Content-Type": "application/x-www-form-urlencoded"
    }

    log.info("Посылаем запрос POST %s", settings.OLVEN_HOST)
    data = download_data(
        method="POST",
        url="http://{}".format(settings.OLVEN_HOST),
        data=post,
        headers=headers,
        timeout=settings.SCHEDULE_IMPORT_TIMEOUT
    )

    with open(filepath, "wb") as f:
        f.write(data)

    log.info("Файл %s успешно скачан за %s секунд", filepath, time.time() - start)

    return filepath


def make_olven_post_data(method, departure_day=None, **kwargs):
    if departure_day:
        data = "<departure_day>{}</departure_day>".format(departure_day.strftime("%d.%m.%Y"))
    else:
        data = ""

    for k, v in kwargs.items():
        data += "<{}>{}</{}>".format(k, v, k)

    post = """query=<?xml version="1.0" encoding="windows-1251"?>
    <request>
        <method>{method}</method>
        <auth>
            <login>{login}</login>
            <password>{password}</password>
        </auth>
        <data>
            {data}
        </data>
    </request>""".format(**{"method": method, "data": data,
                            "login": settings.OLVEN_USER,
                            "password": settings.OLVEN_PASSWORD})

    return post
