# coding: utf8

import os
import os.path
from collections import defaultdict
from datetime import timedelta
from shutil import rmtree

from django.conf import settings
from django.db import connection, transaction, DatabaseError
from django.utils.encoding import is_protected_type

from common.models.schedule import Route, RThread, RTStation
from travel.rasp.library.python.common23.date import environment
from common.utils.date import RunMask
from common.utils.iterrecipes import chunker
from travel.rasp.admin.lib import tmpfiles
from travel.rasp.admin.lib.logs import get_current_file_logger
from travel.rasp.admin.lib.mysqlutils import MysqlFileWriter, MysqlModelUpdater, MysqlFileReader
from travel.rasp.admin.lib.tmpfiles import temp_manager
from travel.rasp.admin.scripts.schedule.utils.errors import RaspImportError
from travel.rasp.admin.scripts.schedule.utils.schedule_validation import ScheduleValidator, simple_thread_validator
from travel.rasp.admin.www.utils.mysql import (fast_delete_threads, fast_delete_routes, fast_delete_package_threads_and_routes,
                             fast_delete_supplier_threads_and_routes)

log = get_current_file_logger()


class RouteSaver(object):
    def save_route(self, route):
        route.save()

    def save_thread(self, thread):
        if not thread.route_id:
            thread.route = Route.objects.get(route_uid=thread.route.route_uid)
        thread.save()

    def save_rtstation(self, rts):
        rts.save()

    def load(self):
        pass


class RouteLoader(object):
    script_protected = False

    def __init__(self, write_to_log=None, saver=RouteSaver):
        self.log = write_to_log or log

        self.route_pre_save_hooks = []
        self.thread_pre_save_hooks = []

        if isinstance(saver, RouteSaver):
            self.saver = saver
        else:
            self.saver = saver()

    # routes
    def add_route(self, route):
        self.prepare_route(route)

        self.run_route_pre_save_hooks(route)

        self.saver.save_route(route)

        for thread in route.threads:
            thread.route = route

            self.add_thread(thread)

        self.route_added_report(route)

    def route_added_report(self, route):
        self.log.info(u"Маршрут %s добавлен успешно", route.route_uid)

    def run_route_pre_save_hooks(self, route):
        for hook in self.route_pre_save_hooks:
            hook(route)

    def add_presave_route_hook(self, hook):
        self.route_pre_save_hooks.append(hook)

    # threads
    def add_thread(self, thread):
        self.prepare_thread(thread)

        self.run_thread_pre_save_hooks(thread)

        self.saver.save_thread(thread)

        if thread.rtstations:
            for rts in thread.rtstations:
                rts.thread = thread
                self.saver.save_rtstation(rts)

        self.thread_added_report(thread)

    def thread_added_report(self, thread):
        self.log.info(u"Добавили нитку %s%s", thread.uid, u" #" + thread.template_text if thread.template_text else u"")

    def thread_update_report(self, thread):
        self.log.info(u"Обновили нитку %s%s", thread.uid, u" #" + thread.template_text if thread.template_text else u"")

    def run_thread_pre_save_hooks(self, thread):
        for hook in self.thread_pre_save_hooks:
            hook(thread)

    def add_presave_thread_hook(self, hook):
        self.thread_pre_save_hooks.append(hook)

    def prepare_route(self, route):
        route.script_protected = self.script_protected

        if not route.route_uid:
            route.route_uid = route.threads[0].gen_route_uid()

        if not route.comment:
            route.comment = ''

    def prepare_thread(self, thread):
        thread.changed = True

        if len(thread.rtstations) < 2:
            self.log.warning(u"У нитки %s %d станций", thread.uid, len(thread.rtstations))

        if not thread.uid:
            thread.gen_uid()


class BulkRouteLoader(RouteLoader):
    def __init__(self, log=None):
        super(BulkRouteLoader, self).__init__(log)

        self.saver = BulkRouteSaver(self.log)

    def load(self):
        self.saver.load()


class BulkRouteSaver(RouteSaver):
    def __init__(self, log=None):
        self.log = log

        self.tmp_dir = temp_manager.get_tmp_dir('bulk_route_saver')

        self.route_writer = self.get_writer('route', Route)
        self.thread_writer = self.get_writer('thread', RThread)
        self.rts_writer = self.get_writer('rts', RTStation)

    def get_writer(self, name, model):
        outfile = open(self.get_filepath(name), 'wb')
        return MysqlFileWriter(outfile, get_columns(model))

    def get_filepath(self, name):
        return os.path.join(self.tmp_dir, name + '.tsv')

    def clean(self):
        if not settings.DEBUG:
            rmtree(self.tmp_dir)

    def save_route(self, route):
        row = get_values(route)
        self.route_writer.writerow(row)

    def save_thread(self, thread):
        row = get_values(thread, exclude=['route_id', 'id'])
        row += [thread.route.route_uid]
        self.thread_writer.writerow(row)

    def save_rtstation(self, rts):
        row = get_values(rts, exclude=['thread_id', 'id'])
        row += [rts.thread.uid]
        self.rts_writer.writerow(row)

    @transaction.atomic
    def load(self):
        self.log.info(u"Заливаем расписание")
        self.route_writer.close()
        self.thread_writer.close()
        self.rts_writer.close()

        self.cursor = connection.cursor()
        self.cursor.execute('set foreign_key_checks = 0')
        self.load_routes()
        self.load_threads()
        self.load_rtstations()
        self.cursor.execute('set foreign_key_checks = 1')

        self.clean()

    def load_routes(self):
        self.log.info(u"Заливаем маршруты")
        columns = get_columns(Route)
        columns = u','.join(columns)

        sql = ur"""
LOAD DATA LOCAL INFILE '%(filepath)s'
INTO TABLE www_route
CHARACTER SET utf8
FIELDS ENCLOSED BY '"'
(%(columns)s)
""" % {
            'columns': columns,
            'filepath': self.get_filepath('route')
        }

        self.execute_and_check(sql)

        self.log.info(u"Файл успешно залит")

    def load_threads(self):
        self.log.info(u"Заливаем нитки")
        columns = get_columns(RThread, exclude=['route_id', 'id'])
        columns = u','.join(columns)
        self.log.info(u"Заменяем route_uid на id")
        self.replace_route_uid_with_route_id()
        self.log.info(u"Заливаем файл")

        sql = ur"""
LOAD DATA LOCAL INFILE '%(filepath)s'
INTO TABLE www_rthread
CHARACTER SET utf8
FIELDS ENCLOSED BY '"'
(%(columns)s,route_id)
""" % {
            'columns': columns,
            'filepath': self.get_filepath('thread2')
        }

        self.execute_and_check(sql)

        self.log.info(u"Файл успешно залит")

    def load_rtstations(self):
        self.log.info(u"Заливаем станции ниток")
        columns = get_columns(RTStation, exclude=['thread_id', 'id'])
        columns = u','.join(columns)
        self.log.info(u"Заменяем uid на id")
        self.replace_thread_uid_with_thread_id()
        self.log.info(u"Заливаем файл")

        sql = ur"""
LOAD DATA LOCAL INFILE '%(filepath)s'
INTO TABLE www_rtstation
CHARACTER SET utf8
FIELDS ENCLOSED BY '"'
(%(columns)s,thread_id)
""" % {
            'columns': columns,
            'filepath': self.get_filepath('rts2')
        }

        self.execute_and_check(sql)

        self.log.info(u"Файл успешно залит")

    def replace_route_uid_with_route_id(self):
        route_uids = set()

        with open(self.get_filepath('thread')) as f:
            for line in f:
                route_uid = line.split('\t')[-1].strip().strip('"')
                route_uids.add(route_uid)

        uid_map = {}
        for route_uid_chunk in chunker(route_uids, RThread.IMPORT_UID_CHUNK_SIZE):
            route_values = Route.objects.filter(route_uid__in=route_uid_chunk).values('id', 'route_uid')
            for v in route_values:
                uid_map[v['route_uid']] = v['id']

        with open(self.get_filepath('thread2'), 'wb') as new_file:
            for line in open(self.get_filepath('thread')):
                parts = line.split('\t')
                route_uid = parts[-1].strip().strip('"')
                route_id = uid_map[route_uid]

                parts[-1] = parts[-1].replace(route_uid, str(route_id))
                new_file.write('\t'.join(parts))

    def replace_thread_uid_with_thread_id(self):
        thread_uids = set()
        with open(self.get_filepath('rts')) as f:
            for line in f:
                thread_uid = line.split('\t')[-1].strip().strip('"')
                thread_uids.add(thread_uid)

        uid_map = {}
        for thread_uid_chunk in chunker(thread_uids, RThread.IMPORT_UID_CHUNK_SIZE):
            thread_values = RThread.objects.filter(uid__in=thread_uid_chunk).values('id', 'uid')
            for v in thread_values:
                uid_map[v['uid']] = v['id']

        with open(self.get_filepath('rts2'), 'wb') as new_file:
            for line in open(self.get_filepath('rts')):
                parts = line.split('\t')
                thread_uid = parts[-1].strip().strip('"')
                id = uid_map[thread_uid]

                parts[-1] = parts[-1].replace(thread_uid, str(id))
                new_file.write('\t'.join(parts))

    def check_message(self, mysql_message):
        """Records: 0  Deleted: 0  Skipped: 0  Warnings: 0"""

        if not mysql_message:
            self.log.warning(u"Пустое сообщение от mysql")
            return

        status = {}
        parts = mysql_message.strip().split()
        for name, value in zip(parts[::2], parts[1::2]):
            name = name.lower().replace(u":", u"")
            value = int(value)

            status[name] = value

        if status.get('skipped', 0) > 0:
            raise DatabaseError(u"Some rows were skipped")

        if status.get('warnings', 0) > 0:
            self.cursor.execute("show warnings")
            raise DatabaseError(u"Some warnings were gotten:\n{}".format(
                u"\t\n".join(u"\t".join(map(unicode, r)) for r in self.cursor.fetchall())
            ))

    def execute_and_check(self, sql):
        self.log.debug(sql)
        self.cursor.execute(sql)
        self.check_message(connection.connection.info())


class CompactThreadNumberBuilder(object):
    """Генерируем первый свободный номер нитки"""

    def __init__(self, route_query):
        self.thread_ordinal_numbers = defaultdict(set)

        for route in route_query:
            self.thread_ordinal_numbers[route.route_uid] = set(route.get_all_stored_thread_ordinal_numbers())

    def build_for(self, thread):
        route_uid = thread.route.route_uid
        existed_ordinal_numbers = self.thread_ordinal_numbers[route_uid]

        ordinal_number = 0
        while ordinal_number in existed_ordinal_numbers:
            ordinal_number += 1

        thread.ordinal_number = ordinal_number

        self.thread_ordinal_numbers[route_uid].add(thread.ordinal_number)

        thread.gen_uid()


class MaskUpdater(object):
    def __init__(self):
        self.masks = {}

        self.today = environment.today()

    def prepare(self, thread_qs):
        thread_qs.update(year_days=RunMask.EMPTY_YEAR_DAYS)

    def get_mask(self, thread):
        if hasattr(thread, 'mask'):
            return thread.mask

        else:
            return thread.get_mask(today=self.today)

    def add_thread(self, thread):
        mask = self.get_mask(thread)

        if not thread.import_uid:
            log.error(u'У нитки не заполнен import_uid не обновляем ему маску')
            return

        if thread.import_uid in self.masks:
            self.masks[thread.import_uid] |= mask
        else:
            self.masks[thread.import_uid] = mask

    def update(self):
        for import_uid, mask in self.masks.iteritems():
            if not import_uid:
                log.error(u'Пустой import_uid пропускаем нитку')
                continue

            RThread.objects.filter(import_uid=import_uid).update(year_days=str(mask))


class ThreadMysqlModelUpdaterByImportUid(MysqlModelUpdater):
    def __init__(self, working_dir, fields=None, load_on_context_exit=True):
        super(ThreadMysqlModelUpdaterByImportUid, self).__init__(
            RThread, working_dir, fields=fields, load_on_context_exit=load_on_context_exit,
            tmp_attrs=['import_uid']
        )

    def process(self):
        import_uids = []

        with open(self.filepath) as f:
            reader = MysqlFileReader(f, self.all_column_names)
            for rowdict in reader:
                import_uids.append(rowdict['import_uid'])

        uuid_to_id = {}

        for import_uuid_chunk in chunker(import_uids, RThread.IMPORT_UID_CHUNK_SIZE):
            uuid_to_id.update(RThread.objects.filter(import_uid__in=import_uuid_chunk)
                                     .order_by()
                                     .values_list('import_uid', 'id'))

        processed_filepath = self.get_filepath('_processed')
        with open(processed_filepath, 'wb') as processed_file, open(self.filepath) as f:
            writer = MysqlFileWriter(processed_file, self.column_names)
            reader = MysqlFileReader(f, self.all_column_names)

            for rowdict in reader:
                rowdict['id'] = uuid_to_id[rowdict['import_uid']]
                writer.writedict(rowdict)

        self.filepath = processed_filepath


class BulkMaskUpdater(MaskUpdater):
    def __init__(self, *args, **kwargs):
        super(BulkMaskUpdater, self).__init__(*args, **kwargs)

    @tmpfiles.clean_temp
    def update(self):
        log.info(u"Обновляем маски")

        if not all(self.masks.keys()):
            raise RaspImportError(u'Нашли пустой import_uid не обновляем маски')

        with ThreadMysqlModelUpdaterByImportUid(
            tmpfiles.get_tmp_dir(), fields=('year_days',)
        ) as updater:
            for import_uid, mask in self.masks.iteritems():
                if not import_uid:
                    log.error(u'Пустой import_uid пропускаем нитку')
                    continue

                updater.add_dict({
                    'id': None,
                    'import_uid': import_uid,
                    'tz_year_days': str(mask)
                })

        log.info(u"Обновили маски")


class SavePastDaysMaskUpdater(MaskUpdater):
    def __init__(self, days_to_past=14):
        super(SavePastDaysMaskUpdater, self).__init__()

        self.days_to_past = days_to_past

        self.past_mask = RunMask.range(self.today - timedelta(self.days_to_past), self.today, today=self.today)

    def prepare(self, thread_qs):
        for thread in thread_qs.only('import_uid', 'year_days'):
            mask = self.get_mask(thread)

            thread.mask = mask & self.past_mask

            self.add_thread(thread)


class BulkSavePastDaysMaskUpdater(SavePastDaysMaskUpdater, BulkMaskUpdater):
    pass


class RouteUpdaterError(RaspImportError):
    pass


class RouteUpdater(RouteLoader):
    def __init__(self, route_query, log=None, saver=RouteSaver, mask_updater=MaskUpdater,
                 remove_old_routes=False, thread_number_builder=CompactThreadNumberBuilder,
                 update_fields=None,
                 _from_fabric=False):

        if not _from_fabric:
            raise RouteUpdaterError(u"Нельзя создавать напрямую RouteUpdater воспользуйтесь класс "
                                    u"методом create_route_updater")

        super(RouteUpdater, self).__init__(log, saver)

        self.update_fields = update_fields
        self.remove_old_routes = remove_old_routes
        self.uuid_to_uid = {}
        self.all_thread_uuids = set()
        self.deleted_thread_uuids = set()

        thread_query = RThread.objects.filter(route__in=route_query)

        self.route_query = route_query
        self.thread_query = thread_query
        self.done = False

        if isinstance(mask_updater, MaskUpdater):
            self.mask_updater = mask_updater
        else:
            self.mask_updater = mask_updater()

        if isinstance(thread_number_builder, CompactThreadNumberBuilder):
            self.thread_number_builder = thread_number_builder
        else:
            self.thread_number_builder = thread_number_builder(self.route_query)

    @classmethod
    def create_route_updater(cls, route_query, log=None, saver=RouteSaver,
                             mask_updater=MaskUpdater,
                             thread_number_builder=CompactThreadNumberBuilder,
                             update_fields=None,
                             remove_old_routes=False):

        route_updater = cls(route_query, log=log, saver=saver,
                            mask_updater=mask_updater, thread_number_builder=thread_number_builder,
                            remove_old_routes=remove_old_routes, update_fields=update_fields,
                            _from_fabric=True)

        route_updater.prepare()

        return route_updater

    def prepare(self):
        if self.remove_old_routes:
            self.do_remove_old_routes()

        self.prepare_route_values()
        self.prepare_thread_values()

        self.mask_updater.prepare(self.affected_threads)

    def do_remove_old_routes(self):
        thread_qs = self.thread_query.exclude(route__script_protected=True)

        uid_to_uuid_list = list(thread_qs.values_list('uid', 'import_uid'))
        self.uuid_to_uid.update({uuid: uid for uid, uuid in uid_to_uuid_list})

        self.deleted_thread_uuids = set(
            thread_qs.filter(route__script_protected=True).values_list('import_uid', flat=True)
        )

        fast_delete_threads(thread_qs, log=self.log)

        self.delete_empty_routes()

    def prepare_route_values(self):
        db_route_uids = list(self.route_query.values_list('route_uid', flat=True))

        self.processed_route_uids = set()
        self.excluded_route_uids = set(
            self.route_query.filter(script_protected=True).values_list('route_uid', flat=True)
        )
        self.existed_route_uids = set(db_route_uids) - self.excluded_route_uids

    def prepare_thread_values(self):
        existed_threads_pairs = list(self.thread_query.values_list('uid', 'import_uid'))
        db_thread_import_uuids = [t[1] for t in existed_threads_pairs]

        self.uuid_to_uid.update({uuid: uid for uid, uuid in existed_threads_pairs})

        self.excluded_thread_uuids = set(
            self.thread_query.filter(route__script_protected=True)
                .values_list('import_uid', flat=True)
        )
        self.processed_thread_uuids = set()
        self.db_threads_uuids = set(db_thread_import_uuids) - self.excluded_thread_uuids
        self.existed_thread_uuids = set(db_thread_import_uuids) - self.excluded_thread_uuids

        self.affected_threads = self.thread_query.exclude(import_uid__in=self.excluded_thread_uuids)

    def add_route(self, route):
        self.prepare_route(route)

        if self.is_ignored_route(route):
            self.log.info(u"Пропускаем маршрут %s", route.route_uid)
            return

        if self.is_first_route_encounter(route):
            self.processed_route_uids.add(route.route_uid)
            self.run_route_pre_save_hooks(route)

        if self.need_to_save_route(route):
            self.saver.save_route(route)

            self.new_route_saved(route)

        for thread in route.threads:
            thread.route = route

            self.add_thread(thread)

    def add_thread(self, thread):
        self.prepare_thread(thread)

        if self.is_ignored_thread(thread):
            self.log.info(u"Пропускаем нитку %s %s", thread.route.route_uid, thread.import_uid)
            return

        if thread.route.route_uid not in self.existed_route_uids:
            raise RouteUpdaterError(u"Добавляем нитку для несуществующего маршрута %s" %
                                    thread.route.route_uid)

        if self.is_first_thread_encounter(thread):
            self.processed_thread_uuids.add(thread.import_uid)
            self.run_thread_pre_save_hooks(thread)

        if self.need_to_save_thread(thread):
            self.thread_number_builder.build_for(thread)

            self.saver.save_thread(thread)

            if thread.rtstations:
                for rts in thread.rtstations:
                    rts.thread = thread
                    self.saver.save_rtstation(rts)

            self.existed_thread_uuids.add(thread.import_uid)
            self.uuid_to_uid[thread.import_uid] = thread.uid
            self.thread_added_report(thread)

        else:
            # Нитка уже лежит в базе, берем ее uid
            thread.uid = self.uuid_to_uid[thread.import_uid]

            if self.update_fields:
                # FIXME: не работает для BulkRouteLoader
                self.update_thread_fields(thread)
                self.thread_update_report(thread)

        self.mask_updater.add_thread(thread)

    def update_thread_fields(self, thread):
        """
        Обновление полей нитки при импорте.
        Т.к. решение о добавлении нитки происходит на основании uuid, значения полей не участвующих в формировании uuid
        не импортируются если uuid уже есть в БД.
        Чтобы эти значения появились в БД надо их скопировать в существующую запись.

        В update_fields должны быть указаны поля не участвующие в формировании uuid.
        """
        db_thread = RThread.objects.get(uid=thread.uid)

        for field in self.update_fields:
            db_field_value = getattr(db_thread, field)
            new_field_value = getattr(thread, field)
            if db_field_value != new_field_value:
                setattr(db_thread, field, new_field_value)
                self.log.info(u"Обновление поля {field} с {old_value} на {new_value}".format(
                    field=field, old_value=db_field_value, new_value=new_field_value))

        db_thread.save()

    def update(self):
        self.log.info(u"Заливаем данные")
        self.saver.load()

        self.mask_updater.update()

        self.set_changed_flags()

        self.log.info(u"Удаляем устаревшие данные")
        self.delete_obsolete_threads()

        self.delete_empty_routes()

        self.done = True

        self.log.info(u"Данные залиты")

    def get_affected_thread_uids(self):
        if not self.done:
            raise RouteUpdaterError(u'Импорт еще не закнчился, нельзя вызывать {}'.format(
                'get_affected_thread_uids'
            ))

        affected_uuids = self.deleted_thread_uuids | self.processed_thread_uuids | \
            self.existed_thread_uuids

        return {self.uuid_to_uid[uuid] for uuid in affected_uuids}

    def prepare_thread(self, thread):
        super(RouteUpdater, self).prepare_thread(thread)

        if not thread.import_uid:
            thread.gen_import_uid()

    def is_first_route_encounter(self, route):
        return route.route_uid not in self.processed_route_uids

    def need_to_save_route(self, route):
        return route.route_uid not in self.existed_route_uids

    def is_first_thread_encounter(self, thread):
        return thread.import_uid not in self.processed_thread_uuids

    def need_to_save_thread(self, thread):
        return thread.import_uid not in self.existed_thread_uuids

    def is_ignored_route(self, route):
        return route.route_uid in self.excluded_route_uids

    def is_ignored_thread(self, thread):
        return thread.route.route_uid in self.excluded_route_uids

    def delete_obsolete_threads(self):
        obsolete_threads = self.existed_thread_uuids - self.processed_thread_uuids - self.excluded_thread_uuids

        for uuids in chunker(obsolete_threads, RThread.IMPORT_UID_CHUNK_SIZE):
            fast_delete_threads(RThread.objects.filter(import_uid__in=uuids, year_days=RunMask.EMPTY_YEAR_DAYS))

        fast_delete_threads(self.thread_query.filter(route__script_protected=False,
                            import_uid__isnull=True))

    def delete_empty_routes(self):
        route_query_for_delete = self.route_query.filter(script_protected=False, rthread__isnull=True)
        fast_delete_routes(route_query_for_delete, log=self.log)

    def new_route_saved(self, route):
        self.existed_route_uids.add(route.route_uid)
        self.route_added_report(route)

    def set_changed_flags(self):
        existed_updated_thread_uuids = self.db_threads_uuids & self.processed_thread_uuids

        # Несчитаем табло только не измененным рейсам
        for uuids in chunker(existed_updated_thread_uuids, RThread.IMPORT_UID_CHUNK_SIZE):
            RThread.objects.filter(import_uid__in=uuids). \
                filter(changed=False).update(path_and_time_unchanged=True)

        for uuids in chunker(self.processed_thread_uuids, RThread.IMPORT_UID_CHUNK_SIZE):
            RThread.objects.filter(import_uid__in=uuids).update(changed=True)

    def get_message(self):
        return u"Загрузили %s ниток" % len(self.processed_thread_uuids)

    def add_report_to_reporter(self, reporter):
        message = self.get_message()
        if message:
            reporter.add_report(u"Отчет о импорте", message)


class PackageRouteUpdater(RouteUpdater):
    package = None

    @classmethod
    def create_route_updater(
        cls,
        package,
        log=None,
        saver=RouteSaver,
        mask_updater=MaskUpdater,
        thread_number_builder=CompactThreadNumberBuilder,
        update_fields=None,
        remove_old_routes=False
    ):
        route_query = Route.objects.filter(two_stage_package=package)
        route_updater = cls(
            route_query,
            log=log,
            saver=saver,
            mask_updater=mask_updater,
            thread_number_builder=thread_number_builder,
            remove_old_routes=remove_old_routes,
            update_fields=update_fields,
            _from_fabric=True
        )
        route_updater.package = package
        route_updater.prepare()

        return route_updater

    def do_remove_old_routes(self):
        thread_qs = self.thread_query.exclude(route__script_protected=True)

        uid_to_uuid_list = list(thread_qs.values_list('uid', 'import_uid'))
        self.uuid_to_uid.update({uuid: uid for uid, uuid in uid_to_uuid_list})
        self.deleted_thread_uuids = {import_uid for __, import_uid in uid_to_uuid_list}

        fast_delete_package_threads_and_routes(self.package, log=self.log)

        self.delete_empty_routes()


class SupplierRouteUpdater(RouteUpdater):
    supplier = None

    @classmethod
    def create_route_updater(
        cls,
        supplier,
        log=None,
        saver=RouteSaver,
        mask_updater=MaskUpdater,
        thread_number_builder=CompactThreadNumberBuilder,
        update_fields=None,
        remove_old_routes=False
    ):
        route_query = Route.objects.filter(supplier=supplier, two_stage_package=None)
        route_updater = cls(
            route_query,
            log=log,
            saver=saver,
            mask_updater=mask_updater,
            thread_number_builder=thread_number_builder,
            remove_old_routes=remove_old_routes,
            update_fields=update_fields,
            _from_fabric=True
        )
        route_updater.supplier = supplier
        route_updater.prepare()

        return route_updater

    def do_remove_old_routes(self):
        thread_qs = self.thread_query.exclude(route__script_protected=True)

        uid_to_uuid_list = list(thread_qs.values_list('uid', 'import_uid'))
        self.uuid_to_uid.update({uuid: uid for uid, uuid in uid_to_uuid_list})
        self.deleted_thread_uuids = {import_uid for __, import_uid in uid_to_uuid_list}

        fast_delete_supplier_threads_and_routes(self.supplier, log=self.log)

        self.delete_empty_routes()


def get_simple_schedule_validator(log):
    schedule_validator = ScheduleValidator(log)
    schedule_validator.add_thread_validator(simple_thread_validator)

    return schedule_validator


def get_route_loader_with_default_validator(log):
    schedule_validator = ScheduleValidator(log)
    schedule_validator.add_thread_validator(simple_thread_validator)

    route_loader = RouteLoader(log)
    route_loader.add_presave_thread_hook(schedule_validator.validate_thread)

    return route_loader, schedule_validator


def get_values(obj, exclude=('id',)):
    return [get_field_value(obj, f) for f in obj._meta.local_fields if f.column not in exclude]


def get_columns(model, exclude=('id',)):
    return [f.column for f in model._meta.local_fields if f.column not in exclude]


def get_field_value(obj, field):
    if field.rel is None:
        value = field._get_val_from_obj(obj)
        if is_protected_type(value):
            return value
        else:
            return field.value_to_string(obj)
    else:
        return getattr(obj, field.attname)
