# coding: utf-8

from travel.rasp.admin.lib.logs import get_collector


VALIDATION_CONTEXT = u"Schedule Validation"


class ValidationFilter(object):
    def filter(self, record):
        return getattr(record, 'context', u"").startswith(VALIDATION_CONTEXT)


class ScheduleValidator(object):
    def __init__(self, log=None):
        self.log = log
        self.collector = get_collector('')
        self.collector.addFilter(ValidationFilter())

        self.route_validators = []
        self.thread_validators = []

    def validate_route(self, route):
        try:
            self.log.add_context(VALIDATION_CONTEXT)
            self.log.add_context(route.route_uid)

            for validator in self.route_validators:
                validator(route, self.log)

            for thread in route.threads:
                self.log.add_context(thread.uid)
                try:
                    for validator in self.thread_validators:
                        validator(thread, self.log)
                finally:
                    self.log.remove_context(thread.uid)

        finally:
            self.log.remove_context(route.route_uid)
            self.log.remove_context(VALIDATION_CONTEXT)

    def validate_thread(self, thread):
        try:
            self.log.add_context(VALIDATION_CONTEXT)
            self.log.add_context(thread.route.route_uid)
            self.log.add_context(thread.uid)

            for validator in self.thread_validators:
                validator(thread, self.log)

        finally:
            self.log.remove_context(thread.uid)
            self.log.remove_context(thread.route.route_uid)
            self.log.remove_context(VALIDATION_CONTEXT)

    def add_route_validator(self, validator):
        self.route_validators.append(validator)

    def add_thread_validator(self, thread):
        self.thread_validators.append(thread)

    def get_message(self):
        return self.collector.get_collected().replace(VALIDATION_CONTEXT + u": ", u"")

    def add_report_to_reporter(self, reporter):
        message = self.get_message()
        if message:
            reporter.add_report(u"Ошибки в маршрутах", message)


def simple_thread_validator(thread, log):
    if len(thread.rtstations) < 2:
        log.error(u"Количество станций в нитке меньше 2")
        return

    first_rts = thread.rtstations[0]
    if first_rts.tz_arrival is not None:
        log.error(u"Прибытие на первую станцию не None")

    if first_rts.tz_departure != 0:
        log.error(u"Отправление с первой станции не 0")
        return

    for index, rts in enumerate(thread.rtstations):
        if not rts.station:
            true_order = index + 1
            log.error(u"Отсутствует станция на остановке c порядковым номером %s", true_order)
            return

    for index, middle_rts in enumerate(thread.rtstations[1:-1]):
        true_order = index + 2  # порядковый номер станции
        context = u"%s %s" % (true_order, middle_rts.station.title)

        try:
            log.add_context(context)

            if middle_rts.tz_arrival is None:
                log.error(u"Прибытие на заполнено")
                continue

            if middle_rts.tz_departure is None:
                log.error(u"Отправление на заполнено")
                continue

            if middle_rts.tz_arrival > middle_rts.tz_departure:
                log.error(u"Прибытие позже отправления")
                continue

        finally:
            log.remove_context(context)

    last_rts = thread.rtstations[-1]

    if last_rts.tz_departure is not None:
        log.error(u"На последней станции отправление не None")

    if last_rts.tz_arrival is None:
        log.error(u"На последней станции не указано прибытие")
        return
