# coding: utf-8

from travel.rasp.admin.lib.logs import get_current_file_logger
from travel.rasp.admin.scripts.schedule.utils import RaspImportError


log = get_current_file_logger()


class FillMiddleRTStationError(RaspImportError):
    pass


def fill_middle_times(rtstations, use_distance=False, ignore_bad_duration=False,
                      interpret_fuzzy=True, standard_stop_time=1):
    u"""
    Заполняет средние значения для departure и arrival, если возможно.
    Дополнительные аттрибуты для rtstation:
        distance - расстояние от начала маршрута

    @param rtstations:
    @param use_distance:
    @return: was_filled
    """
    # Заполняем не заполненные arrival руководствуясь расстоянием или средним значением

    start_interval = rtstations[0]
    start_interval.distance = 0

    end_interval = rtstations[-1]

    if end_interval.arrival is None:
        raise FillMiddleRTStationError(u"Не указано время прибытия на конечную станцию")

    # Исходя из этой проверки мы знаем, что в конце концов все промежуточные времена будут заполнены

    was_filled = False
    interval = []
    for rts in rtstations[1:]:
        if rts.arrival is not None:
            if interval:
                fill_times_in_interval(start_interval, rts, interval,
                                       use_distance, ignore_bad_duration,
                                       interpret_fuzzy=interpret_fuzzy)
                was_filled = True
                interval = []

            start_interval = rts
        # Если не известно прибытие и отправление
        elif rts.departure is None:
            interval.append(rts)

    was_filled = fill_empty_arrivals_or_departures(rtstations, standard_stop_time) or was_filled

    return was_filled


def fill_times_in_interval(start_interval, end_interval, interval,
                           use_distance=False, ignore_bad_duration=False,
                           interpret_fuzzy=True):

    interval_duration = end_interval.arrival - start_interval.departure

    if interval_duration <= 0 and ignore_bad_duration:
        for rts in interval:
            rts.arrival = start_interval.arrival
            rts.departure = start_interval.departure

        return

    if not interval_duration > 0:
        raise FillMiddleRTStationError(u"Общее время в пути на сегменте %s %s должно быть больше нуля" %
                                       (start_interval.station, end_interval.station))

    need_distances_in_rts = [start_interval] + interval + [end_interval]

    distances_filled = is_good_distances(need_distances_in_rts)

    if use_distance and not distances_filled:
        log.warning(u"Дистанции заполнены не верно, вычисляем промежуточное время не исользуя расстояние")

    if use_distance and distances_filled:
        # Указана дистанция пробуем вычислить аривал по ней
        # Так как data одинакова для началной станции и второй станции,
        # то нужно вручную обнулить расстояние до начальной станции
        start_distance = start_interval.distance

        interval_distance = end_interval.distance - start_distance

        average_speed = float(interval_distance) / interval_duration

        for rts in interval:
            rts.is_fuzzy = True
            if interpret_fuzzy:
                rts.is_searchable_from = False
                rts.in_station_schedule = False

            distance_from_start = rts.distance - start_distance

            arrival = distance_from_start / average_speed + start_interval.departure

            rts.arrival = int(arrival)
            rts.departure = rts.arrival + 1
    else:

        leg_count = len(interval) + 2

        p2p_duration = interval_duration / leg_count

        current_duration = start_interval.departure

        for rts in interval:
            rts.is_fuzzy = True
            if interpret_fuzzy:
                rts.is_searchable_from = False
                rts.in_station_schedule = False

            current_duration += p2p_duration
            rts.arrival = current_duration - 1
            rts.departure = current_duration


def fill_last_stations_using_middle_speed(rtstations, middle_speed, interpret_fuzzy=True):
    km_per_minutes = middle_speed / 60.

    if rtstations[-1].arrival is not None:
        return

    last_filled_index = None
    last_filled_rts = None

    for index, rts in enumerate(rtstations):
        if rts.departure is not None:
            last_filled_index = index
            last_filled_rts = rts

    if last_filled_index is None:
        raise RaspImportError(u"Не можем расчитать время на конечных станциях: не заполнены времена"
                              u" отпралений в маршруте")

    need_distances_in_rts = rtstations[last_filled_index:]

    if not is_good_distances(need_distances_in_rts):
        raise RaspImportError(u"Не можем расчитать время на конечных станциях: не верно указаны"
                              u" расстояния между станциями")

    last_distance = last_filled_rts.distance
    last_departure = last_filled_rts.departure

    for rts in rtstations[last_filled_index + 1:]:
        interval_distance = rts.distance - last_distance
        duration = int(km_per_minutes * interval_distance)

        rts.arrival = last_departure + duration
        rts.departure = rts.arrival + 1

        last_distance = rts.distance
        last_departure = rts.departure

        rts.is_fuzzy = True
        if interpret_fuzzy:
            rts.is_searchable_from = False
            rts.in_station_schedule = False

        log.debug(u"Рассчитали время прибытия на одну из последних станций %s %s", rts.station.title, rts.arrival)

    rtstations[-1].departure = None


def is_good_distances(rts_list):
    for rts in rts_list:
        if not hasattr(rts, 'distance'):
            return False

    current_distance = -1

    is_good_dist = True

    for rts in rts_list:
        distance = getattr(rts, 'distance', None)

        if distance is None:
            log.warning(u"Не указана дистанция на станции %s", rts.station.title)
            is_good_dist = False
            break

        if not isinstance(distance, (float, int)):
            log.warning(u"Не верно указана дистанция '%s' на станции %s", distance, rts.station.title)
            is_good_dist = False
            break

        if distance < 0 or distance <= current_distance:
            log.warning(u"Дистанция %s на станции %s не больше предыдущей или меньше нуля", distance, rts.station.title)
            is_good_dist = False
            break

        current_distance = distance

    return is_good_dist


def fill_empty_arrivals_or_departures(rtstations, standard_stop_time):
    was_filled = False

    for rts in rtstations[1:-1]:
        if rts.arrival is None and rts.departure is not None:
            rts.arrival = rts.departure - standard_stop_time
            was_filled = True

        if rts.departure is None and rts.arrival is not None:
            rts.departure = rts.arrival + standard_stop_time
            was_filled = True

    return was_filled
