#!/usr/bin/env
# coding: utf-8
# Скрипт заполнения координат городов из геокодера, геобазы или по жд станции
# для городов с аэропортами и по жд станции для остальных

import travel.rasp.admin.scripts.load_project  # noqa

import logging

from django.db import transaction
from django.db.models import Q

from common.models.geo import Settlement
from common.utils.geobase import geobase
from travel.rasp.admin.lib.geocoder import get_geocode_coords
from travel.rasp.admin.lib.logs import print_log_to_stdout, create_current_file_run_log, get_script_log_context, ylog_context


log = logging.getLogger(__name__)


@transaction.atomic
def fill_settlement_coords():
    log.info("Start")

    # Сначала заполним города с аэропортами - им нужна высокая точность
    # Не презаписываем координаты, если они уже есть
    avia_settlements = Settlement.objects.filter(
        hidden=False,
        type_choices__contains='plane'
    ).filter(
        Q(latitude__isnull=True) | Q(longitude__isnull=True)
    )

    for settlement in avia_settlements:
        latitude = None
        longitude = None

        # Сначала пробуем геокодером
        try:
            latitude, longitude = get_geocode_coords(settlement)
            log.info("Use geocoder: %s", unicode(settlement))
        except Exception:
            # Потом геобазой
            if settlement._geo_id:
                r = geobase.region_by_id(settlement._geo_id)
                latitude = r.latitude
                longitude = r.longitude
                log.info("Use geobase: %s", unicode(settlement))
            else:
                # крупная жд станция
                stations = settlement.station_set.filter(
                    t_type=1, hidden=False, majority__in=(1, 2)
                )

                if stations:
                    latitude = stations[0].latitude
                    longitude = stations[0].longitude
                    log.info("Use station: %s", unicode(settlement))

        if latitude and longitude:
            settlement.latitude = latitude
            settlement.longitude = longitude
            settlement.save()

    # Для остальных городов работаем по-старому
    for settlement in Settlement.objects.exclude(type_choices__contains='plane'):
        stations = settlement.station_set.filter(t_type=1, hidden=False, majority__in=(1, 2))
        if stations:
            settlement.longitude = stations[0].longitude
            settlement.latitude = stations[0].latitude
            settlement.save()

    log.info("Finish")


if __name__ == '__main__':
    with ylog_context(**get_script_log_context()):
        create_current_file_run_log()

        from time import time
        start_t = time()
        from optparse import OptionParser

        parser = OptionParser()
        parser.add_option("-v", "--verbose", dest="verbose", action="store_true")

        (options, args) = parser.parse_args()

        if options.verbose:
            print_log_to_stdout(log)

        fill_settlement_coords()
        log.info(u'Done %s', time() - start_t)
