# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
from collections import defaultdict

from common.utils.point_slugs import VALID_SLUG_RE, STATION_SLUG_VARIANTS, make_station_slug_variant, \
    make_settlement_slug_variant, SETTLEMENT_SLUG_VARIANTS
from common.utils.progress import PercentageStatus
from travel.rasp.admin.lib import tmpfiles
from travel.rasp.admin.lib.mysqlutils import MysqlModelUpdater

log = logging.getLogger(__name__)

SLUG_CACHE = set()


def fill_slug_all():
    from common.models.geo import Station, CityMajority, Settlement, StationMajority
    SLUG_CACHE.clear()
    tmp_dir = tmpfiles.get_tmp_dir()
    fields = ['slug']
    with MysqlModelUpdater(Settlement, tmp_dir, fields=fields) as settlement_updater, \
            MysqlModelUpdater(Station, tmp_dir, fields=fields) as station_updater:
        status = PercentageStatus(Settlement.objects.count() + Station.objects.count(), log)

        def fill_slug_stations(points):
            fill_slug_list(status, points, STATION_SLUG_VARIANTS, make_station_slug_variant, station_updater)

        def fill_slug_settlements(points):
            fill_slug_list(status, points, SETTLEMENT_SLUG_VARIANTS, make_settlement_slug_variant, settlement_updater)

        # заполняем slug для списков в порядке убывания важности, чтобы названия столиц стран не конфликтовали с
        # названиями станций и имели самые короткие и красивые имена
        fill_slug_settlements(Settlement.objects.filter(
            majority__id=CityMajority.CAPITAL_ID, hidden=False))
        fill_slug_settlements(Settlement.objects.filter(
            majority__id__in=[CityMajority.REGION_CAPITAL_ID, CityMajority.POPULATION_MILLION_ID], hidden=False))
        fill_slug_settlements(Settlement.objects.filter(
            majority__id__gt=CityMajority.POPULATION_MILLION_ID, hidden=False))
        fill_slug_stations(Station.objects.filter(
            majority__id__lte=StationMajority.NOT_IN_SEARCH_ID, hidden=False))
        fill_slug_stations(Station.objects.filter(
            majority__id__gt=StationMajority.NOT_IN_SEARCH_ID, hidden=False))
        fill_slug_settlements(Settlement.objects.filter(hidden=True))
        fill_slug_stations(Station.objects.filter(hidden=True))
    SLUG_CACHE.clear()


def fill_slug_list(status, points, slug_variants, make_slug_variant, updater):
    for args in slug_variants:
        slugs_list = defaultdict(list)
        for obj in points:
            slug = make_slug_variant(obj, **args)
            slugs_list[slug].append(obj)

        next_try_list = []
        for slug, objects in slugs_list.items():
            if slug and len(objects) == 1 and slug not in SLUG_CACHE:
                write_slug(status, objects[0], slug, updater)
            else:
                next_try_list.extend(objects)
        points = next_try_list
    for miss in points:
        log.error('Не удалось сгенерировать уникальный slug для {}'.format(miss.point_key))


def write_slug(status, obj, slug, updater):
    try:
        status.step()
        if not VALID_SLUG_RE.match(slug):
            log.error('invalid slug={} obj.id={} title_en={}'.format(slug, obj.id, obj.title_en))
            return
        obj.slug = slug
        updater.add(obj)
        SLUG_CACHE.add(slug)
    except Exception:
        log.error('Ошибка при записи slug=%r, object=%r')
        raise
