# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

from collections import defaultdict
from itertools import groupby, chain

from django.db.models import Q

from common.models.geo import Settlement, Station, Station2Settlement
from common.models.transport import TransportType, TransportSubtype
from common.data_api.baris.instance import baris
from route_search.models import ZNodeRoute2
from travel.rasp.library.python.sitemap.models.common import CommonSitemap


class SearchAllDaysSitemap(CommonSitemap):
    changefreq = 'daily'
    priority = '0.5'
    protocol = 'https'

    ALL_TRANSPORT_ID = 0

    _t_type_id_to_prefix = {
        ALL_TRANSPORT_ID: 'all-transport',
        TransportType.TRAIN_ID: 'train',
        TransportType.PLANE_ID: 'plane',
        TransportType.BUS_ID: 'bus',
        TransportType.RIVER_ID: 'water',
        TransportType.SEA_ID: 'water',
        TransportType.SUBURBAN_ID: 'suburban',
        TransportType.WATER_ID: 'water',
    }

    train_and_suburban_set = {TransportType.TRAIN_ID, TransportType.SUBURBAN_ID}

    _items_cache = None

    def __init__(self, use_cache=True):
        self.settlements_data = {
            settlement_id:
                {
                    'slug': slug,
                    'title': title.lower()
                } for settlement_id, slug, title in Settlement.objects.values_list('id', 'slug', 'title')
        }

        self.stations_data = {
            station_id:
                {
                    'slug': slug,
                    'title': title.lower()
                } for station_id, slug, title in Station.objects.values_list('id', 'slug', 'title')
        }

        self.plane_stations_settlement = {
            s2s.station_id: s2s.settlement_id
            for s2s in Station2Settlement.objects.all()
        }
        self.plane_stations_settlement.update({
            station_id: settlement_id
            for station_id, settlement_id in
                Station.objects.values_list('id', 'settlement_id').filter(
                    t_type_id=TransportType.PLANE_ID, settlement_id__isnull=False
                )
        })

        self.use_cache = use_cache

        super(SearchAllDaysSitemap, self).__init__()

    def items(self):
        if not self.use_cache:
            return list(self._items(to_make_avia_items=True, to_make_train_suburban=True))

        if not SearchAllDaysSitemap._items_cache:
            SearchAllDaysSitemap._items_cache = list(self._items(to_make_avia_items=True, to_make_train_suburban=True))
        return SearchAllDaysSitemap._items_cache

    def location(self, item):
        from_slug, to_slug, transport_type_code = item
        # Для электричек поиски на сегодня, для остальных - на все дни
        pattern = '/{}/{}--{}/today' if transport_type_code == 'suburban' else '/{}/{}--{}'
        return pattern.format(transport_type_code, from_slug, to_slug)

    def _items(self, to_make_avia_items, to_make_train_suburban=False):
        if to_make_avia_items:
            for item in self._make_avia_items():
                yield item

        query = self._get_query(self._build_znr_query())
        routes = query

        if to_make_train_suburban:
            train_suburban_query = self._get_query(self._build_znr_train_suburban_query())

            train_suburban_routes = []
            for values in train_suburban_query:
                values = list(values)
                values[2] = TransportType.SUBURBAN_ID
                train_suburban_routes.append(tuple(values))

            routes = chain(query, train_suburban_routes)

        for (settlement_from_id, settlement_to_id), noderoute_iter in groupby(
            routes,
            lambda (settlement_from, settlement_to, t_type, station_from, station_to, good_for_start, good_for_finish):
            (settlement_from, settlement_to)
        ):
            # Словари более узких поисков сгруппированные внутри более широких и сгруппированные по типу транспорта
            # Словарь множеств станций, таких что есть поиск город-станция
            city_city_by_city_st_set = defaultdict(set)
            # Словарь множеств станций, таких что есть поиск станция-город
            city_city_by_st_city_set = defaultdict(set)
            # Словарь количеств поисков станция-станция для пары станция-город
            city_st_by_st_st_count = defaultdict(lambda: defaultdict(int))
            # Словарь количеств поисков станция-станция для пары город-станция
            st_city_by_st_st_count = defaultdict(lambda: defaultdict(int))
            # Множество типов транспорта между городами
            t_type_ids = set()
            # Множества ЖД-станций в городах
            rail_stations_from = set()
            rail_stations_to = set()

            for (_, _, t_type_id, station_from_id, station_to_id, good_for_start, good_for_finish) in set(noderoute_iter):
                t_type_ids.add(t_type_id)

                if t_type_id in self.train_and_suburban_set and station_from_id != station_to_id:
                    # Для ЖД возвращаем все поиски станция-станция
                    yield self._make_stations_item(station_from_id, station_to_id, t_type_id)

                if settlement_from_id and settlement_to_id and settlement_from_id == settlement_to_id:
                    continue

                # Для ЖД в словари кладем станции без города или станции в городе с good_for_start или good_for_finish
                if not t_type_id in self.train_and_suburban_set or not settlement_to_id or good_for_finish:
                    city_city_by_city_st_set[t_type_id].add(station_to_id)
                if not t_type_id in self.train_and_suburban_set or not settlement_from_id or good_for_start:
                    city_city_by_st_city_set[t_type_id].add(station_from_id)

                if t_type_id in self.train_and_suburban_set:
                    if settlement_from_id and good_for_start:
                        city_st_by_st_st_count[t_type_id][station_to_id] += 1
                    if settlement_to_id and good_for_finish:
                        st_city_by_st_st_count[t_type_id][station_from_id] += 1

                    if settlement_to_id and settlement_from_id and good_for_start and good_for_finish:
                        rail_stations_from.add(station_from_id)
                        rail_stations_to.add(station_to_id)

            for t_type_id in city_city_by_city_st_set:
                if t_type_id == TransportType.PLANE_ID:
                    continue

                if settlement_from_id and settlement_to_id:
                    if (t_type_id not in self.train_and_suburban_set or
                        (t_type_id == TransportType.SUBURBAN_ID and
                         len(city_city_by_city_st_set[t_type_id]) > 1 and len(city_city_by_st_city_set[t_type_id]) > 1)
                    ):
                        # Возвращаем поиски город-город, если не нужно сужать
                        yield self._make_settlements_item(settlement_from_id, settlement_to_id, t_type_id)

                    elif t_type_id == TransportType.TRAIN_ID:
                        # Для поезда возвращаем город-город
                        # Если для каждого города есть только одна одноименная станция, то сужаем до станций (станция-станция уже добавлена)
                        station_from_same_title = self._get_same_title_station(city_city_by_st_city_set[t_type_id], settlement_from_id)
                        station_to_same_title = self._get_same_title_station(city_city_by_city_st_set[t_type_id], settlement_to_id)
                        if not station_from_same_title or not station_to_same_title:
                            yield self._make_settlements_item(settlement_from_id, settlement_to_id, t_type_id)

                if t_type_id not in self.train_and_suburban_set:
                    # Для не ЖД возвращаем все поиски станция-город и город-станция
                    if settlement_from_id:
                        for station_to_id in city_city_by_city_st_set[t_type_id]:
                            yield self._make_to_station_item(settlement_from_id, station_to_id, t_type_id)
                    if settlement_to_id:
                        for station_from_id in city_city_by_st_city_set[t_type_id]:
                            yield self._make_from_station_item(station_from_id, settlement_to_id, t_type_id)

                else:
                    # Для ЖД возвращаем поиски станция-город и город-станция только если нельзя их сузить
                    # Сужение до одноименных станций поездов уже произведено
                    for station_to_id in city_st_by_st_st_count[t_type_id]:
                        if settlement_from_id and city_st_by_st_st_count[t_type_id][station_to_id] > 1:
                            yield self._make_to_station_item(settlement_from_id, station_to_id, t_type_id)
                    for station_from_id in st_city_by_st_st_count[t_type_id]:
                        if settlement_to_id and st_city_by_st_st_count[t_type_id][station_from_id] > 1:
                            yield self._make_from_station_item(station_from_id, settlement_to_id, t_type_id)

            # Поиски город-город всеми типами транспорта
            if settlement_from_id and settlement_to_id and settlement_from_id != settlement_to_id:
                if to_make_avia_items and (settlement_from_id, settlement_to_id) in self.avia_settlements_searches:
                    t_type_ids.add(TransportType.PLANE_ID)
                if len(t_type_ids) > 1:
                    if (
                        not t_type_ids.issubset(self.train_and_suburban_set) or
                        (len(rail_stations_from) > 1 and len(rail_stations_to) > 1)
                    ):
                        yield self._make_settlements_item(settlement_from_id, settlement_to_id, self.ALL_TRANSPORT_ID)
                    elif len(rail_stations_from) == 1 and len(rail_stations_to) > 1:
                        yield self._make_from_station_item(
                            rail_stations_from.pop(), settlement_to_id, self.ALL_TRANSPORT_ID)
                    elif len(rail_stations_from) > 1 and len(rail_stations_to) == 1:
                        yield self._make_to_station_item(
                            settlement_from_id, rail_stations_to.pop(), self.ALL_TRANSPORT_ID)
                    elif len(rail_stations_from) == 1 and len(rail_stations_to) == 1:
                        yield self._make_stations_item(
                            rail_stations_from.pop(), rail_stations_to.pop(), self.ALL_TRANSPORT_ID)

    def _make_avia_items(self):
        """
        Формирование списка поисков самолетами из БАРиС.
        В процессе также формируется self.avia_settlements_searches
        """
        self.avia_settlements_searches = set()
        from_station_items = set()
        to_station_items = set()

        for station_from_id, station_to_id in baris.get_p2p_summary():
            if station_from_id not in self.stations_data or station_to_id not in self.stations_data:
                continue

            settlement_from_id = self.plane_stations_settlement.get(station_from_id, None)
            settlement_to_id = self.plane_stations_settlement.get(station_to_id, None)

            if settlement_from_id and settlement_to_id and settlement_from_id == settlement_to_id:
                continue

            # Поиск станция-город
            if settlement_to_id:
                from_item = (station_from_id, settlement_to_id)
                if from_item not in from_station_items:
                    from_station_items.add(from_item)
                    yield self._make_from_station_item(station_from_id, settlement_to_id, TransportType.PLANE_ID)

            # Поиск город-станция
            if settlement_from_id:
                to_item = (settlement_from_id, station_to_id)
                if to_item not in to_station_items:
                    to_station_items.add(to_item)
                    yield self._make_to_station_item(settlement_from_id, station_to_id, TransportType.PLANE_ID)

            # Поиск город-город
            if settlement_from_id and settlement_to_id:
                settlements_item = (settlement_from_id, settlement_to_id)
                if settlements_item not in self.avia_settlements_searches:
                    self.avia_settlements_searches.add(settlements_item)
                    yield self._make_settlements_item(settlement_from_id, settlement_to_id, TransportType.PLANE_ID)

    def _get_same_title_station(self, stations_ids, settlement_id):
        if len(stations_ids) == 1:
            station_id = stations_ids.pop()
            if self.stations_data[station_id]['title'] == self.settlements_data[settlement_id]['title']:
                return station_id

    def _get_query(self, query):
        return (
            ZNodeRoute2.objects.filter(query)
                .values_list(
                    'settlement_from_id', 'settlement_to_id', 't_type_id',
                    'station_from_id', 'station_to_id', 'good_for_start',  'good_for_finish'
                )
                .distinct()
                .order_by('settlement_from_id', 'settlement_to_id')
        )

    def _build_znr_query(self):
        """
        Запрос к ZNodeRoute2
        """
        return Q(t_type_id__in=list(self._t_type_id_to_prefix.keys()))

    def _build_znr_train_suburban_query(self):
        return Q(t_type_id=TransportType.TRAIN_ID, thread__t_subtype__code__in=TransportSubtype.get_train_search_codes())

    def _make_stations_item(self, station_from_id, station_to_id, url_transport_id):
        return (
            self.stations_data[station_from_id]['slug'],
            self.stations_data[station_to_id]['slug'],
            self._t_type_id_to_prefix[url_transport_id]
        )

    def _make_settlements_item(self, settlement_from_id, settlement_to_id, url_transport_id):
        return (
            self.settlements_data[settlement_from_id]['slug'],
            self.settlements_data[settlement_to_id]['slug'],
            self._t_type_id_to_prefix[url_transport_id]
        )

    def _make_from_station_item(self, station_id, settlement_to_id, url_transport_id):
        return (
            self.stations_data[station_id]['slug'],
            self.settlements_data[settlement_to_id]['slug'],
            self._t_type_id_to_prefix[url_transport_id]
        )

    def _make_to_station_item(self, settlement_from_id, station_id, url_transport_id):
        return (
            self.settlements_data[settlement_from_id]['slug'],
            self.stations_data[station_id]['slug'],
            self._t_type_id_to_prefix[url_transport_id]
        )


class LastochkaSitemap(SearchAllDaysSitemap):

    def _build_znr_query(self):
        lastochka_subtypes = [
            TransportSubtype.LAST_SUBURBAN_ID,  # ласточки-электрички
            TransportSubtype.LASTDAL_ID  # ласточки-поезда
        ]

        return super(LastochkaSitemap, self)._build_znr_query() & Q(thread__t_subtype_id__in=lastochka_subtypes)

    def items(self):
        items = super(LastochkaSitemap, self)._items(to_make_avia_items=False)
        # убираем тип транспорта
        directions = {(slug_from, slug_to) for (slug_from, slug_to, _) in items}

        return list(directions)

    def location(self, item):
        slug_from, slug_to = item
        return '/lastochka/{}--{}'.format(slug_from, slug_to)
