# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

from collections import defaultdict

from common.models.geo import StationTerminal, Station
from common.models.transport import TransportType
from travel.rasp.library.python.sitemap.models.common import CommonSitemap


class StationScheduleSitemap(CommonSitemap):
    def __init__(self):
        station_terminals = defaultdict(list)
        for terminal in StationTerminal.objects.all():
            station_terminals[terminal.station_id].append(terminal.name)

        self._items = list()
        for station in Station.objects.all():
            if station.type_choices:
                if station.t_type_id == TransportType.TRAIN_ID:
                    # Отправление для основного подтипа страницы добавляем всегда
                    self._items.append(StationScheduleItem(station.id))

                    choices_set = station.type_choices_set
                    if 'train' in choices_set:
                        # Для поездов добавляем прибытие
                        self._items.append(StationScheduleItem(station.id, event='arrival'))
                        if 'suburban' in choices_set:
                            # Если есть поезда и электрички, то добавляем страницы с /suburban
                            self._items.append(StationScheduleItem(station.id, subtype='suburban'))

                elif station.t_type_id == TransportType.PLANE_ID:
                    # Прибытие и отправление для аэропорта добавляем всегда
                    self._items.append(StationScheduleItem(station.id))
                    self._items.append(StationScheduleItem(station.id, event='arrival'))
                    # Если есть терминалы, то добавляем отправление и прибытие для каждого
                    if station.id in station_terminals:
                        for terminal_name in station_terminals[station.id]:
                            self._items.append(
                                StationScheduleItem(station.id, terminal=terminal_name)
                            )
                            self._items.append(
                                StationScheduleItem(station.id, event='arrival', terminal=terminal_name)
                            )

                else:
                    self._items.append(StationScheduleItem(station.id))

        super(StationScheduleSitemap, self).__init__()

    def items(self):
        return self._items

    def location(self, item):
        url_subtype = '{}/'.format(item.subtype) if item.subtype else ''

        url_params = ''
        if item.event:
            url_params ='?event={}'.format(item.event)
        if item.terminal:
            pref = '&' if url_params else '?'
            url_params = url_params + '{}terminal={}'.format(pref, item.terminal)

        return '/station/{}/{}{}'.format(item.station_id, url_subtype, url_params)


class StationScheduleItem(object):
    def __init__(self, station_id, subtype=None, event=None, terminal=None):
        self.station_id = station_id
        self.subtype = subtype
        self.event = event
        self.terminal = terminal
