#!/usr/bin/env python
# coding: utf-8

"""Формирование строки остановок электрички"""

import travel.rasp.admin.scripts.load_project  # noqa

import logging
import json

from django.conf import settings

from common.apps.suburban_events.scripts.calc_suburban_keys import create_suburban_key_from_first_rts
from common.models.schedule import RThread
from common.models.transport import TransportType
from common.xgettext.common import get_schedule_autotext
from travel.rasp.admin.lib.maintenance.flags import flags
from travel.rasp.admin.lib.logs import print_log_to_stdout, create_current_file_run_log, get_script_log_context, ylog_context
from travel.rasp.admin.lib.utils import TheEstimator


log = logging.getLogger(__name__)


test_table = u"""
Москва-Киевская	07.46	07.46	07.46	07.46	07.46	07.46	07.46	07.46	07.46	07.46
Москва-сорт.	07.51	07.51	-	-	-	-	07.51	07.51	07.51	-
Матвеевская	07.55	07.55	-	-	-	-	07.55	-	-	-
Очаково	07.59	07.59	-	-	07.59	07.59	07.59	07.59	07.59	07.59
Востряково	08.02	-	-	-	-	-	08.02	-	-	-
Солнечная	08.06	08.06	08.06	-	08.06	08.06	08.06	08.06	08.06	08.06
Переделкино	08.08	08.08	-	-	-	-	08.08	08.08	-	-
Мичуринец	08.12	-	-	-	-	-	08.12	-	-	-
Внуково	08.15	08.15	-	-	-	-	08.15	08.15	08.15	-
Лесной Городок	08.20	08.20	-	-	08.20	08.20	08.20	08.20	08.20	08.20
Толстопальцево	08.24	08.24	-	-	-	-	08.24	08.24	08.24	08.24
Кокошкино	08.26	-	-	-	-	-	08.26	08.26	-	08.26
Крекшино	08.32	08.32	-	-	-	-	08.32	08.32	08.32	08.32
Победа	08.34	08.34	-	-	-	-	08.34	08.34	-	08.34
Апрелевка	08.38	08.38	08.38	-	08.38	08.38	08.38	08.38	08.38	08.38
Дачная	08.41	08.41	-	-	08.41	08.41	-	08.41	-	08.41
Алабино	08.45	08.45	-	-	08.45	08.45	-	08.45	-	08.45
Селятино	08.48	08.48	-	-	08.48	08.48	-	-	08.48	08.48
Рассудово	08.52	-	-	-	08.52	08.52	-	-	-	08.52
Ожигово	08.55	08.55	-	-	08.55	08.55	-	-	-	08.55
Бекасово-1	09.00	09.00	09.00	-	09.00	09.00	09.00	09.00	09.00	09.00
Зосимова Пустынь	09.03	-	-	-	09.03	09.03	-	-	-	-
Нара	09.10	09.10	09.10	-	09.10	09.10	09.10	09.10	09.10	-
Латышская	09.16	09.16	-	-	09.16	09.16	-	-	09.16	-
Башкино	09.20	09.20	-	-	09.20	-	-	-	-	-
Ворсино	09.28	09.28	-	-	09.28	09.28	-	-	-	-
Балабаново	09.35	09.35	-	-	09.35	09.35	09.35	-	09.35	-
Обнинское	09.48	09.48	-	-	09.48	09.48	-	-	09.48	09.48
Шемякино	09.56	09.56	-	-	09.56	-	-	-	-	-
Малоярославец	10.05	10.05	10.05	10.05	10.05	10.05	10.05	10.05	10.05	10.05
"""

test_results = u"""
везде
кроме: Востряково, Мичуринец, Кокошкино, Рассудово, Зосимова Пустынь
Солнечная, Апрелевка, Бекасово-1, Нара, Малоярославец
без остановок
Очаково, Солнечная, Лесной Городок, Апрелевка, далее везде
Очаково, Солнечная, Лесной Городок, Апрелевка, далее везде, кроме: Башкино, Шемякино
до ст. Апрелевка везде, далее Бекасово-1, Нара, Балабаново, Малоярославец
до ст. Алабино везде, кроме: Матвеевская, Востряково, Мичуринец, далее Бекасово-1, Нара, Малоярославец
Москва-сорт., Очаково, Солнечная, Внуково, Лесной Городок, Толстопальцево, Крекшино, Апрелевка, Селятино, Бекасово-1,\
 Нара, Латышская, Балабаново, Обнинское, Малоярославец
Очаково, Солнечная, Лесной Городок, далее до ст. Бекасово-1 везде, далее Обнинское, Малоярославец
"""


def _test():
    texts = [r.strip() for r in test_results.strip().split("\n")]

    variants_by_station = []

    for row in test_table.strip().split("\n"):
        cells = row.split("\t")

        station = cells[0]
        stops = (cell != '-' for cell in cells[1:])

        variants_by_station.append([(stop, station) for stop in stops])

    routes = zip(*variants_by_station)

    for text, route in zip(texts, routes):
        generated_text = route_text(list(route[1:]))

        if generated_text != text:
            print u"Ожидалось:", text
            print u"Получилось:", generated_text
            print u"Маршрут:"
            for stop, station in route:
                print stop and "x" or " ", station


class StopsThen(object):
    def __init__(self, stops, rest):
        self.weight = rest.weight + len(stops) + 2
        self.stops = stops
        self.rest = rest

    def text(self, lang):
        """{stops}, далее {rest}"""

        key = 'suburban_stops.' + self.__class__.__name__.lower()
        text_template = get_schedule_autotext(key, lang)

        stops = u", ".join(stop.L_suburban_title(lang=lang) for stop in self.stops)
        rest = self.rest.text(lang)

        return text_template.format(stops=stops, rest=rest)


class TillStopEverywhereThen(object):
    def __init__(self, stop, rest):
        self.weight = rest.weight + 3
        self.stop = stop
        self.rest = rest

    def text(self, lang):
        """до {stop} везде, далее {rest}"""

        key = 'suburban_stops.' + self.__class__.__name__.lower()
        text_template = get_schedule_autotext(key, lang)

        stop = self.stop.L_suburban_title(lang)
        rest = self.rest.text(lang)

        return text_template.format(stop=stop, rest=rest)


class TillStopNonStopThen(object):
    def __init__(self, stop, rest):
        self.weight = rest.weight + 4
        self.stop = stop
        self.rest = rest

    def text(self, lang):
        """до {stop} без остановок, далее {rest}"""

        key = 'suburban_stops.' + self.__class__.__name__.lower()
        text_template = get_schedule_autotext(key, lang)

        stop = self.stop.L_suburban_title(lang)
        rest = self.rest.text(lang)

        return text_template.format(stop=stop, rest=rest)


class TillStopEverywhereExceptThen(object):
    def __init__(self, stop, skips, rest):
        self.weight = rest.weight + 5 + len(skips)
        self.stop = stop
        self.skips = skips
        self.rest = rest

    def text(self, lang):
        """до {stop} везде, кроме: {except_stops}, далее {rest}"""

        key = 'suburban_stops.' + self.__class__.__name__.lower()
        text_template = get_schedule_autotext(key, lang)

        stop = self.stop.L_suburban_title(lang)
        except_stops = u", ".join(skip.L_suburban_title(lang=lang) for skip in self.skips)
        rest = self.rest.text(lang)

        return text_template.format(stop=stop, rest=rest, except_stops=except_stops)


class Stops(object):
    def __init__(self, stops):
        self.weight = len(stops)
        self.stops = stops

    def text(self, lang):
        return u", ".join(stop.L_suburban_title(lang=lang) for stop in self.stops)


class Everywhere(object):
    weight = 1

    def text(self, lang):
        """везде"""

        key = 'suburban_stops.' + self.__class__.__name__.lower()

        return get_schedule_autotext(key, lang)


class NonStop(object):
    weight = 1

    def text(self, lang):
        """без остановок"""

        key = 'suburban_stops.' + self.__class__.__name__.lower()

        return get_schedule_autotext(key, lang)


class Except(object):
    def __init__(self, skips):
        self.weight = len(skips) + 1
        self.skips = skips

    def text(self, lang):
        """кроме: {except_stops}"""

        key = 'suburban_stops.' + self.__class__.__name__.lower()
        text_template = get_schedule_autotext(key, lang)

        except_stops = u", ".join(skip.L_suburban_title(lang=lang) for skip in self.skips)

        return text_template.format(except_stops=except_stops)


class EverywhereExcept(object):
    def __init__(self, skips):
        self.weight = len(skips) + 2
        self.skips = skips

    def text(self, lang):
        """везде, кроме: {except_stops}"""

        key = 'suburban_stops.' + self.__class__.__name__.lower()
        text_template = get_schedule_autotext(key, lang)

        except_stops = u", ".join(skip.L_suburban_title(lang=lang) for skip in self.skips)

        return text_template.format(except_stops=except_stops)


def split_variants(route, first):
    for split in range(min(5, len(route) - 1), len(route)):
        our, their = route[:split], route[split + 1:]
        split_stop, split_station = route[split]

        if not split_stop:
            continue

        skips = [station for stop, station in our if not stop]
        stops = [station for stop, station in our if stop]

        if their:
            rest = route_text(their, first=False)

            yield StopsThen(stops + [split_station], rest)

            if not skips:
                yield TillStopEverywhereThen(split_station, rest)
                continue

            if not stops:
                yield TillStopNonStopThen(split_station, rest)
                continue

            if len(skips) < 8:
                yield TillStopEverywhereExceptThen(split_station, skips, rest)
                continue

        else:
            if not skips:
                yield Everywhere()
                continue

            if not stops:
                # first в данном случае значит, что это - единственная фраза
                if first:
                    yield NonStop()
                    continue

            yield Stops(stops + [split_station])

            if len(skips) < 8:
                if first:
                    yield Except(skips)
                else:
                    yield EverywhereExcept(skips)

                continue


route_text_cache = {}


def route_text(route, first=True):
    """first - текст в начале строчки"""

    key = tuple(route + [first])

    cached = route_text_cache.get(key)

    if cached:
        return cached

    best_variant = None

    for variant in split_variants(route, first):
        if best_variant is None or best_variant.weight > variant.weight:
            best_variant = variant

    route_text_cache[key] = best_variant

    return best_variant


def stops_text(rtstations):
    stations_with_stops = [(
        rts.tz_arrival != rts.tz_departure and not rts.is_technical_stop,
        rts.station,
    ) for rts in rtstations]

    if (not stations_with_stops[0][0] or
            not stations_with_stops[-1][0]):
        log.error("Route stations must start and stop with stops")
        return

    return {
        lang: route_text(stations_with_stops[1:]).text(lang)
        for lang in settings.FRONTEND_LANGUAGES
    }


def walk_thread(thread, recount_schedule_on_the_fly=False):
    stops = []

    # list нужен, чтобы при пробеге отсечь последний кусок, ну и быть увренным в кэшировании
    rtstations = list(thread.path.select_related('station'))

    schedules = dict((s.station_id, s) for s in thread.stationschedule_set.all())

    for rtstation in rtstations:
        stop = rtstation.tz_arrival != rtstation.tz_departure and not rtstation.is_technical_stop
        stops.append((stop, rtstation.station.suburban_title))

        if recount_schedule_on_the_fly and rtstation.tz_arrival is None:
            create_suburban_key_from_first_rts(rtstation)

    for i, rtstation in enumerate(rtstations[:-2]):
        if rtstation.tz_arrival == rtstation.tz_departure:
            continue

        text = stops_text(rtstations[i:])
        if not text:
            log.error(u'Ошибки генерации остановок для нитки %s с остановки %s %s',
                      thread.uid, rtstation.station.id, rtstation.station.title)
            continue

        schedule = schedules.get(rtstation.station_id)

        if schedule:
            schedule.stops = text['ru']
            schedule.stops_translations = json.dumps(text, ensure_ascii=False)

            schedule.save()
        else:
            log.error("Schedule not found for rtstation %s" % rtstation.id)


def run():
    threads = RThread.objects.filter(
        t_type_id=TransportType.SUBURBAN_ID
    ).exclude(  # Не считать остановки для экспрессов от ТИС и СвПК
        express_type__isnull=False,
        supplier__code__in=['tis', 'svrpk']
    )

    if flags['partial_preparation']:
        threads = threads.filter(route__rthread__changed=True).distinct()

    counter = TheEstimator(threads.count(), log)

    for thread in threads:
        walk_thread(thread)

        counter.update(1)


if __name__ == '__main__':
    with ylog_context(**get_script_log_context()):
        create_current_file_run_log()

        from optparse import OptionParser

        parser = OptionParser()
        parser.add_option("-v", "--verbose", dest="verbose", action="store_true")

        (options, args) = parser.parse_args()

        if options.verbose:
            print_log_to_stdout(log)

        run()
