# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

""" Хранит различные вспомогательные константы и методы для работы с данными.
    Нужен, чтобы избежать копипастов в заполняющих скриптах.
"""

import os
import time
import sys

import warnings
import logging
import codecs
from subprocess import call
from datetime import timedelta, date

import six
from django.conf import settings
from django.db import transaction

from common.data_api.file_wrapper.config import get_wrapper_creator
from common.models.geo import Station
from common.models.transport import TransportType
from common.utils.warnings import RaspDeprecationWarning
from travel.rasp.admin.lib.lock import acquire_exclusive_lock
from travel.rasp.admin.lib.exceptions import SimpleUnicodeException
from travel.rasp.admin.scripts.schedule.utils.route_loader import CompactThreadNumberBuilder
from travel.rasp.admin.scripts.utils.file_wrapper.mds_utils import get_run_state_key
from travel.rasp.admin.scripts.utils.file_wrapper.registry import FileType


log = logging.getLogger(__name__)


def get_aviacompany(code, log=log):
    from schedule.utils.plane import AviaCompanyFinder
    warnings.warn('[2015-08-06] Use AviaCompanyFinder', RaspDeprecationWarning, stacklevel=2)

    company_finder = AviaCompanyFinder()
    return company_finder.get_company(code)


def comma_separated_optparse_callback(option, opt_str, value, parser, type=unicode):
    values = filter(None, value.split(u','))
    values = map(lambda x: type(x.strip()), values)
    setattr(parser.values, option.dest, values)


def set_going(year_days, day, month, going):
    """ Ставит в дни хождения (year_days) в нужный день (day)
        нужного месяца (month) 1 или 0 (going) и возвращает результат.
    """
    month = int(month)
    day = int(day)
    if month not in range(1, 13) or day not in range(1, 32):
        err_msg = " Неверный месяц или день: %s, %s!" % (str(month), str(day))
        raise Exception(err_msg)
    index = (month - 1) * 31 + day - 1
    year_days = year_days[:index] + str(going) + year_days[index + 1:]
    return year_days


def airport_search(code=None, title=None, title_en=None, not_raise=False):
    u""" Ищем аэропорт по коду, а затем по названию( русскому потом английскому)
    """
    stations = []
    if code:
        stations = Station.objects.filter(sirena_id=code)
        if not stations:
            stations = Station.objects.filter(code_set__system__code='iata',
                                              code_set__code=code)
        if not stations:
            stations = Station.objects.filter(code_set__system__code='icao',
                                              code_set__code=code)
    if not stations and title:
        stations = Station.objects.filter(title=title, t_type_id=TransportType.get_plane_type().id)
    if not stations and title_en:
        stations = Station.objects.filter(title_en=title_en,
                                          t_type_id=TransportType.get_plane_type().id)

    if len(stations) == 1:
        return stations[0]
    elif not stations:
        if not_raise:
            return None
        raise Station.DoesNotExist(code, title, title_en)
    elif len(stations) > 1:
        if not_raise:
            return None
        raise Station.MultipleObjectsReturned(code, title, title_en)


@transaction.atomic
def full_create_thread(thread, log=log, make_import_uid=True):
    """
    Добавляет нитку со станциями,
    станции должны лежать списком в аттрибуте rtstations
    Возвращает добавленную нитку.
    """

    if make_import_uid and not thread.import_uid:
        thread.gen_import_uid()

    thread.changed = True

    if not thread.ordinal_number:
        CompactThreadNumberBuilder([thread.route]).build_for(thread)

    thread.save(force_insert=True)

    if len(thread.rtstations) < 2:
        log.warning(u"Унитки %s %d станций", thread.uid, len(thread.rtstations))
    if thread.rtstations:
        for rts in thread.rtstations:
            rts.thread = thread
            rts.save(force_insert=True)

    log.info(u"Добавили нитку %s%s", thread.uid,
             u" #" + thread.template_text if thread.template_text else u"")
    return thread


@transaction.atomic
def full_create_route(route, log=log, make_import_uid=True):
    """
    Добавляет маршрут с нитками,
    нитки должны лежать списком в аттрибуте threads.
    Возвращает добавленный маршрут.
    """

    route.script_protected = False
    if not route.comment:
        route.comment = ''
    route.save(force_insert=True)
    for thread in route.threads:
        thread.route = route
        full_create_thread(thread, log, make_import_uid=make_import_uid)
    log.info(u"Маршрут %s добавлен успешно", route.route_uid)
    return route


def esr_leading_zeros(code):
    try:
        if len(code) != 3 and len(code) != 6:
            return u"%06d" % int(code)
    except ValueError:
        pass

    return code


def spk_leading_zeros(code):
    if len(code) != 4:
        code = u"%04d" % int(code)
    return code


def delete_old_export_files(template, older_then=1, log=log):
    u"""Удаляет позавчерашний(или старее чем указаон в older_then) экспорт и раньше.
    template - это шаблон, в который подставляется дата экспорта."""
    border = date.today() - timedelta(older_then)
    export_dir = os.path.dirname(template)
    log.debug(u"export_dir = %s", export_dir)
    for filename in os.listdir(export_dir):
        try:
            filepath = os.path.join(export_dir, filename)
            day = date(*time.strptime(filepath, template)[:3])
            if day < border:
                log.info(u"Удаляем старый файл экспорта %s", filepath)
                os.remove(filepath)
        except ValueError:
            # Не тот файл пропускаем
            pass


class StateSaver(object):
    BASE_STATE_PATH = os.path.join(settings.LOG_PATH, 'run_states')

    def __init__(self, filename):
        self.path = self.get_state_path(filename)
        self.key = get_run_state_key(filename)

        dirpath = os.path.dirname(self.path)
        if not os.path.exists(dirpath):
            os.makedirs(dirpath)

    def set_state(self, state):
        file_wrapper = get_wrapper_creator(FileType.STATE_SAVER, key=self.key).get_file_wrapper(self.path)
        with file_wrapper.open('w') as f:
            f.write(state.encode('utf8'))
        file_wrapper.upload()

    def get_state(self):
        file_wrapper = get_wrapper_creator(FileType.STATE_SAVER, key=self.key).get_file_wrapper(self.path)
        if file_wrapper.is_file_data_exist():
            with file_wrapper.open() as f:
                data = f.read().strip().decode('utf8')
            return data
        else:
            return None

    def clean_state(self):
        file_wrapper = get_wrapper_creator(FileType.STATE_SAVER, key=self.key).get_file_wrapper(self.path)
        file_wrapper.delete()

    @staticmethod
    def get_state_path(filename):
        return os.path.join(StateSaver.BASE_STATE_PATH, filename)


def run_action_list(action_list, state_saver, log, on_error=None, is_continue=False):
    runner = ActionGroupListRunner(action_list, state_saver, log, on_error, is_continue)
    runner.run()
    return runner.failed_non_critical_actions


class ActionListCriticalError(SimpleUnicodeException):
    pass


class ActionListRunError(SimpleUnicodeException):
    pass


class ActionGroupListRunner(object):
    def __init__(self, action_groups, state_saver, log, on_error_callback=None, is_continue=False):
        self.action_groups = action_groups

        self.log = log
        self.on_error_callback = on_error_callback
        self.is_continue = is_continue

        self.state_saver = state_saver
        self.failed_non_critical_actions = []
        self.bad_action = None
        self.found_continuation = False
        self.last_state = None

        if self.is_continue:
            self.last_state = self.state_saver.get_state()
            if not self.last_state:
                self.log.warning(u"Последний статус пустой, запускаем скрипт заново")
            else:
                self.log.info(u"Пробуем продолжить скрипт с %s", self.last_state)
        else:
            self.state_saver.clean_state()

    def run(self):
        old_dir = os.path.normpath(os.getcwdu())

        try:
            return self.try_run()
        finally:
            os.chdir(old_dir)

    def try_run(self):
        for order, action in enumerate(self.action_iterator()):

            self.current_state = get_actionlist_state(action, order)

            if self.last_state and self.current_state != self.last_state:
                self.log.info(u"Пропускаем %s", self.current_state)
                continue

            elif self.last_state and self.current_state == self.last_state:
                self.log.info(u"Продолжаем скрипт с %s", self.current_state)
                self.last_state = None
                self.found_continuation = True

            self.state_saver.set_state(self.current_state)

            try:
                self.process_action(action)
            except ActionListCriticalError:
                self.log.critical(u"Критическая ошибка при выполнении списка задач, прерываем запуск!!!")
                break

        self.process_results()

    def process_results(self):
        if self.is_continue and not self.found_continuation:
            error_msg = u"Не смогли продолжить скрипт с %s" % self.last_state
            self.log.error(error_msg)

            self.on_error(error_msg)
            return

        if self.bad_action:
            self.log.error(u"Выполнение оборвалось на %s", self.bad_action)

            self.on_error(self.bad_action)

        else:
            self.state_saver.clean_state()

    def process_action(self, action):
        if isinstance(action, six.string_types):
            self.log.info(self.current_state)
        else:
            self.log.info(u'Запускаем %s' % self.current_state)

            critical, real_action = action
            try:
                self.run_action(real_action)
                self.log.info(u"%s отработал успешно", self.current_state)
            except ActionListRunError:
                if critical:
                    self.bad_action = self.current_state
                    raise ActionListCriticalError(self.current_state)
                else:
                    self.failed_non_critical_actions.append(self.current_state)
                    self.log.error(u"%s failed", self.current_state)

    def _get_entry_point_from_action(self, action):
        entry_point, entry_point_index = None, None
        for i, act_part in enumerate(action):
            if act_part.endswith('.py'):
                if not entry_point:
                    entry_point = act_part[:-3].replace('/', '.')
                    entry_point_index = i
                else:
                    raise Exception("Can't find unique entrypoint script in: {})".format(action))

        if not entry_point:
            raise Exception("Can't find entrypoint script in: ".format(action))

        return entry_point, entry_point_index

    def run_action(self, real_action):
        if isinstance(real_action, RunScript):
            try:
                real_action.run()
            except Exception:
                self.log.exception(u"Ошибка при выполнении RunSCript.run %s", real_action.entry_point_short)
                raise ActionListRunError(self.current_state)
        elif callable(real_action):
            try:
                real_action()
            except Exception:
                self.log.exception(u"Ошибка при выполнении функции %s", real_action.__name__)
                raise ActionListRunError(self.current_state)
        else:
            process_env = os.environ.copy()
            if real_action[0] == 'python':  # admin script call in Arcadia
                call_action = real_action[1:]  # remove 'python'

                entry_point, entry_point_index = self._get_entry_point_from_action(call_action)
                process_env['Y_PYTHON_ENTRY_POINT'] = 'travel.rasp.admin.scripts.{}'.format(entry_point)
                call_action.pop(entry_point_index)  # remove entry script

                try:
                    warnings_index = call_action.index('-W')
                except ValueError:
                    warnings_index = None
                if warnings_index is not None:
                    process_env['PYTHONWARNINGS'] = call_action[warnings_index + 1]
                    del call_action[warnings_index:warnings_index + 2]  # remove '-W <value>'

                call_action.insert(0, sys.executable)

            else:  # another external calls
                call_action = real_action

            ret_code = call(call_action, env=process_env)
            if ret_code != 0:
                self.log.error(u"Скрипт упал!!! С кодом %s" % ret_code)
                raise ActionListRunError(self.current_state)

    def action_iterator(self):
        for action_group in self.action_groups:
            if isinstance(action_group, tuple):
                params, actions = action_group
                self.process_params(params)
                for action in actions:
                    yield action
            else:
                ValueError("Unexpected action type %s" % type(action_group))

    def on_error(self, error_msg):
        if self.on_error_callback:
            return self.on_error_callback(error_msg)

    def process_params(self, params):
        self.params = params
        if 'base_path' in params:
            self.log.info(u"Устанавливаем текущую директорию в %s", params['base_path'])
            os.chdir(params['base_path'])


class RunScript(object):
    def __init__(self, entry_point_short, args=None, flock_file=None, entry_point_prefix='travel.rasp.admin.scripts', env=None):
        self.entry_point_short = entry_point_short
        self.entry_point_prefix = entry_point_prefix
        if self.entry_point_prefix:
            self.entry_point = '{}.{}'.format(self.entry_point_prefix, self.entry_point_short)
        else:
            self.entry_point = self.entry_point_short

        self.args = args if args is not None else []
        self.flock_file = flock_file
        self.env = env if env is not None else {'PYTHONWARNINGS': 'ignore'}

    def run(self):
        process_env = os.environ.copy()
        process_env['Y_PYTHON_ENTRY_POINT'] = self.entry_point
        process_env.update(self.env)

        cmd = [sys.executable] + self.args
        if self.flock_file:
            with acquire_exclusive_lock(self.flock_file):
                return call(cmd, env=process_env)
        else:
            return call(cmd, env=process_env)

    def __unicode__(self):
        return self.entry_point

    def __str__(self):
        return self.__unicode__().encode('utf8')


def get_actionlist_state(action, order):
    status_line = u"%d " % order
    if isinstance(action, basestring):
        return status_line + u"notification " + action

    critical, real_action = action
    if critical:
        status_line += u"critical "

    if callable(real_action):
        return status_line + u"function %s" % real_action.__name__
    elif isinstance(real_action, (list, tuple)):
        return status_line + u"script " + u" ".join(real_action)
    else:
        return status_line + u"script " + str(real_action)
