# coding: utf-8

from __future__ import unicode_literals

import logging
from datetime import date, timedelta

from django.db import transaction

from common.models.schedule import RThread, Route, TrainSchedulePlan
from common.models_utils import model_iter
from common.utils.date import RunMask
from travel.rasp.admin.scripts.support_methods import set_going
from travel.rasp.admin.www.utils.mysql import fast_delete_routes, fast_delete_threads


log = logging.getLogger(__name__)


@transaction.atomic
def remove_hide_obsolete_threads():
    """
    Удаляем нитки от Фетисова, которые не попадают в диапазон -3, +200.
    Скрываем аналогичные нитки, которые помечены как не удалять при импорте.
    Удаляем пустые маршруты от Фетисова.
    """

    def check_and_delete_af(check_mask, thread_qs):
        """
        Восстанавливаем дни хождения базовой нитке, и если восстановленные
        дни хождения не попадают в диапазон, удаляем всю группу ниток
        @param check_mask: Маска для проверки диапазона
        @param thread_qs: Запрос из базовых ниток
        """

        threads_to_delete_ids = []

        for thread in model_iter(thread_qs):
            mask = thread.get_original_mask()
            # Нет пересечения с контрольной маской удаляем нитку
            if not mask & check_mask:
                changes = list(thread.thread_changes.all())
                for t in [thread] + changes:
                    log.info('Удаляем нитку %s %s', t.uid, t.type.code)
                threads_to_delete_ids.append(thread.id)
                threads_to_delete_ids.extend(c.id for c in changes)

        fast_delete_threads(RThread.objects.filter(pk__in=threads_to_delete_ids))

    def check_and_hide(check_mask, route_qs):
        for route in model_iter(route_qs):
            # Формируем общую маску для маршрута
            mask = RunMask()
            for thread in route.rthread_set.all():
                mask |= RunMask(thread.year_days)

            # Нет пересечения с контрольной маской скрываем маршрут
            if not mask & check_mask:
                route.hidden = True
                log.info('Скрыли маршрут %s', route.route_uid)
                route.save()

    today = date.today()
    # -3, +200
    check_mask = RunMask.range(today - timedelta(3), today + timedelta(200),
                               today)

    af_without_schedule = (RThread.objects
                           .filter(schedule_plan__isnull=True, supplier__code='af',
                                   t_type__code='suburban', type__code__in=('basic', 'assignment'))
                           .select_related('type'))

    check_and_delete_af(check_mask, af_without_schedule)

    script_protected_routes = Route.objects.filter(script_protected=True,
                                                   hidden=False,
                                                   rthread__schedule_plan=None).distinct()
    # Проверяем и скрываем маршруты, которые не ходят
    check_and_hide(check_mask, script_protected_routes)

    # Теперь тоже самое но ограничиваясь верхней планкой расписания
    for schedule in TrainSchedulePlan.objects.all():
        if schedule.end_date > today:
            check_mask = RunMask.range(today - timedelta(3),
                                       schedule.end_date + timedelta(1),
                                       today)
            af_threads = (RThread.objects
                          .filter(schedule_plan=schedule, supplier__code='af', t_type__code='suburban',
                                  type__code__in=('basic', 'assignment'))
                          .select_related('type'))
            check_and_delete_af(check_mask, af_threads)

            script_protected_routes = Route.objects.filter(script_protected=True,
                                                           hidden=False,
                                                           rthread__schedule_plan=schedule).distinct()
            check_and_hide(check_mask, script_protected_routes)

    # Удаляем маршруты от Фетисова без ниток
    fast_delete_routes(Route.objects.filter(supplier__code='af', rthread__isnull=True))


@transaction.atomic
def restore_basic_thread_from_changes():
    u""" Восстанавливаем основную нитку из ниток изменений, если изменения были
    более чем 3 дня назад"""
    today = date.today()
    diff = timedelta(3)
    thread_with_changes = set()
    for t in RThread.objects.filter(basic_thread__isnull=False):
        thread_with_changes.add(t.basic_thread)

    threads_to_delete_ids = []

    # Переносим дни хождения из ниток изменений в основную нитку
    # Если в нитке изменений больше не осталось дней хождений то удаляем ее
    for thread in thread_with_changes:
        for t_change in thread.thread_changes.all():
            for day in t_change.get_run_date_list():
                if today - day >= diff:
                    t_change.year_days = set_going(t_change.year_days,
                                                   day.day, day.month, '0')
                    thread.year_days = set_going(thread.year_days,
                                                 day.day, day.month, '1')
            if t_change.year_days == 372 * '0':
                threads_to_delete_ids.append(t_change.id)
            else:
                t_change.changed = True
                t_change.save()
        thread.changed = True
        thread.save()

    fast_delete_threads(RThread.objects.filter(pk__in=threads_to_delete_ids))
