# coding: utf-8

import travel.rasp.admin.scripts.load_project  # noqa

import argparse
import os
import re
from datetime import datetime, timedelta

import pytz

from travel.rasp.library.python.common23.date import environment
from travel.rasp.admin.lib.logs import get_script_log_context, ylog_context


def run(args):
    filename_re, use_os_datetime = parse_mask(args.mask)

    if args.verbose:
        print 'Pattern %s' % filename_re.pattern

    now = environment.now_aware()
    remove_before_dt = now - timedelta(31 * args.month_ago)

    for filepath, filename in gen_path_and_names(args):
        if need_to_remove(filename, filepath, filename_re, remove_before_dt, use_os_datetime):
            if args.verbose:
                print 'Removing %s' % filepath
            os.remove(filepath)

    if args.verbose:
        print 'Done'


def parse_mask(mask):
    datetime_parse_keys = {
        '%Y': '(?P<year>\d{4})',
        '%m': '(?P<month>\d{2})',
        '%d': '(?P<day>\d{2})',
        '%H': '(?P<hour>\d{2})',
        '%M': '(?P<minute>\d{2})',
        '%S': '(?P<second>\d{2})'
    }

    use_os_datetime = True

    for key in datetime_parse_keys:
        if key in mask:
            use_os_datetime = False

            mask = mask.replace(key, datetime_parse_keys[key])

    if not mask.endswith('$'):
        mask += '$'

    return re.compile(mask, re.U), use_os_datetime


def gen_path_and_names(args):
    for path in args.paths:
        if not (os.path.exists(path) and os.path.isdir(path)):
            continue

        for root, dirs, files in os.walk(path):
            if not args.recursive:
                dirs[:] = []

            for filename in files:
                filepath = os.path.join(root, filename)
                yield filepath, filename


def need_to_remove(filename, filepath, filename_re, remove_before_dt, use_os_datetime):
    match = filename_re.match(filename)
    if not match:
        return False

    if use_os_datetime:
        file_dt = pytz.UTC.localize(datetime.utcfromtimestamp(os.stat(filepath).st_mtime))
    else:
        file_dt = pytz.timezone('Europe/Moscow').localize(datetime(
            _get_int_group_or_none(match, 'year'),
            _get_int_group_or_none(match, 'month'),
            _get_int_group_or_none(match, 'day'),
            _get_int_group_or_none(match, 'hour'),
            _get_int_group_or_none(match, 'minute'),
            _get_int_group_or_none(match, 'second')
        ))

    return file_dt < remove_before_dt


def _get_int_group_or_none(match, group):
    value = match.group(group)
    if value:
        return int(value)
    else:
        return None


if __name__ == '__main__':
    with ylog_context(**get_script_log_context()):
        parser = argparse.ArgumentParser(description=r'Delete old files using datetime re mask e.g. %Y-%m-%d.*\./log')
        parser.add_argument('mask', metavar='FILENAME_RE', type=str,
                            help='file_mask')
        parser.add_argument('paths', metavar='PATHS', type=str,
                            help='directories to scan', nargs="+")

        parser.add_argument('-m', '--month-ago', type=float, default=3)

        parser.add_argument('-r', '--recursive', action='store_true')
        parser.add_argument('-v', '--verbose', action='store_true')

        args = parser.parse_args()

        run(args)
