# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import os

from django.conf import settings

from common.data_api.file_wrapper.config import FileWrapperConfigs
from common.data_api.file_wrapper.wrappers import LocalFileWrapper, MdsFileWrapper, MdsDirectoryWrapper, LocalDirectoryWrapper
from common.db.mds.clients import mds_s3_public_client, mds_s3_common_client
from common.settings.configuration import Configuration
from travel.rasp.admin.scripts.utils.file_wrapper.mds_utils import get_rasp_export_key, get_rasp_media_export_key


log = logging.getLogger(__name__)


class FileType(object):
    # directories
    MEDIA_DATA_EXPORT = 'MEDIA_DATA_EXPORT'
    MEDIA_DATA_EXPORT_WORK = 'MEDIA_DATA_EXPORT_WORK'
    MEDIA_RASP_ROOT = 'MEDIA_RASP_ROOT'

    # temporary import files
    SCHEDULE_TEMPORARY = 'SCHEDULE_TEMPORARY'
    SCHEDULE_UPLOAD = 'SCHEDULE_UPLOAD'

    # files
    BUS_STATION_CODES = 'BUS_STATION_CODES'
    BUS_STATION_CODES_WORK = 'BUS_STATION_CODES_WORK'
    T_TYPES_BY_GEOID = 'T_TYPES_BY_GEOID'
    T_TYPES_BY_GEOID_WORK = 'T_TYPES_BY_GEOID_WORK'
    STATION_XML_GZ = 'STATION_XML_GZ'
    STATION_XML_GZ_WORK = 'STATION_XML_GZ_WORK'

    # state_saver
    STATE_SAVER = 'STATE_SAVER'

    # logs
    ADMIN_RUNS = 'ADMIN_RUNS'
    CRON_RUNS = 'CRON_RUNS'


FILE_PATH = {
    FileType.STATION_XML_GZ: os.path.join(settings.EXPORT_PUBLIC_PATH, 'stations.xml.gz'),
    FileType.T_TYPES_BY_GEOID: os.path.join(settings.EXPORT_PATH, 't_types_by_geoid.json'),
    FileType.BUS_STATION_CODES: os.path.join(settings.EXPORT_PATH, 'bus_station_codes.json')
}

FILE_DIR_PATH = {
    FileType.MEDIA_DATA_EXPORT: os.path.join(settings.MEDIA_ROOT, 'data/export'),
    FileType.MEDIA_RASP_ROOT: os.path.join(settings.MEDIA_ROOT, 'root'),
}


FileWrapperConfigs.register_config(
    key=FileType.MEDIA_DATA_EXPORT,
    config={
        Configuration.DEVELOPMENT: {
            'class': LocalDirectoryWrapper,
        },
        'default': {
            'class': MdsDirectoryWrapper,
            'kwargs': {
                'mds_client': mds_s3_common_client,
                'download_params': {
                    'prefix': get_rasp_media_export_key(),
                    'only_new': False,
                    'remove_base_path': get_rasp_media_export_key()
                },
                'upload_params': {
                    'mds_base_path': get_rasp_media_export_key()
                }
            }
        }
    }
)

FileWrapperConfigs.register_config(
    key=FileType.MEDIA_DATA_EXPORT_WORK,
    config={
        Configuration.DEVELOPMENT: {
            'class': LocalDirectoryWrapper,
        },
        'default': {
            'class': MdsDirectoryWrapper,
            'kwargs': {
                'mds_client': mds_s3_common_client,
                'upload_params': {
                    'mds_base_path': get_rasp_media_export_key(force_work=True)
                }
            }
        }
    }
)


def schedule_temporary_preparer(kwargs):
    relative_path = kwargs.pop('relative_path', None)
    if relative_path:
        kwargs.update({
            'upload_params': {'mds_base_path': relative_path},
            'delete_params': {'prefix': relative_path}
        })


FileWrapperConfigs.register_config(
    key=FileType.SCHEDULE_TEMPORARY,
    config={
        Configuration.DEVELOPMENT: {
            'class': LocalDirectoryWrapper,
        },
        'default': {
            'class': MdsDirectoryWrapper,
            'kwargs': {
                'mds_client': mds_s3_common_client
            },
            'kwargs_preparer': schedule_temporary_preparer
        }
    }
)


FileWrapperConfigs.register_config(
    key=FileType.SCHEDULE_UPLOAD,
    config={
        Configuration.DEVELOPMENT: {
            'class': LocalFileWrapper,
        },
        'default': {
            'class': MdsFileWrapper,
            'kwargs': {
                'mds_client': mds_s3_common_client,
                'bucket': mds_s3_common_client.bucket
            }
        }
    }
)


FileWrapperConfigs.register_config(
    key=FileType.ADMIN_RUNS,
    config={
        Configuration.DEVELOPMENT: {
            'class': LocalFileWrapper,
        },
        'default': {
            'class': MdsFileWrapper,
            'kwargs': {
                'mds_client': mds_s3_common_client,
                'bucket': mds_s3_common_client.bucket
            }
        }
    }
)


FileWrapperConfigs.register_config(
    key=FileType.CRON_RUNS,
    config={
        Configuration.DEVELOPMENT: {
            'class': LocalFileWrapper,
        },
        'default': {
            'class': MdsFileWrapper,
            'kwargs': {
                'mds_client': mds_s3_common_client,
                'bucket': mds_s3_common_client.bucket
            }
        }
    }
)


FileWrapperConfigs.register_config(
    key=FileType.STATE_SAVER,
    config={
        Configuration.DEVELOPMENT: {
            'class': LocalFileWrapper,
        },
        'default': {
            'class': MdsFileWrapper,
            'kwargs': {
                'mds_client': mds_s3_common_client,
                'bucket': mds_s3_common_client.bucket
            }
        }
    }
)


def get_mds_root_kwargs(root):
    return {
        Configuration.DEVELOPMENT: {
            'class': LocalDirectoryWrapper,
        },
        'default': {
            'class': MdsDirectoryWrapper,
            'kwargs': {
                'mds_client': mds_s3_public_client,
                'upload_params': {
                    'mds_base_path': root,
                    'ignore_files': ['readme', 'readme.md'],
                    'ignore_case': True
                }
            }
        }
    }


FileWrapperConfigs.register_config(
    key=FileType.MEDIA_RASP_ROOT,
    config=get_mds_root_kwargs('rasp-root')
)


def get_mds_config(key):
    return {
        'class': MdsFileWrapper,
        'kwargs': {
            'mds_client': mds_s3_common_client,
            'bucket': mds_s3_common_client.bucket,
            'key': key
        }
    }


FileWrapperConfigs.register_config(
    key=FileType.T_TYPES_BY_GEOID,
    config={
        Configuration.DEVELOPMENT: {
            'class': LocalFileWrapper,
        },
        'default': get_mds_config(get_rasp_export_key(FILE_PATH[FileType.T_TYPES_BY_GEOID]))
    }
)

FileWrapperConfigs.register_config(
    key=FileType.T_TYPES_BY_GEOID_WORK,
    config={
        Configuration.DEVELOPMENT: {
            'class': LocalFileWrapper,
        },
        'default': get_mds_config(get_rasp_export_key(FILE_PATH[FileType.T_TYPES_BY_GEOID], force_work=True))
    }
)

FileWrapperConfigs.register_config(
    key=FileType.BUS_STATION_CODES,
    config={
        Configuration.DEVELOPMENT: {
            'class': LocalFileWrapper,
        },
        'default': get_mds_config(get_rasp_export_key(FILE_PATH[FileType.BUS_STATION_CODES]))
    }
)

FileWrapperConfigs.register_config(
    key=FileType.BUS_STATION_CODES_WORK,
    config={
        Configuration.DEVELOPMENT: {
            'class': LocalFileWrapper,
        },
        'default': get_mds_config(get_rasp_export_key(FILE_PATH[FileType.BUS_STATION_CODES], force_work=True))
    }
)

FileWrapperConfigs.register_config(
    key=FileType.STATION_XML_GZ,
    config={
        Configuration.DEVELOPMENT: {
            'class': LocalFileWrapper,
        },
        'default': get_mds_config(get_rasp_export_key(FILE_PATH[FileType.STATION_XML_GZ]))
    }
)

FileWrapperConfigs.register_config(
    key=FileType.STATION_XML_GZ_WORK,
    config={
        Configuration.DEVELOPMENT: {
            'class': LocalFileWrapper,
        },
        'default': get_mds_config(get_rasp_export_key(FILE_PATH[FileType.STATION_XML_GZ], force_work=True))
    }
)
