# coding: utf-8

from __future__ import unicode_literals

import os.path
import shutil
from datetime import datetime
from collections import OrderedDict

from django.conf import settings

from travel.rasp.library.python.common23.date import environment


PATH_DATE_FORMAT = '%Y-%m-%d'
PATH_DATETIME_FORMAT = '%Y-%m-%d_%H%M%S'


def parse_date(dirname):
    try:
        return datetime.strptime(dirname, PATH_DATETIME_FORMAT)
    except ValueError:
        pass

    try:
        return datetime.strptime(dirname, PATH_DATE_FORMAT)
    except ValueError:
        pass


def get_schedule_temporary_path(supplier_or_package):
    """
    :type supplier_or_package: common.models.schedule.Supplier | importinfo.models.two_stage_import.TwoStageImportPackage
    :rtype: str
    """  # noqa: E501
    subpath = get_subpath_by_supplier_or_package(supplier_or_package)

    dir_path = os.path.join(settings.DATA_PATH, 'schedule', 'temporary', subpath)

    if not os.path.exists(dir_path):
        os.makedirs(dir_path)

    return dir_path


def get_schedule_temporary_date_path(supplier_or_package, today=None):
    """
    :type supplier_or_package: common.models.schedule.Supplier | importinfo.models.two_stage_import.TwoStageImportPackage
    :param today: datetime.date
    :rtype: str
    """  # noqa: E501
    today = today or environment.today()

    dir_path = os.path.join(
        get_schedule_temporary_path(supplier_or_package),
        today.strftime(PATH_DATE_FORMAT)
    )

    if not os.path.exists(dir_path):
        os.makedirs(dir_path)

    return dir_path


def get_schedule_temporary_date_filepath(filename, supplier_or_package, today=None):
    """
    :type filename: str
    :type supplier_or_package: common.models.schedule.Supplier | importinfo.models.two_stage_import.TwoStageImportPackage
    :param today: datetime.date
    :rtype: str
    """  # noqa: E501
    dir_path = get_schedule_temporary_date_path(supplier_or_package, today)

    return os.path.join(dir_path, filename)


def get_schedule_temporary_today_dir(supplier_or_package, today=None):
    subpath = get_subpath_by_supplier_or_package(supplier_or_package)
    today = today or environment.today()
    dir_path = os.path.join(settings.SCHEDULE_TEMPORARY_PATH, subpath, today.strftime(PATH_DATE_FORMAT))

    return dir_path


def remove_schedule_temporary_today_dir(supplier_or_package, today=None):
    dir_path = get_schedule_temporary_today_dir(supplier_or_package, today)

    if not os.path.exists(dir_path):
        return

    error_files = []

    def on_error(_f, path, exc_info):
        error_files.append(path)

    shutil.rmtree(dir_path, onerror=on_error)

    return error_files


def remove_cysix_xml(supplier_or_package, today=None):
    dir_path = get_schedule_temporary_today_dir(supplier_or_package, today)
    filepath = os.path.join(dir_path, 'cysix.xml')

    if os.path.exists(filepath):
        os.remove(filepath)


def get_subpath_by_supplier_or_package(supplier_or_package):
    if hasattr(supplier_or_package, 'code'):
        return supplier_or_package.code

    else:
        return '%s/%s' % (supplier_or_package.supplier.code, supplier_or_package.id)


def get_schedule_filepath(filename, supplier):
    dir_path = os.path.join(settings.DATA_PATH, 'schedule', supplier.code)
    dir_path = str(dir_path)
    if not os.path.exists(dir_path):
        os.makedirs(dir_path)
    return os.path.join(dir_path, filename)


def get_schedule_temporary_datetime_filepath(filename, supplier, now=None):
    now = now or environment.now()

    dir_path = os.path.join(settings.DATA_PATH, 'schedule', 'temporary', supplier.code,
                            now.strftime(PATH_DATETIME_FORMAT))
    dir_path = str(dir_path)
    if not os.path.exists(dir_path):
        os.makedirs(dir_path)
    return os.path.join(dir_path, filename)


def get_tablo_temporary_datetime_filepath(filename, supplier, now=None):
    now = now or environment.now()

    dir_path = os.path.join(settings.DATA_PATH, 'tablo', 'temporary', supplier.code,
                            now.strftime(PATH_DATETIME_FORMAT))
    dir_path = str(dir_path)
    if not os.path.exists(dir_path):
        os.makedirs(dir_path)
    return os.path.join(dir_path, filename)


def get_tablo_temporary_date_filepath(filename, supplier, now=None):
    now = now or environment.now()

    dir_path = os.path.join(settings.DATA_PATH, 'tablo', 'temporary', supplier.code,
                            now.strftime(PATH_DATE_FORMAT))
    dir_path = str(dir_path)
    if not os.path.exists(dir_path):
        os.makedirs(dir_path)
    return os.path.join(dir_path, filename)


def get_last_temporary_date_and_file_or_none(filename, supplier, get_empty_files_too=True):
    for date, filepath in get_all_temporary_date_files(filename, supplier, get_empty_files_too).items():
        return date, filepath
    return None, None


def get_last_temporary_date_file(filename, supplier, get_empty_files_too=True):
    return get_last_temporary_date_and_file_or_none(filename, supplier, get_empty_files_too=True)[1]


def get_all_temporary_date_files(search_filename, supplier, get_empty_files_too=True):
    """
    Возвращает OrderedDict(дата: файл) в начале файл последней версии
    :type search_filename: str
    :type supplier: common.models.schedule.Supplier
    :type get_empty_files_too: bool
    """
    supplier_dir = os.path.join(settings.DATA_PATH, 'schedule', 'temporary', supplier.code)
    if os.path.exists(supplier_dir):
        dates = os.listdir(supplier_dir)
    else:
        dates = []

    by_dates = OrderedDict()
    for date_dir in dates:
        if os.path.isdir(os.path.join(supplier_dir, date_dir)):
            try:
                date_ = datetime.strptime(date_dir, PATH_DATE_FORMAT).date()
                by_dates[date_] = None
            except ValueError:
                pass

    for date_ in list(by_dates.keys()):
        filepath = os.path.join(supplier_dir, date_.strftime(PATH_DATE_FORMAT), search_filename)
        if os.path.exists(filepath) and os.path.isfile(filepath):
            if get_empty_files_too or os.path.getsize(filepath):
                by_dates[date_] = filepath

    by_dates = OrderedDict((k, by_dates[k]) for k in sorted(by_dates.keys(), reverse=True)
                           if by_dates[k] is not None)
    return by_dates


def get_stored_supplier_codes(type_):
    type_path = get_type_path(type_)

    if not os.path.isdir(type_path):
        return []

    return [
        d
        for d in os.listdir(type_path)
        if os.path.isdir(os.path.join(type_path, d))
    ]


def get_type_path(type_):
    return os.path.join(settings.DATA_PATH, type_, 'temporary')


def get_all_temporary_dirs_by_dt(type_, supplier):
    supplier_code = supplier if isinstance(supplier, basestring) else supplier.code

    supplier_dir = os.path.join(get_type_path(type_), supplier_code)

    dates = os.listdir(supplier_dir)
    result = OrderedDict()

    def get_dt(dir_name):
        try:
            return datetime.strptime(dir_name, PATH_DATE_FORMAT)
        except ValueError:
            pass

        try:
            return datetime.strptime(dir_name, PATH_DATETIME_FORMAT)
        except ValueError:
            pass

    for dt_dir_name in dates:
        dt_dir_path = os.path.join(supplier_dir, dt_dir_name)
        if os.path.isdir(dt_dir_path):
            dt = get_dt(dt_dir_name)

            if dt is not None:
                result[dt] = dt_dir_path

    result = OrderedDict(
        (k, result[k])
        for k in sorted(result.keys(), reverse=True)
    )

    return result


def mark_error(filepath, log):
    dirpath, filename = os.path.split(filepath)
    new_filename = 'error_' + filename
    new_path = os.path.join(dirpath, new_filename)
    os.rename(filepath, new_path)
    log.error(u"Пометили файл %s как ошибочный. %s", filepath, new_path)
