# coding: utf-8

"""
Скрипт:
1. для российских станций и городов заполняет пустой title_ru из title
2. проставляет станциям и городам склонения
"""

import travel.rasp.admin.scripts.load_project  # noqa

import json
import requests
from traceback import format_exc

from django.conf import settings
from django.db.models import Q
from django.utils.encoding import smart_unicode

from common.models.geo import Settlement, Station, Country
from common.settings.configuration import Configuration
from common.settings.utils import define_setting
from travel.rasp.admin.lib.logs import get_script_log_context, ylog_context
from travel.rasp.admin.lib.mail import mail_datachanges
from travel.rasp.admin.lib.exceptions import SimpleUnicodeException


define_setting(
    'INFLECTOR_URL',
    default='http://hamzard.yandex.net:8891/wizard',
    env={
        Configuration.PRODUCTION: 'http://reqwizard.yandex.net:8891/wizard',
        Configuration.TESTING: 'http://hamzard.yandex.net:8891/wizard'
    }
)
INFLECTOR_TIMEOUT = 30
MAX_RETRY_COUNT = 3

GENITIVE = u'род'  # Родительный (Откуда? Из)
ACCUSATIVE = u'вин'  # Винительный (Куда? В)
LOCATIVE = u'пр'  # Предложный (Где? В)


class InflectorError(SimpleUnicodeException):
    pass


def get_inflections(text):
    data = get_inflector_response(text)
    try:
        data = json.loads(data)
        return {inflection['Grammar']: inflection['Text'] for inflection in data['Form']}
    except Exception:
        raise InflectorError(u'Получили странные данные от склонятора %r: %s', data, smart_unicode(format_exc()))


def get_inflector_response(text):
    params = {
        'action': 'inflector',
        'text': text,
    }
    error = u''
    for i in xrange(MAX_RETRY_COUNT):
        try:
            response = requests.get(settings.INFLECTOR_URL, params=params, timeout=INFLECTOR_TIMEOUT)
            if response.status_code == requests.codes.ok:
                return response.text
        except requests.RequestException:
            error = smart_unicode(format_exc())

    raise InflectorError(u'Ошибка получения данных от склонятора. %s', error)


def fill_inflections(obj, errors, inflected):
    model = obj.__class__
    try:
        inflections = get_inflections(obj.title_ru)
        changed = {}
        if not inflections:
            errors.append(
                (u'Не нашли склонений для %s %s %s:\n' % (type(obj).__name__, obj.id, obj.title))
            )
            return

        if not obj.title_ru_accusative and ACCUSATIVE in inflections:
            title_ru_accusative = inflections[ACCUSATIVE]
            changed[u'Куда? В '] = title_ru_accusative
            obj.title_ru_accusative = title_ru_accusative

        if not obj.title_ru_genitive and GENITIVE in inflections:
            title_ru_genitive = inflections[GENITIVE]
            changed[u'Откуда? Из '] = title_ru_genitive
            obj.title_ru_genitive = title_ru_genitive

        if not obj.title_ru_locative and LOCATIVE in inflections:
            title_ru_locative = inflections[LOCATIVE]
            changed[u'Где? В '] = title_ru_locative
            obj.title_ru_locative = title_ru_locative

        info = model.__name__ + u' %d %s: ' % (obj.id, obj.title)
        inflected.append(info + u', '.join([question + answer for question, answer in changed.items()]))
        obj.save()

    except InflectorError, e:
        errors.append((u"Склоняли %s %s %s:\n" % (type(obj).__name__, obj.id, obj.title)) + e.msg)


def inflect():
    models = [Settlement, Station]
    errors = []
    results = {}
    for model in models:
        inflected = []
        for obj in model.objects.exclude(title_ru__isnull=True).exclude(title_ru=''):
            if not (obj.title_ru_accusative and obj.title_ru_genitive and obj.title_ru_locative):
                fill_inflections(obj, errors, inflected)

        if inflected:
            results[model.__name__] = inflected

    report = u''
    error = u'\n'.join(errors)
    if error:
        report += u'Случились ошибки при склонении\n' + error + u'\n'
        report += u"\n"
    for model_name, inflected in results.iteritems():
        report += u'Просклоняли %d ' % len(inflected) + model_name + u's\n\n'
        report += u'\n'.join(inflected)
        report += u'\n\n'

    if report:
        mail_datachanges(u'Склонения станций и городов', report)


def fill_title_ru_from_title():
    report = u''
    models = [Settlement, Station]
    for model in models:
        for obj in model.objects.filter(country=Country.RUSSIA_ID).filter(Q(title_ru__isnull=True) | Q(title_ru='')):
            if not obj.title:
                return

            obj.title_ru = obj.title
            obj.save()

            report += u'Для {cls} {id} {title} заполнили title_ru из title\n'.format(
                cls=model.__name__, id=obj.id, title=obj.title)

    if report:
        mail_datachanges(u'Заполнение title_ru из title', report)


if __name__ == '__main__':
    with ylog_context(**get_script_log_context()):
        fill_title_ru_from_title()
        inflect()
