# coding: utf-8
""" Корректирует таблицу:
        www_station
    Проставляет города, регионы и time_zone тем станциям, у которых их нет
    и проставляет time_zone станциям основываясь на информации из таблицы
        www_region (time_zone у которой был проставлен из geobase)
"""

import travel.rasp.admin.scripts.load_project  # noqa

import logging
import time as os_time

from django.db import connection, transaction
from django.db.models import Q, F

from common.models.geo import Station, Country
from common.data_api.baris.instance import baris
from travel.rasp.admin.lib.logs import create_current_file_run_log, get_script_log_context, ylog_context


log = logging.getLogger(__name__)


@transaction.atomic
def run():
    log.info(u"Start www_stations.py")
    start = os_time.time()
    cursor = connection.cursor()

    # update fuzzy
    Station.objects.all().update(fuzzy_only=False)
    cursor.execute("""
    SELECT s1.id FROM www_station s1 JOIN www_rtstation rts ON rts.station_id = s1.id
        GROUP BY s1.id HAVING MIN(rts.is_fuzzy) = 1
    """)
    fuzzy_station_ids = [row[0] for row in cursor.fetchall()]

    Station.objects.filter(pk__in=fuzzy_station_ids).update(fuzzy_only=True)

    # проставим станциям регионы их городов, если не совпадают
    for s in (Station.objects.filter(settlement__region__isnull=False)
                     .exclude(region=F('settlement__region'))
                     .select_related('settlement__region', 'region')):

        old_region = s.region

        s.region = s.settlement.region

        log.info(u"Сменили стации %s %s регион с %s %s на %s %s",
                 s.id, s.title, old_region and old_region.id, old_region and old_region.title,
                 s.region_id, s.region.title)

        s.save()

    # проставим станциям страны их городов, если не заполнены
    for s in (Station.objects.filter(settlement__country__isnull=False)
                     .exclude(country=F('settlement__country'))
                     .select_related('settlement__country', 'country')):
        old_country = s.country

        s.country = s.settlement.country

        log.info(u"Сменили стации %s %s страну с %s %s на %s %s",
                 s.id, s.title,
                 old_country and old_country.id, old_country and old_country.title,
                 s.country_id, s.country.title)

        s.save()

    # Setting time_zone for stations by region
    for s in Station.objects.filter(region__time_zone__isnull=False, time_zone__isnull=True):
        s.time_zone = s.region.time_zone
        s.save()

    # Проставляем станциям временные зоны их городов
    for s in Station.objects.filter(time_zone__isnull=True, settlement__time_zone__isnull=False):
        log.info(u"Станции %s %s поменяли временную зону %s на зону ее города %s",
                 s.id, unicode(s), s.time_zone, s.settlement.time_zone)
        s.time_zone = s.settlement.time_zone
        s.save()
    # Если станция не имеет временной зоны и находится в стране, которая живет по временной зоне столицы
    # и не привязана к городу, проставим ей временную зону столицы
    for s in (Station.objects.exclude(country__in=Country.mast_have_region_countries())
                     .filter(settlement__isnull=True, time_zone__isnull=True, country__isnull=False)
                     .select_related('country')):
        capital = s.country.get_capital()
        if capital:
            log.info(u"Станции %s %s поменяли временную зону %s на зону столицы %s (%s)",
                     s.id, unicode(s), s.time_zone, capital.time_zone, unicode(capital))
            s.time_zone = capital.time_zone
            s.save()

    # Выставление признака "аэроэкспрессы" аэропортам
    Station.objects.filter(rtstation__thread__express_type='aeroexpress').update(has_aeroexpress=True)

    ensure_stations_visibility(cursor)

    for s in Station.objects.filter(hidden=False, properties__isnull=False):
        s.properties.all().delete()

    # Привязка с странам
    for station in Station.objects.filter(
        Q(settlement__country__isnull=False) | Q(region__country__isnull=False), country=None
    ):
        if station.settlement and station.settlement.country:
            station.country = station.settlement.country
            station.save()
        elif station.region and station.region.country:
            station.country = station.region.country
            station.save()

    # Обновляем ненормальные express_id и sirena_id
    cursor.execute("UPDATE www_station SET sirena_id = NULL, express_id = NULL")
    cursor.execute("""
    UPDATE www_station s JOIN www_stationcode sc ON sc.station_id = s.id
    JOIN www_codesystem cs ON cs.id = sc.system_id AND cs.code='sirena'
    SET s.sirena_id = sc.code
    """)
    cursor.execute("""
    UPDATE www_station s JOIN www_stationcode sc ON sc.station_id = s.id
    JOIN www_codesystem cs ON cs.id = sc.system_id AND cs.code='express'
    SET s.express_id = sc.code
    """)

    cursor.execute("""
UPDATE `www_station` s JOIN `www_directionmarker` dm ON dm.station_id = s.id
    JOIN `www_direction` d ON dm.direction_id = d.id
    JOIN `www_suburbanzone` sz ON d.suburban_zone_id = sz.id
SET s.suburban_zone_id = sz.id
WHERE s.suburban_zone_id IS NULL""")

    log.info(u"Время работы %.0f сек", os_time.time() - start)


def ensure_stations_visibility(cursor):
    """Раскрываем только станции, через которые ходят маршруты,
    и останавливаются на них
    """
    current_hidden, current_visible = set(), set()
    for station_id, hidden in Station.objects.all().values_list('id', 'hidden'):
        if hidden:
            current_hidden.add(station_id)
        else:
            current_visible.add(station_id)

    cursor.execute("""
        SELECT s.id FROM www_station s
        JOIN www_rtstation rts ON rts.station_id = s.id
        JOIN www_rthread t ON t.id = rts.thread_id
        JOIN www_route r ON r.id = t.route_id
        WHERE (
            rts.tz_departure IS NULL OR rts.tz_arrival IS NULL
            OR (rts.tz_arrival != rts.tz_departure)
        ) AND r.hidden = FALSE AND rts.is_technical_stop = FALSE;""")
    new_visible = set(row[0] for row in cursor.fetchall())

    # Аэропорты загружаем из БАРиС
    new_departure_airports, new_arrival_airports = baris.get_station_summary()
    new_visible = new_visible | new_departure_airports | new_arrival_airports

    to_hide = current_visible - new_visible
    to_show = current_hidden & new_visible
    Station.objects.filter(pk__in=to_hide).update(hidden=True)
    Station.objects.filter(pk__in=to_show).update(hidden=False)


if __name__ == '__main__':
    with ylog_context(**get_script_log_context()):
        create_current_file_run_log()

        run()
