#!/usr/bin/env python
# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import travel.rasp.admin.scripts.load_project  # noqa

import json
import logging
import time as os_time
from collections import defaultdict
from itertools import groupby, combinations, product

from django.db import connection, transaction
from django.db.models import Q

from common.utils.settlement import (
    fill_best_rts_for_settlement, fetch_station_settlement_ids_with_none, fetch_station_settlement_ids
)
from route_search.models import ZNodeRoute2
from travel.rasp.admin.scripts import suburban_stops
from travel.rasp.admin.lib.maintenance.flags import flags
from travel.rasp.admin.lib import tmpfiles
from travel.rasp.admin.lib.logs import print_log_to_stdout, create_current_file_run_log, get_script_log_context, ylog_context
from travel.rasp.admin.lib.mysqlutils import MysqlFileWriter, MysqlModelLoader, MysqlModelUpdater
from common.models.geo import StationMajority, Station
from common.models.schedule import RThread, RTStation, Route
from common.models.transport import TransportType
from common.models_utils import model_iter
from common.utils.date import RunMask
from common.utils.progress import PercentageStatus


log = logging.getLogger(__name__)


class ZNodeRouteImporter(object):
    urban_type = TransportType.objects.get(code='urban')

    def __init__(self, partial=False, log_prepared=False):
        self.changed = Q()
        self.partial = partial
        self.log_prepared = log_prepared
        self.cursor = connection.cursor()
        self.has_suburban_stops = True

        self.not_in_search = StationMajority.objects.get(code='not_in_search')

        if self.partial:
            self.changed = Q(changed=True, path_and_time_unchanged=False)
            self.partial = True

        self.station2settlement_ids = fetch_station_settlement_ids_with_none()
        self.station2settlement_ids_without_none = fetch_station_settlement_ids()

    @transaction.atomic
    def run(self):
        start = os_time.time()
        log.info(u'Начинаем z_noderoute2')

        self.setup()

        self.clean_znoderoute()

        self.refill_main_table()

        self.fix_empty_settlements()

        self.tear_down()

        log.info(u'Выполнено за %s сек.', os_time.time() - start)

    def setup(self):
        log.info(u'Предварительные действия')

        self.compute_stops()

    def tear_down(self):
        log.info(u'Заключительные действия')

        self.cursor.execute('DROP TABLE IF EXISTS suburban_stops')

    @tmpfiles.clean_temp
    def compute_stops(self):
        threads = (
            RThread.objects
            .filter(self.changed, route__hidden=False)
            .filter(t_type_id=TransportType.SUBURBAN_ID)

            # Не считать остановки для экспрессов от ТИС и СвПК
            .exclude(express_type__isnull=False, supplier__code__in=['tis', 'svrpk'])
            .exclude(year_days=372 * '0')

            .select_related('route', 't_type', 'supplier')
        )

        threads_total = threads.count()
        if not threads_total:
            log.info(u'Электрички не менялись, не считаем остановки')
            self.has_suburban_stops = False
            return

        log.info(u'Считаем остановки пригородных поездов')

        rtstation_query = (
            Q(station__majority__lt=self.not_in_search.id) &
            Q(is_technical_stop=False)
        )

        self.cursor.execute('DROP TABLE IF EXISTS suburban_stops')
        self.cursor.execute('''CREATE TABLE suburban_stops (
                            rtstation_from_id INTEGER NOT NULL DEFAULT 0,
                            rtstation_to_id INTEGER NOT NULL DEFAULT 0,
                            stops TEXT,
                            stops_translations TEXT
                          ) DEFAULT CHARSET=utf8
                       ''')

        status = PercentageStatus(threads_total, log)

        chunksize = 500

        log.info(u"Пересчитываем остановки для %s ниток", threads_total)

        log.info(u'Формируем временный файл')
        suburban_tmp_file = tmpfiles.get_tmp_filepath('suburban_stops.tmp', 'znoderoute2')
        with open(suburban_tmp_file, 'w') as suburban_file:
            writer = MysqlFileWriter(suburban_file, fields=('rtstation_from_id', 'rtstation_to_id',
                                                            'stops', 'stops_translations'))

            for thread in model_iter(threads, chunksize=chunksize):
                rtstations = list(thread.rtstation_set.filter(rtstation_query))

                for index_from, rts_from in enumerate(rtstations):
                    if rts_from.tz_arrival == rts_from.tz_departure:
                        continue

                    for index_to, rts_to in enumerate(rtstations[index_from + 1:]):
                        index_to = index_from + index_to + 1

                        if rts_to.tz_arrival == rts_to.tz_departure:
                            continue

                        stops_translations = suburban_stops.stops_text(
                            rtstations[index_from:index_to + 1]
                        )

                        writer.writedict({
                            'rtstation_from_id': rts_from.id,
                            'rtstation_to_id': rts_to.id,
                            'stops': stops_translations['ru'],
                            'stops_translations': json.dumps(stops_translations, ensure_ascii=False)
                        })

                status.step()

        log.info(u'Заливаем данные во временную таблицу')
        self.cursor.execute('''
            LOAD DATA LOCAL INFILE %s
            INTO TABLE suburban_stops
            CHARACTER SET utf8
            FIELDS ENCLOSED BY '"'
            LINES TERMINATED BY '\n'
            (rtstation_from_id, rtstation_to_id, stops, stops_translations)
        ''', [suburban_tmp_file])

        self.cursor.execute('ALTER TABLE suburban_stops ADD KEY (rtstation_from_id, rtstation_to_id)')
        log.info(u'Остановки электричек подготовлены')

    @tmpfiles.clean_temp
    def refill_main_table(self):
        log.info(u'Заливаем данные в www_znoderoute2')

        rts_query_set = RTStation.objects.filter(thread__route__hidden=False)
        rts_query_set = rts_query_set.filter(is_technical_stop=False)
        rts_query_set = rts_query_set.filter(station__majority__lt=self.not_in_search.id)

        rts_query_set = rts_query_set.exclude(thread__t_type=self.urban_type.id)
        rts_query_set = rts_query_set.exclude(thread__year_days=RunMask.EMPTY_YEAR_DAYS)

        if self.partial:
            rts_query_set = rts_query_set.filter(thread__changed=True)
            rts_query_set = rts_query_set.filter(thread__path_and_time_unchanged=False)

        rts_query_set = rts_query_set.select_related('thread', 'thread__route', 'station') \
                                     .order_by('thread', 'id')

        all_rts_iter = model_iter(rts_query_set, chunksize=10000)

        loader = MysqlModelLoader(ZNodeRoute2, tmpfiles.get_tmp_dir('znoderoute2'))
        status = PercentageStatus(rts_query_set.count(), log)

        with loader:
            for thread, rts_iter in groupby(all_rts_iter, lambda rts: rts.thread):
                rtstations = list(rts_iter)
                best_rts_for_start_from_settlement, best_rts_for_finish_in_settlement = \
                    fill_best_rts_for_settlement(rtstations, self.station2settlement_ids_without_none)
                for rts_from, rts_to in combinations(rtstations, 2):
                    if not rts_from.is_searchable_from:
                        continue

                    if not rts_to.is_searchable_to:
                        continue

                    if rts_from.departure_code_sharing and rts_to.arrival_code_sharing:
                        continue

                    if rts_from.tz_departure == rts_from.tz_arrival:
                        continue

                    if rts_to.tz_departure == rts_to.tz_arrival:
                        continue

                    from_settlement_ids = self.station2settlement_ids[rts_from.station_id]
                    to_settlement_ids = self.station2settlement_ids[rts_to.station_id]

                    for settlement_from_id, settlement_to_id in product(from_settlement_ids, to_settlement_ids):
                        loader.add(ZNodeRoute2(
                            route_id=thread.route_id,
                            thread_id=thread.id,
                            t_type_id=thread.t_type_id,
                            settlement_from_id=settlement_from_id,
                            station_from_id=rts_from.station_id,
                            rtstation_from=rts_from,
                            settlement_to_id=settlement_to_id,
                            station_to_id=rts_to.station_id,
                            rtstation_to=rts_to,
                            stops_translations='',
                            supplier_id=thread.supplier_id,
                            two_stage_package_id=thread.route.two_stage_package_id,
                            good_for_start=best_rts_for_start_from_settlement.get(settlement_from_id, None) == rts_from,
                            good_for_finish=best_rts_for_finish_in_settlement.get(settlement_to_id, None) == rts_to,
                        ))

                status.step(len(rtstations))

        if self.has_suburban_stops:
            log.info(u'Заливаем переводы остановок')
            self.cursor.execute('''
                UPDATE www_znoderoute2 z2
                JOIN suburban_stops AS stops
                    ON (z2.rtstation_from_id = stops.rtstation_from_id AND z2.rtstation_to_id = stops.rtstation_to_id)
                SET z2.stops_translations = stops.stops_translations
            ''')

        log.info(u'Done')

    def fix_empty_settlements(self):
        if self.partial:
            log.info(u"Поправляем таблицы")
            stations_with_settlement = list(
                Station.objects.filter(settlement__isnull=False).values_list('id', 'settlement_id'))
            self.fix_znoderoute('settlement_from_id', 'station_from_id', stations_with_settlement)
            self.fix_znoderoute('settlement_to_id', 'station_to_id', stations_with_settlement)

    @staticmethod
    def fix_znoderoute(settlement_field, station_field, stations_with_settlement):
        znoderoute_to_station = ZNodeRoute2.objects.filter(
            **{'{}__isnull'.format(settlement_field): True}).values_list('id', station_field)

        znoderoutes_by_station = defaultdict(list)
        for znoderoute_id, station_id in znoderoute_to_station:
            znoderoutes_by_station[station_id].append(znoderoute_id)

        updater = MysqlModelUpdater(ZNodeRoute2, tmpfiles.get_tmp_dir('znoderoute2_fix'))
        with updater:
            for station_id, settlement_id in stations_with_settlement:
                znoderoute_ids = znoderoutes_by_station.get(station_id)
                if not znoderoute_ids:
                    continue
                for znoderoute in ZNodeRoute2.objects.filter(pk__in=znoderoute_ids):
                    setattr(znoderoute, settlement_field, settlement_id)
                    updater.add(znoderoute)

    def clean_znoderoute(self):
        log.info(u'Очищаем www_znoderoute2')
        if not self.partial:
            # полный пересчет
            log.info(u'TRUNCATE TABLE')
            self.cursor.execute('TRUNCATE TABLE www_znoderoute2')
        else:
            # частичный пересчет
            log.info(u'Частичный пересчет. Удаляем измененные нитки')
            rthread_ids = list(RThread.objects.filter(
                changed=True, path_and_time_unchanged=False).values_list('id', flat=True))
            if rthread_ids:
                self.cursor.execute('DELETE FROM www_znoderoute2 WHERE thread_id IN %s;', (rthread_ids, ))

            log.info(u'Удаляем скрытые маршруты')
            route_ids = list(Route.objects.filter(hidden=True).values_list('id', flat=True))
            if route_ids:
                self.cursor.execute('DELETE FROM www_znoderoute2 WHERE route_id IN %s;', (route_ids, ))


@transaction.atomic
def import_noderoute():
    ZNodeRouteImporter(options.log_prepared).run()


if __name__ == '__main__':
    with ylog_context(**get_script_log_context()):
        create_current_file_run_log()

        from optparse import OptionParser

        parser = OptionParser()
        parser.add_option("-v", "--verbose",
                          dest="verbose", action="store_true",
                          help="Print log messages to STDOUT")
        parser.add_option("-l", "--log-prepared",
                          action="store_true",
                          help="Log prepared data and query")

        (options, args) = parser.parse_args()

        if options.verbose:
            print_log_to_stdout()

        partial = flags['partial_preparation']

        ZNodeRouteImporter(partial, options.log_prepared).run()
