# coding: utf-8
"""
Создает таблицу z_station_schedule
"""

import travel.rasp.admin.scripts.load_project  # noqa

import gc
import json
import logging
import re
import time as os_time
from datetime import timedelta, datetime

from django.conf import settings
from django.db.models import Q
from django.db import connection, transaction

from common.models.schedule import RThread, RThreadType, TrainSchedulePlan, StationSchedule, ScheduleExclusion
from common.models.transport import TransportType
from common.models_utils import model_iter
from travel.rasp.library.python.common23.date import environment
from common.utils.caching import cache_until_switch, cache_method_result
from common.utils.date import RunMask
from common.utils.progress import PercentageStatus
from common.utils.calendar_matcher import get_matcher, commaseparated_datelist

from travel.rasp.admin.lib.maintenance.flags import flags
from travel.rasp.admin.lib import tmpfiles
from travel.rasp.admin.lib.logs import print_log_to_stdout, create_current_file_run_log, get_script_log_context, ylog_context
from travel.rasp.admin.lib.mysqlutils import MysqlModelUpdater
from travel.rasp.admin.lib.sql import fast_delete


log = logging.getLogger(__name__)


def collect_gc_callback(obj, *args, **kwargs):
    gc.collect()


class ScheduleProcessor(object):
    name = 'Abstract processor'  # название процессора

    extrapolation_limit_with_schedule_plan = False

    # settings
    do_prepare_threads = True
    # длина известного расписания. по этому параметру строятся precalc_days_text для ниток и экстраполяции
    known_schedule_length = 40
    use_schedule_plan = False
    use_thread_changes = False  # учитывать ли нитки изменения при расчете дней хождения
    templates = 'basic'  # набор шаблонов
    extrapolate = False  # экстраполировать ли расписание
    # какое минимальное количество совпадений должно случиться при сопоставлении шаблонов
    template_min_match_days = None

    # filters
    supplier = None  # фильтр по поставщику
    transport_type = None  # фильтр по типу транспорта

    # class specific objects
    matcher = None

    extrapolation_limit_length = 365 - 40
    search_first_day_in_past_days = 0

    def __init__(self, now_aware=None, partial=True, route_id=None):
        self.now_aware = now_aware or environment.now_aware()
        self.partial = partial
        self.route_id = route_id

    def get_thread_filter(self):
        return Q()

    def get_station_schedule_filter(self):
        return Q()

    @transaction.atomic
    def run(self):
        """Обработка расписаний"""

        log.info(u'----- %s start -----' % self.name)

        self.cursor = connection.cursor()

        self._prepare_queryset()

        start = os_time.time()

        if self.do_prepare_threads:
            self.prepare_threads()

        log.info('Cleaning old station_schedules')
        fast_delete(self.station_schedule_qs)

        self.fill_station_schedules()

        self.clean_from_exclusions()

        log.info(u'%s finished in %s sec' % (self.name, os_time.time() - start))

    @tmpfiles.clean_temp
    def prepare_threads(self):
        """Предварительная подготовка ниток"""

        log.info(u"Threads prepare start")

        start = os_time.time()

        log.info(u'Всего нужно обработать %s ниток', self.thread_qs.count())
        progress = PercentageStatus(self.thread_qs.count(), log=log, callback=collect_gc_callback)

        tmp_dir = tmpfiles.get_tmp_dir()
        fields = ['year_days', 'translated_days_texts', 'translated_except_texts']
        with MysqlModelUpdater(RThread, tmp_dir, fields=fields) as thread_updater:
            for thread in model_iter(self.thread_qs, chunksize=100):
                if len(thread.year_days) != RunMask.MASK_LENGTH:
                    log.error(u'Неправильная маска у нитки %s', thread.uid)
                    continue

                self._prepare_matcher(thread)
                if self.extrapolate:
                    is_extrapolated = self.extrapolate_thread(thread)
                    if is_extrapolated:
                        self.post_extrapolate(thread)
                self.precalc_days_texts(thread)

                thread_updater.add(thread)
                progress.step()

        log.info(u"Threads prepare finished in %s sec.", (os_time.time() - start))

    @transaction.atomic
    def fill_station_schedules(self):
        log.info(u'Filling table')

        # SQL raw modifiers
        sql, additional_params = self.thread_qs.query.get_compiler(connection=connection).as_sql()
        conditions_sql = sql[sql.index('WHERE') + 5:sql.find('ORDER BY') > 0 and sql.index('ORDER BY') or None]
        additional_conditions = u" AND %s " % conditions_sql

        sql = """
INSERT INTO www_stationschedule (
    station_id,
    route_id,
    thread_id,
    rtstation_id,
    is_fuzzy
)
SELECT
    www_rtstation.station_id,
    www_route.id,
    www_rthread.id,
    www_rtstation.id,
    www_rtstation.is_fuzzy
FROM
    www_route INNER JOIN
    www_rthread ON www_route.id = www_rthread.route_id INNER JOIN
    www_rtstation ON www_rtstation.thread_id = www_rthread.id INNER JOIN
    www_supplier ON www_rthread.supplier_id = www_supplier.id INNER JOIN
    www_transporttype ON www_rthread.t_type_id = www_transporttype.id LEFT JOIN
    www_rtstation AS rts_to ON (rts_to.thread_id = www_rthread.id AND rts_to.id = www_rtstation.id + 1) LEFT JOIN
    importinfo_twostagebusimportpackage ON www_route.two_stage_package_id = importinfo_twostagebusimportpackage.id
    LEFT JOIN
        red_metaroute ON www_route.red_metaroute_id = red_metaroute.id
WHERE
    www_route.hidden = 0 AND
    www_rtstation.is_technical_stop = 0 AND
    www_rtstation.in_station_schedule = 1 AND
    COALESCE(www_rtstation.tz_arrival, -1) <> COALESCE(www_rtstation.tz_departure, -1) AND
    www_rthread.tz_year_days <> REPEAT('0', 372) AND
    COALESCE(www_rtstation.departure_code_sharing, 0) = 0
        """ + additional_conditions

        log.debug(sql % additional_params)

        self.cursor.execute(sql, additional_params)

    def find_template(self, mask, schedule_plan=None):
        text, template, extrapolable = self.matcher.find_template(
            mask,
            self.known_schedule_length,
            self.use_schedule_plan and schedule_plan,
            self.template_min_match_days
        )

        return text, template, extrapolable

    def should_extrapolate_thread(self, thread):
        """Нужно ли в процессоре экстраполировать нитку"""
        return self.extrapolate and thread.has_extrapolatable_mask

    @cache_method_result
    def get_schedule_change_date(self, date_):
        return TrainSchedulePlan.get_schedule_end_period(date_)

    def extrapolate_thread(self, thread):
        u"""Произвести экстраполяцию нитки
        @return: is_extrapolated"""
        if not self.should_extrapolate_thread(thread):
            return False

        thread_today = self.now_aware.astimezone(thread.pytz).date()
        mask = thread.get_mask(today=thread_today)

        text, template, extrapolable = self.find_template(mask, thread.schedule_plan)

        if not extrapolable:
            return False

        extrapolate_to = self.get_extrapolation_to(thread, mask)
        extrapolate_from = max(thread_today - timedelta(14), self.matcher.get_template_first_day(mask))

        extrapolated = mask | (
            RunMask(days=template.days) & RunMask.range(extrapolate_from, extrapolate_to, include_end=True)
        )

        if extrapolated:
            if len(str(extrapolated)) != RunMask.MASK_LENGTH:
                log.error(u'Неправильная маска у нитки %s', thread.uid)
                return

            thread.year_days = extrapolated

        return extrapolated.difference(mask)

    def get_extrapolation_to(self, thread, mask):
        # Экстраполируем до следующей смены расписания или предела для маски (что раньше)

        thread_today = self.now_aware.astimezone(thread.pytz).date()
        extrapolation_limit = thread_today + timedelta(self.extrapolation_limit_length)

        extrapolate_to_dates = [extrapolation_limit]

        if self.extrapolation_limit_with_schedule_plan:
            if thread.schedule_plan and thread.schedule_plan.end_date:
                schedule_change_date = thread.schedule_plan.end_date

            else:
                schedule_change_date = self.get_schedule_change_date(mask.dates()[-1])

            if schedule_change_date:
                extrapolate_to_dates.append(schedule_change_date)

        return min(extrapolate_to_dates)

    def precalc_days_texts(self, thread):
        path = list(thread.path)

        if len(path) < 2:
            log.error(u'Неправильный путь у нитки %s' % thread.uid)

            return

        thread_today = self.now_aware.astimezone(thread.pytz).date()

        # На сколько дней считать расписание
        # TODO: добавить поддержку графиков
        first_run = RunMask.first_run(thread.year_days, thread_today) or thread_today

        naive_start_dt = datetime.combine(first_run, thread.tz_start_time)
        start_dt = thread.pytz.localize(naive_start_dt)

        last_rts = path[-1]
        arrival_dt = last_rts.pytz.localize(naive_start_dt + timedelta(minutes=last_rts.tz_arrival))

        duration = int((arrival_dt - start_dt).total_seconds() / 60)

        days = (duration + 1440 - 1) / 1440

        mask = thread.get_mask(today=thread_today)

        except_mask = RunMask(today=thread_today)

        if self.use_thread_changes:
            except_mask = reduce(
                lambda _except, _thread:
                _except | RunMask(_thread.year_days, today=thread_today),
                thread.thread_changes.all(),
                except_mask
            )

        days_texts = []
        except_texts = []

        run_days_mask = mask | except_mask

        for shift in range(-1, days + 2):  # от -1 до days + 1
            mask = self.get_shifted(run_days_mask, shift)
            text, _, extrapolated = self.find_template(mask, thread.schedule_plan)

            days_texts.append(text)

            if self.use_thread_changes:
                mask = self.get_shifted(except_mask, shift)
                text = commaseparated_datelist(mask.dates())

                except_texts.append(text)
            else:
                except_texts.append(None)

        thread.translated_days_texts = json.dumps(days_texts, ensure_ascii=False)
        thread.translated_except_texts = json.dumps(except_texts, ensure_ascii=False)

    def _prepare_matcher(self, thread):
        thread_today = self.now_aware.astimezone(thread.pytz).date()

        self.template_first_day = thread_today - timedelta(settings.DAYS_TO_PAST)

        self.matcher = get_matcher(self.templates, thread_today, self.template_first_day, 365)
        self.matcher.search_first_day_in_past_days = self.search_first_day_in_past_days

    def _prepare_queryset(self):
        """Подготовить QuerySet для итерации по ниткам"""
        qs = RThread.objects.filter(self.get_thread_filter())
        ss_qs = StationSchedule.objects.filter(self.get_station_schedule_filter())

        if self.partial:
            qs = qs.filter(changed=True)
            ss_qs = ss_qs.filter(Q(thread__changed=True) | Q(route__hidden=True))

        if self.route_id:
            qs = qs.filter(route=self.route_id)
            ss_qs = ss_qs.filter(route=self.route_id)

        self.thread_qs = qs
        self.station_schedule_qs = ss_qs

    @transaction.atomic
    def clean_from_exclusions(self):
        log.info(u"Очистка от исключений")
        for exclusion in ScheduleExclusion.get_all_exclusion_pairs():
            self.cursor.execute("DELETE FROM www_stationschedule WHERE thread_id = %s AND station_id = %s",
                                [exclusion[0].id, exclusion[1].id])

    @cache_until_switch
    def get_shifted(self, run_days_mask, shift):
        return run_days_mask.shifted(shift)

    def post_extrapolate(self, thread):
        pass


class TrainThreadExtrapolationChecker(object):
    def __init__(self):
        self.plain_number_regexp = re.compile("^(\d+)")
        self.extrapolated_numbers_ranges = [
            (1, 168),
            (301, 398),
            (601, 998),
        ]

    def should_extrapolate(self, thread):
        if thread.route.script_protected:
            return False
        plain_number = self._get_plain_number(thread)

        if plain_number is not None:
            for low, high in self.extrapolated_numbers_ranges:
                if low <= plain_number <= high:
                    return True

    def _get_plain_number(self, thread):
        m = self.plain_number_regexp.match(thread.number)
        if m:
            return int(m.group(1))


class TrainProcessor(ScheduleProcessor):
    name = 'Train'
    extrapolation_limit_with_schedule_plan = True
    use_schedule_plan = True
    known_schedule_length = 40
    extrapolate = False
    template_min_match_days = 7
    search_first_day_in_past_days = 14

    def __init__(self, now_aware=None, partial=False, route_id=None):
        super(TrainProcessor, self).__init__(now_aware, partial, route_id)
        self.extrapolation_checker = TrainThreadExtrapolationChecker()

    def get_thread_filter(self):
        return Q(t_type_id=TransportType.TRAIN_ID) & Q(route__two_stage_package__isnull=True)

    def get_station_schedule_filter(self):
        return Q(thread__t_type_id=TransportType.TRAIN_ID) & Q(route__two_stage_package__isnull=True)

    def should_extrapolate_thread(self, thread):
        return self.extrapolation_checker.should_extrapolate(thread)

    def post_extrapolate(self, thread):
        """
        Вычитаем дни хождения остальных основных ниток из экстраполированной маски,
        чтобы не было дубликатов после экстраполяции.

        FIXME: Вычитание работает не правильно,
        если нитки стартуют близко к полуночи и могут стартовать в один и тот же день.
        Или если отличаются временные зоны ниток.
        """

        thread_today = self.now_aware.astimezone(thread.pytz).date()

        if thread.type_id != RThreadType.BASIC_ID:
            return

        other_basic_threads = (RThread.objects.filter(route=thread.route_id, type_id=RThreadType.BASIC_ID)
                                      .exclude(id=thread.id))
        if not other_basic_threads:
            return

        start_rts = thread.path[0]
        other_basic_mask = RunMask(today=thread_today)

        for other_basic_thread in other_basic_threads:
            if start_rts.station_id != other_basic_thread.path[0].station_id:
                continue

            other_basic_mask |= other_basic_thread.get_mask(today=thread_today)

        if other_basic_mask:
            log.info(u"Вычитаем изменения из %s %s дней; %s",
                     thread.uid, len(other_basic_mask.dates()), other_basic_mask.dates())

        thread.year_days = str(thread.get_mask(today=thread_today).difference(other_basic_mask))


class TisSuburbanProcessor(TrainProcessor):
    name = 'Tis suburban'

    def get_thread_filter(self):
        return Q(t_type_id=TransportType.SUBURBAN_ID) & Q(supplier__code='tis')

    def get_station_schedule_filter(self):
        return Q(thread__t_type_id=TransportType.SUBURBAN_ID) & Q(thread__supplier__code='tis')


class BusProcessor(ScheduleProcessor):
    name = 'Bus'
    known_schedule_length = 9
    extrapolate = False

    def get_thread_filter(self):
        return (
            Q(t_type_id=TransportType.BUS_ID) &
            Q(route__two_stage_package__isnull=True) &
            Q(route__red_metaroute__isnull=True) &
            ~Q(supplier__code='mta')
        )

    def get_station_schedule_filter(self):
        return (
            Q(thread__t_type_id=TransportType.BUS_ID) &
            Q(route__two_stage_package__isnull=True) &
            Q(route__red_metaroute__isnull=True) &
            ~Q(thread__supplier__code='mta')
        )


class RedBusProcessor(ScheduleProcessor):
    name = 'Red Bus'
    known_schedule_length = 9
    extrapolate = True

    def get_thread_filter(self):
        return Q(t_type_id=TransportType.BUS_ID) & Q(route__red_metaroute__isnull=False)

    def get_station_schedule_filter(self):
        return Q(thread__t_type_id=TransportType.BUS_ID) & Q(route__red_metaroute__isnull=False)


class MTABusProcessor(ScheduleProcessor):
    name = 'Bus MTA'
    known_schedule_length = 30
    extrapolate = False

    def get_thread_filter(self):
        return Q(t_type_id=TransportType.BUS_ID) & Q(route__two_stage_package__isnull=True) & Q(supplier__code='mta')

    def get_station_schedule_filter(self):
        return (
            Q(thread__t_type_id=TransportType.BUS_ID) &
            Q(route__two_stage_package__isnull=True) &
            Q(thread__supplier__code='mta')
        )


class AFSuburbanProcessor(ScheduleProcessor):
    name = 'AF Suburban'
    templates = 'all'
    use_thread_changes = True
    use_schedule_plan = True

    def get_thread_filter(self):
        return Q(t_type_id=TransportType.SUBURBAN_ID) & Q(supplier__code='af')

    def get_station_schedule_filter(self):
        return Q(thread__t_type_id=TransportType.SUBURBAN_ID) & Q(thread__supplier__code='af')


class WaterProcessor(ScheduleProcessor):
    name = 'Water'
    known_schedule_length = 30

    def get_thread_filter(self):
        return Q(t_type_id=TransportType.WATER_ID) & Q(route__two_stage_package__isnull=True)

    def get_station_schedule_filter(self):
        return Q(thread__t_type_id=TransportType.WATER_ID) & Q(route__two_stage_package__isnull=True)


class CysixScheduleProcessor(ScheduleProcessor):
    name = 'Cysix'
    do_prepare_threads = False

    def get_thread_filter(self):
        return Q(route__two_stage_package__isnull=False)

    def get_station_schedule_filter(self):
        return Q(route__two_stage_package__isnull=False)


def extrapolate_and_gen_textmask(processors, partial=True, route_id=None):
    now_aware = environment.now_aware()
    start = os_time.time()

    gc.disable()
    try:
        for klass in processors:
            klass(now_aware=now_aware, partial=partial, route_id=route_id).run()
    finally:
        gc.collect()
        gc.enable()

    log.info('z_station_schedule finished in %s sec' % (os_time.time() - start))


if __name__ == '__main__':
    with ylog_context(**get_script_log_context()):
        create_current_file_run_log()

        from optparse import OptionParser

        parser = OptionParser(description=u'Заполнить расписания станций нитками, пересчитанными в локальное время')
        parser.add_option("-v", "--verbose", dest="verbose", action="store_true")
        parser.add_option("-p", "--partial", dest="partial", action="store_true")

        parser.add_option("--bus", dest="bus", action="store_true")
        parser.add_option("--tis", dest="tis", action="store_true")
        parser.add_option("--af", dest="af", action="store_true")
        parser.add_option('--water', dest='water', action='store_true')
        parser.add_option('--cysix', dest='cysix', action='store_true')
        parser.add_option('--mta', dest='mta', action='store_true')

        parser.add_option("--route_id", dest="route_id")

        (options, args) = parser.parse_args()

        if options.verbose:
            print_log_to_stdout()

        if not options.bus and \
           not options.tis and \
           not options.af and \
           not options.cysix and \
           not options.mta:
            options.bus = True
            options.tis = True
            options.af = True
            options.water = True
            options.cysix = True
            options.mta = True

        processors = []
        if options.tis:
            processors.append(TrainProcessor)
            processors.append(TisSuburbanProcessor)

        if options.bus:
            processors.append(BusProcessor)
            processors.append(RedBusProcessor)

        if options.af:
            processors.append(AFSuburbanProcessor)

        if options.water:
            processors.append(WaterProcessor)

        if options.cysix:
            processors.append(CysixScheduleProcessor)

        if options.mta:
            processors.append(MTABusProcessor)

        partial = options.partial or flags['partial_preparation']

        extrapolate_and_gen_textmask(processors, partial=partial, route_id=options.route_id)
