var app = require('app'),
    _ = require('underscore'),
    Backbone = require('Backbone'),
    WwwStationModel = require('modules/www_stations/model');

var StationModel = Backbone.RelationalModel.extend({
    relations: [{
        type: Backbone.HasOne,
        key: 'www_station',
        relatedModel: WwwStationModel
    }],

    parse: function(resp, options) {
        if (!resp.lat || !resp.lon) {
            resp.placed = false;
        } else {
            resp.placed = true;
        }
        return resp;
    },

    isWwwPlaced: function() {
        return this.get('www_station') && this.get('www_station').id;
    },

    getType: function() {
        if (this.isWwwPlaced()) {
            // Привязанная к геокодированной
            if (this.get('placed')) {
                return 'finished';
            }
            // Привязанная к негеокодированной станции
            return 'mapped';
        }
        // Непривязанная, но установленная мышью на карте, готовая к созданию
        if (this.get('placed')) {
            return 'placed';
        }
        // Непришей козе хвост
        return 'none';
    },

    getTypeTTitle: function() {
        var wwwStation = this.get('www_station');
        if (wwwStation) {
            var type = _.findWhere(app.config['t_types'], {id: wwwStation.get('t_type_id') });
            return type.title;
        }

        return false;
    },

    getAdminUrl: function() {
        return app.config.stationAdminUrl + this.get('id');
    },

    map: function(wwwStation, options) {
        var self = this;

        options = options || {};
        options = _.extend({}, options, {
            type: 'POST',
            data: {
                'www_station_id': wwwStation.get('id')
            },
            url: _.result(this, 'url') + '/map',
            success: function(res) {
                res.placed = true;
                self.set(res);
            }
        });

        return Backbone.ajax(options);
    },

    unmap: function(options) {
        var self = this;

        options = options || {};
        options = _.extend({}, options, {
            type: 'POST',
            url: _.result(this, 'url') + '/unmap',
            success: function(res) {
                res.placed = false;
                self.set(res);
            }
        });

        return Backbone.ajax(options);
    },

    createWwwStation: function(options) {
        var wwwStationModel = new WwwStationModel(options);
        this.set('www_station', wwwStationModel);

        // После сохранения станция должна попасть в www коллекцию
        var checkWwwModelInCollection = function(model, wwwStationModel) {
            if (!wwwStationModel || wwwStationModel.isNew()) {
                return;
            }

            this.off('change:www_station', checkWwwModelInCollection);

            if (!app.wwwCollection.contains(wwwStationModel)) {
                app.wwwCollection.add(wwwStationModel);
            }
        };
        this.on('change:www_station', checkWwwModelInCollection, this);
    }
});

module.exports = StationModel;
