var app = require('app'),
    BalloonPlacemark = require('common/balloonPlacemark'),
    StationMappedForm = require('modules/blue_stations/mappedForm'),
    StationForm = require('modules/blue_stations/unMappedForm');


var StationPlacemark = BalloonPlacemark.extend({
    placemarkOptions: {
        draggable: true,
        balloonMinWidth: 400,
        balloonMaxWidth: 500,
        balloonMinHeight: 200,
        balloonMaxHeight: 520,
        balloonAutoPanMargin: 60
    },

    events: {
        'balloonopen': 'onBalloonOpen',
        'balloonclose': 'onBalloonClose',
        'drag': 'dragStart',
        'dragend': 'dragEnd',
    },

    initialize: function(options) {
        this.listenTo(this.model, 'hover', function() {
            this.setStyle('yellow');
        }, this);

        this.listenTo(this.model, 'unhover', function() {
            this.setStyle(this.getStyle());
        }, this);

        this.listenTo(this.model, 'change', function() {
            this.setStyle(this.getStyle());
        }, this);

        this.setStyle(this.getStyle());
    },

    getCoordinates: function() {
        var wwwStation = this.model.get('www_station');
        return [
            this.model.get('lon') || (wwwStation && wwwStation.get('lon')),
            this.model.get('lat') || (wwwStation && wwwStation.get('lat'))
        ];
    },

    iconContent: function() {
        return this.model.get('number');
    },

    hintContent: function() {
        return this.model.get('code') + '<br />' + this.model.get('title');
    },

    getStyle: function() {
        var type = this.model.getType(),
            colors = {
                finished: 'green',
                mapped: 'brown',
                placed: 'yellow',
                none: 'red'
            };

        return colors[type];
    },

    _openBallon: function() {
        var type = this.model.getType(),
            View,
            _this = this;

        if (type === 'finished' || type === 'mapped') {
            View = StationMappedForm;
        } else {
            View = StationForm;
        }

        this.balloonView = new View({
            model: this.model
        });

        this.listenTo(this.balloonView, 'close', function() {
            this.geoObject.balloon.close();
        }, this);

        this.listenTo(this.balloonView, 'coordinatesChanged', function() {

            this.updatePlacemarkCoordinates();

            this.updateModelCoordinates();
            this.model.set('placed', true);

            this.model.collection.fillMissedGeo();

            _this.geoObject.balloon.close();
            _this.geoObject.balloon.open();

        }, this);

        this.geoObject.options.unset('balloonContent');
        var balloonView = this.balloonView;
        var layout = ymaps.templateLayoutFactory.createClass('<div></div>', {
            build: function () {
                layout.superclass.build.call(this);
                this._element.appendChild(balloonView.render().el);
            }
        });

        this.geoObject.options.set('balloonContentLayout', layout);
        this.geoObject.balloon.autoPan();
    },

    onBalloonOpen: function() {
        var map = this.geoObject.getMap();
        map.setCenter(this.getCoordinates());

        // сдвигаем карту вправо, чтобы балун поместился
        var position = map.getGlobalPixelCenter();
        map.setGlobalPixelCenter([ position[0] + 250, position[1] ]);

        this._openBallon();
    },

    onBalloonClose: function() {
        this.balloonView && this.balloonView.close();
        this.geoObject.properties.set('balloonContent', this.balloonContent);
    },

    mapToWwwStationDialog: function() {
        var self = this,
            wwwStation = this.model.get('www_station');

        function confirmTTypeBinding() {
            var text = [
                gettext(gettext('ВНИМАНИЕ! Тип транспорта станции')),
                '"',
                self.model.getTypeTTitle(),
                '".',
                gettext(gettext('Действительно привязать?'))
            ];

            return confirm(text.join(' '));
        }

        function confirmBinding() {
            var text = [
                gettext(gettext('Привязать станцию поставщика')),
                self.model.get('code'),
                self.model.get('title'),
                gettext(gettext('к станции в базе')),
                self.closest.get('id'),
                self.closest.get('title')
            ];

            return confirm(text.join(' ') + '"?');
        }

        return (!wwwStation || wwwStation.get('t_type_id') === 3 || confirmTTypeBinding()) && confirmBinding();
    },

    moveDialog: function() {
        var text = [
            this.model.isWwwPlaced() ? gettext('Переместить') : gettext('Геокодировать'),
            gettext('станцию'),
            this.model.get('www_station').id,
            this.model.get('www_station').title
        ];

        return confirm(text.join(' ') + '?');
    },

    dragStart: function(e) {
        if (this.model.get('www_station')) {
            return;
        }

        var coords = e.get('target').geometry.getCoordinates(),
            closest = app.wwwCollection.findClosest(coords, 1);

        // Рядом есть станция - объединение покажем цветом
        if (closest && closest !== this.model) {
            this.setStyle('green');
            this.closest = closest;
        } else {
            this.setStyle(this.getStyle());
            this.closest = null;
        }
    },

    dragEnd: function(e) {
        // Привязанная станция
        if (this.model.get('www_station') && this.model.get('www_station').id) {
            if (this.moveDialog()) {
                this.updateModelCoordinates();
                this.model.save(null, {
                    humanError: gettext('Не удалось переместить станцию')
                });
            } else {
                this.updatePlacemarkCoordinates();
                this.setStyle(this.getStyle());
            }
            return;
        }

        // "красная" станция
        if (this.closest) { // Привязка к станции в нашей базе
            if (this.mapToWwwStationDialog()) {
                this.model.map(this.closest, {
                    humanError: gettext('Ошибка, новое соответствие не создано')
                });
            } else {
                this.updatePlacemarkCoordinates();
                this.setStyle(this.getStyle());
            }
            return;
        }

        // Корректно установленная, но непривязанная станция
        this.updateModelCoordinates();
        this.model.set('placed', true);

        this.model.collection.fillMissedGeo();
    }
});

module.exports = StationPlacemark;
