var _ = require('underscore'),
    Marionette = require('marionette'),
    BackboneYmaps = require('Backbone.Ymaps'),
    PlacemarkColleciton = require('modules/www_stations/placemarkCollection'),
    WwwStationModel = require('modules/www_stations/model'),
    WwwStationCollection = require('modules/www_stations/collection'),
    WwwStationPlacemark = require('modules/www_stations/placemark'),
    PsevdoGorTransPlacemark = require('modules/www_stations/psevdoGorTransPlacemark'),
    NewWwwStationForm = require('modules/www_stations/newStationForm'),
    WwwStationsControl = require('modules/www_stations/controls'),
    presets = require('modules/www_stations/presets'),
    cookie = require('jquery.cookie');


var WwwStationsController = Marionette.Controller.extend({
    controls: {
        stationsControl: WwwStationsControl
    },

    stationTypes: [],

    initialize: function(options) {
        presets.addToStorage(options.staticUrl);

        this.selectedType = options.defaults.transportType;

        this.app = options.app;

        this.stationTypes = cookie('station_layers_controll_selected');
        if (this.stationTypes) {
            this.stationTypes = JSON.parse(this.stationTypes);
        }
        if (!this.stationTypes || this.stationTypes.length === 0) {
            this.stationTypes = [this._getTransportTypeCode(this.selectedType)];
        }

        this.collection = new WwwStationCollection([], {
            stationTypes: this.stationTypes
        });

        this.app.wwwCollection = this.collection;
        this.app.wwwSettlements = this.collection.settlements;

        this._initContols();
        this._renderMapLayout();
        this._bind();
    },

    // API
    typeUpdate: function(types) {
        var flag = this.stationTypes.length; // чтобы не делать запрос при первом появлении карты
        this.stationTypes.length = 0;
        this.stationTypes.push.apply(this.stationTypes, types);

        cookie('station_layers_controll_selected', JSON.stringify(this.stationTypes));

        flag && this.resetData();
    },

    typeAdd: function(type) {
        var index = this.stationTypes.indexOf(type);
        if (index === -1) {
            this.stationTypes.push(type);
            this.resetData();
        }
    },

    typeRemove: function(type) {
        var index = this.stationTypes.indexOf(type);
        if (index !== -1) {
            this.stationTypes.splice(index, 1);
            this.resetData();
        }
    },

    resetData: function() {
        if (!this._resetData) {
            var self = this;
            this._resetData = _.debounce(function() {
                if (!self.stationTypes.length) {
                    return;
                }

                self.collection.fetch();
            }, 100);
        }

        this._resetData();
    },

    create: function(position, options, callback, context) {
        var model = new WwwStationModel(options),
            form = new NewWwwStationForm({
                model: model
            });

        this.listenTo(form, 'close', function(options) {
            this.app.map.balloon.close();
            form.close();
        }, this);

        model.once('change:id', function() {
            this.collection.add(model);
            this.trigger('created', { model: model });
        }, this);

        var layout = ymaps.templateLayoutFactory.createClass('<div></div>', {
            build: function () {
                layout.superclass.build.call(this);
                this._element.appendChild(form.render().el);
            }
        });

        this.app.map.balloon.open(position, '', {
            'contentLayout': layout,
            'minWidth': 200,
            'maxWidth': 600
        });
    },

    setSelectedType: function(type) {
        var code = this._getTransportTypeCode(type);

        this.selectedType = type;
        if (_.indexOf(this.stationTypes, code) === -1) {
            this.stationTypes.push(code);
        }
    },

    // Private
    _initContols: function() {
        _.each(this.controls, function(control, key) {
            this.controls[key] = control(this);
        }, this);
    },

    _bind: function() {
        this.listenTo(this.app.maps, 'boundschange', this.resetData, this);
        this.listenTo(this.app, 'wwwStations:update', this.resetData, this);
        this.listenTo(this.app, 'wwwStations:typeAdd', this.typeAdd, this);
        this.listenTo(this.app, 'wwwStations:typeRemove', this.typeRemove, this);
        this.listenTo(this.app, 'wwwStations:typeUpdate', this.typeUpdate, this);
        this.listenTo(this.app, 'wwwStations:setSelected', this.setSelectedType, this);
    },

    _renderMapLayout: function() {
        var layer = new PlacemarkColleciton({
            geoItem: WwwStationPlacemark,
            collection: this.collection,
            map: this.app.map
        });

        layer.render();

        // Bubbling
        this.listenTo(layer, 'all', function(name, options) {
            this.trigger(name, options);
        });

        // psevdoGorTrans
        layer = new BackboneYmaps.CollectionView({
            geoItem: PsevdoGorTransPlacemark,
            collection: this.collection.psevdoGortrans,
            map: this.app.map
        });

        layer.render();
    },

    _getTransportTypeCode: function(tTypeId) {
        return this.app.config['t_types'][tTypeId].code;
    }
});

module.exports = WwwStationsController;
