/**
 * Управление отображением на карте станции
 *
 * @param map
 * @param options
 * @constructor
 */
geomap.control.StationLayers = function (map, options) {
    var self = this;

    self.COOKIE = "station_layers_controll_selected";

    geomap.control.StationLayers.superclass.constructor.apply(self, [map, options, 'stationLayers']);
    geomap.control.HasData.prototype._init.call(self);

    self.globalId = 'geomap_StationLayers_' + options.id;
    document[self.globalId] = self;

    self._initMapControl();

    self._additionalDataProviders = [];
    self._stationIconImageInterceptors = [];

    self._requiredTransportType = null;

    self._stationsOnMap = {};

    self.placemarkOptions = self._preparePlacemarkOptions(options.station && options.station.placemarker ? options.station.placemarker : {});

    self.map.events.add(['boundschange'], function (event) {
        self.refresh();
    });
};

extend(geomap.control.StationLayers, geomap.control.AbstractControl);

mixin(geomap.control.StationLayers.prototype, geomap.control.HasData.prototype);

mixin(geomap.control.StationLayers.prototype, {

    refresh: function () {
        var self = this;
        if (self._scheduled) {
            return;
        }
        self._scheduled = true;
        self.map.readyState.onReadyOnce(function (ready) {
            if (ready) {
                self._scheduled = false;
                geomap.control.HasData.prototype.refresh.call(self);
            }
        });
    },

    addAdditionalDataProviders: function (dataProvider, options) {
        var self = this;
        if (!options) {
            options = {};
        }
        self._additionalDataProviders.push(
            {
                provider: dataProvider,
                placemarkOptions: self._preparePlacemarkOptions(options)
            }
        );
    },

    addStationIconImageInterceptor: function(interceptor) {
        var self = this;
        self._stationIconImageInterceptors.push(interceptor);
    },

    setRequiredTransportType: function (requiredTransportType) {
        var self = this;
        if (self._requiredTransportType) {
            self._transportTypesControlIndex[self._requiredTransportType].enable();
        }

        self._requiredTransportType = requiredTransportType;

        var control = self._transportTypesControlIndex[self._requiredTransportType];
        control.select();
        control.disable();
    },

    defaultDataProvider: function (map, callback) {
        var self = this,
            center = map._map.getCenter(),
            bounds = map._map.getBounds(),
            params = {
                center: '' + center[0] + ',' + center[1],
                span: '' + (bounds[1][0] - bounds[0][0]) + ',' + (bounds[1][1] - bounds[0][1]),
                zoom: map._map.getZoom(),
                t_types: self._getSelectedTransportTypes()
            };

        $.getJSON('/admin/redmap/www_stations/', $.param(params, true), function (data) {
            var stations = [];
            if (data.stations) {
                if (data.gort_stations) {
                    stations = data.stations.concat(data.gort_stations);
                } else {
                    stations = data.stations;
                }
            } else if (data.gort_stations) {
                stations = data.gort_stations;
            }

            callback(stations);
        });
    },

    _doLayout: function (currentValue) {
        var self = this,
            additionalStations = self._getAdditionalStations(),
            oldStationsOnMap = self._stationsOnMap;

        self._stationsOnMap = {};
        $.each(self._objectIndex, function (i, station) {
            var oldStation = oldStationsOnMap[i],
                additionalStation = additionalStations[i];

            if (additionalStation) {
                return;
            }
            if (oldStation) {
                if (oldStation.isAdditional()) {
                    oldStation.destroy();
                    delete oldStationsOnMap[i];
                    self._addStation(station, self.placemarkOptions, false);
                } else {
                    self._stationsOnMap[i] = oldStation;
                    delete oldStationsOnMap[i];
                }
            } else {
                self._addStation(station, self.placemarkOptions, false);
            }
        });

        $.each(oldStationsOnMap, function (i, station) {
            station.destroy();
        });

        $.each(additionalStations, function (i, station) {
            self._addStation(station.value, station.placemarkOptions, true);
        });
    },

    defaultStationPlacemarkIconContentProvider: function (station) {
        return null;
    },

    defaultStationPlacemarkOptionsProvider: function (station) {
        var self = this;
        return {
            draggable: true,
            hideIconOnBalloonOpen: false,
            iconLayout: 'default#image',
            iconImageSize: [11, 11],
            iconImageOffset: [-5, -5],
            iconImageHref: self.stationIconImage(station)
        };
    },

    defaultStationPlacemarkHintContentProvider: function (station) {
        return station.title;
    },

    defaultStationPlacemarkBalloonContentProvider: function (station) {
        return '<div><table>' +
            '<tr><td>' + station.id + '</td></tr>' +
            '<tr><td>' + station.title + '</td></tr>' +
            '<tr><td><a href="/admin/www/station/' + station.id + '" target="_blank">' + gettext('Редактировать в геоадмине') + '</a></td></tr>' +
            '</table></div>';
    },

    stationIconImage: function (station) {
        var self = this;
        for (var i = 0; i < self._stationIconImageInterceptors.length; ++i) {
            var result = self._stationIconImageInterceptors[i].call(self, station);
            if (result) {
                return result;
            }
        }
        return self._imgByTransportType[station.t_type];
    },

    setZoom: function (input) {
        var self = this;
        $.each(self._transportTypesItemIndex[input.id].tTypes, function (i, tType) {
            $.cookie('map_' + tType + '_zoom', input.value, { path: "/admin/" });
        });
        self.refresh();
    },

    _afterStationMove: function (station) {
        var maxDistance = 100, // 100 метров
            self = this,
            nearlyStation;

        $.each(self._stationsOnMap, function (i, item) {
            if (station !== item) {
                var distance = station.distance(item);
                if (distance < maxDistance) {
                    maxDistance = distance;
                    nearlyStation = item;
                }
            }
        });

        if (nearlyStation && confirm(gettext('Объединить станцию') + ' ' + station.title() + ' ' + gettext('с') + ' ' + nearlyStation.title() + ' ?')) {
            $.post('/admin/redmap/www_station/join/',
                   {
                       'src_id': station.val().id,
                       'dst_id': nearlyStation.val().id
                   },
                   function (data) {
                       if (data.status == 'ok') {
                           station.destroy();
                       } else {
                           alert(gettext('Не смогли объединить станции'));
                       }
                   },
                   'json'
            );
            station.destroy();
            return;
        }

        if (!confirm(gettext('Переместить станцию') + ' ' + station.title() + ' ?')) {
            station.resetCoordinates();
            return;
        }

        $.post('/admin/redmap/www_station/move/',
               {
                   'station_id': station.val().id,
                   'lon': station.lon(),
                   'lat': station.lat()
               },
               function (data) {
                   if (data.status != 'ok') {
                       alert(gettext('Не смогли переместить станцию') + ' ' + station.title());
                   } else {
                       station.syncCoordinates();
                   }
               },
               'json'
        );
    },

    _getAdditionalStations: function () {
        var self = this,
            stations = {};
        $.each(self._additionalDataProviders, function (i, dataProvider) {
            $.each(dataProvider.provider(), function (i, station) {
                stations[self._objectId(station)] = {
                    value: station,
                    placemarkOptions: dataProvider.placemarkOptions
                };
            });
        });
        return stations;
    },

    _addStation: function (station, placemarkOptions, additional) {
        if (!station) {
            return;
        }
        var self = this,
            result = {

                title: function () {
                    return station.id + ' ' + station.title;
                },

                val: function () {
                    return station;
                },

                getCoordinates: function () {
                    return this.placemark.geometry.getCoordinates();
                },

                lon: function () {
                    return this.getCoordinates()[0];
                },

                lat: function () {
                    return this.getCoordinates()[1];
                },

                syncCoordinates: function () {
                    station.lon = this.lon();
                    station.lat = this.lat();
                },

                resetCoordinates: function() {
                    this.placemark.geometry.setCoordinates(geomap.utils.toPoint(station));
                },

                distance: function (other) {
                    return geomap.utils.distance(this.getCoordinates(), other.getCoordinates());
                },

                destroy: function () {
                    if (this.placemark) {
                        self.map._map.geoObjects.remove(this.placemark);
                        delete self._stationsOnMap[station.id];
                    }
                },

                create: function () {
                    this.placemark = new ymaps.Placemark(
                        geomap.utils.toPoint(station),
                        {
                            iconContent: placemarkOptions.iconContentProvider.call(self, station),
                            hintContent: placemarkOptions.hintContentProvider.call(self, station),
                            balloonContent: placemarkOptions.balloonContentProvider.call(self, station)
                        },
                        placemarkOptions.optionsProvider.call(self, station)
                    );

                    self.map._map.geoObjects.add(this.placemark);
                    self._stationsOnMap[station.id] = this;
                },

                isAdditional: function () {
                    return additional;
                }

            };

        result.create();
        result.placemark.events.add('dragend', function () {
            self._afterStationMove(result);
        });
    },

    _preparePlacemarkOptions: function (options) {
        var self = this;
        return {
            iconContentProvider: options.iconContentProvider || self.defaultStationPlacemarkIconContentProvider,
            optionsProvider: options.optionsProvider || self.defaultStationPlacemarkOptionsProvider,
            hintContentProvider: options.hintContentProvider || self.defaultStationPlacemarkHintContentProvider,
            balloonContentProvider: options.balloonContentProvider || self.defaultStationPlacemarkBalloonContentProvider
        }
    },

    _transportTypesItem: [
        { title: gettext('Автобусы'), tTypes: ['bus'], color: 'yellow', id: 'bus' },
        { title: gettext('Городской транспорт'), tTypes: ['urban'], color: 'black', id: 'urban' },
        { title: gettext('Поезда и пригородные поезда'), tTypes: ['train', 'local_train'], color: 'green', id: 'train' },
        { title: gettext('Псевдо-гортранс'), tTypes: ['pseudo-gortrans'], color: 'grey', id: 'pseudo-gortrans' },
        { title: gettext('Речной и морской транспорт'), tTypes: ['water'], color: 'blue', id: 'sea' },
        { title: gettext('Самолеты и вертолеты'), tTypes: ['plane', 'helicopter'], color: 'orange', id: 'plane' }
    ],

    _initMapControl: function () {
        var self = this,
            items = self._transportTypesItem.sort(function (o1, o2) {
                if (o1.title == o2.title) {
                    return 0;
                }
                return o1.title > o2.title ? 1 : -1;
            }),
            selectedTypes = JSON.parse($.cookie(self.COOKIE)) || [];

        self._controlItems = $.map(items, function (item) {
            var imgUrl = self.map.resourceUrl("rasp/rainbow/" + item.color + "Dot.png"),
                zoom = $.cookie('map_' + item.id + '_zoom') || 10;

            return new ymaps.control.ListBoxItem(
                {
                    data: {
                        content: "<img src='" + imgUrl + "' /> " +
                            "<input type='number' id='" + item.id + "' min='8' max='18' " +
                            " style='height: 22px; border: 1px solid #000; padding: 0; margin: 0;' " +
                            " onchange='document." + self.globalId + ".setZoom(this);' value='" + zoom + "'" +
                            " onclick='event.stopPropagation();'/> " +
                            item.title,
                        id: item.id,
                        tTypes: item.tTypes,
                        img: imgUrl
                    },
                    state: {
                        selected: selectedTypes.indexOf(item.id) !== -1
                    }
                }, {
                    checkbox: true
                }
            );
        });
        self.control = new ymaps.control.ListBox(
            {
                data: {
                    title: gettext('Остановки'),
                    content: gettext('Остановки')
                },
                items: self._controlItems,
                options: {
                    float: 'right'
                }
            }
        );
        self.map._map.controls.add(self.control);

        $.each(self._controlItems, function (i, item) {
            item.events.add(['select', 'deselect'], function () {
                $.cookie(self.COOKIE, JSON.stringify(self._getSelectedTransportTypes()), { path: "/admin/" });
                self.refresh();
            });
        });

        self._transportTypesItemIndex = geomap.utils.buildIndex(self._transportTypesItem);

        self._imgByTransportType = {};
        self._transportTypesControlIndex = {};
        $.each(self._controlItems, function (i, item) {
            $.each(item.data.get('tTypes'), function (j, type) {
                self._imgByTransportType[type] = item.data.get('img');
                self._transportTypesControlIndex[type] = item;
            });
        });

    },

    _getSelectedTransportTypes: function () {
        var self = this,
            tTypes = [];

        $.each(self._controlItems, function (i, item) {
            if (item.isSelected()) {
                tTypes = tTypes.concat(item.data.get('tTypes'));
            }
        });

        return tTypes;
    }

});
