# coding: utf-8

import os.path

from django import forms
from django.conf import settings
from django.contrib import admin
from django.contrib.admin import SimpleListFilter
from django.db import models
from django.utils.translation import ugettext_lazy as _

from common.models.disclaimers import StaticText
from common.models.geo import Country
from common.models.staticpages import ArticleBinding, StaticLink, StaticPage, Partner
from common.models_utils.i18n import L_field
from travel.rasp.admin.lib.admin_options import RaspExportModelAdmin


class BigTextarea(forms.Textarea):
    def __init__(self, *args, **kwargs):
        if 'attrs' not in kwargs:
            kwargs['attrs'] = {}
        kwargs['attrs']['cols'] = 150
        kwargs['attrs']['rows'] = 25
        super(BigTextarea, self).__init__(*args, **kwargs)


class StaticLinkInline(admin.TabularInline):
    model = StaticLink
    extra = 2


class StaticPageAdmin(RaspExportModelAdmin):
    fieldsets = (
        (None, {
            'fields': (
                'parent',
                'title',
                'slug',
                'type',
                'is_published',
                'content',
                'template',
                'order',
                'is_ticket_page',
                'meta_title',
                'meta_description',
                'national_version',
            )
        }),
        ('Advanced', {'fields': ('materialized_path',), 'classes': ('collapse',)}),
        (_(u'Переводы'), {
            'fields': L_field.admin_fields(StaticPage, ['title', 'content', 'meta_title', 'meta_description'],
                                           group_by='lang')
        })
    )
    list_display = ('path_title', 'is_published', 'slug', 'add_child', 'is_ticket_page', 'national_version')
    list_filter = ('is_published', 'is_ticket_page', 'national_version')
    search_fields = ('title', )
    ordering = ('materialized_path', )
    formfield_overrides = {
        models.TextField: {'widget': BigTextarea},
    }
    inlines = (StaticLinkInline,)
    raw_id_fields = ('parent',)
    readonly_fields = ('materialized_path',)

admin.site.register(StaticPage, StaticPageAdmin)


class StaticLinkAdmin(RaspExportModelAdmin):
    list_display = ('title', 'url', 'order')
    search_fields = ('title', )
    ordering = ('order', )

admin.site.register(StaticLink, StaticLinkAdmin)


class ArticleBindingAdmin(RaspExportModelAdmin):
    raw_id_fields = ('page', 'settlement')
    fieldsets = (
        (None, {
            'fields': ('page', 'country', 't_type', 'settlement', 'priority')
        }),
    )
    list_display = ('page', 'country', 't_type', 'settlement', 'priority')
    search_fields = ('page__title', 'settlement__title')

    class ArticleCountryFilter(SimpleListFilter):
        title = _(u'Страна')
        parameter_name = 'country__id__exact'

        FILTER_COUNTRIES = [
            Country.RUSSIA_ID, Country.BELARUS_ID, Country.KAZAKHSTAN_ID, Country.UZBEKISTAN_ID, Country.UKRAINE_ID
        ]

        def lookups(self, request, model_admin):
            return [(c.id, _(c.title)) for c in Country.objects.filter(id__in=self.FILTER_COUNTRIES)]

        def queryset(self, request, queryset):
            if self.value():
                return queryset.filter(country=self.value())
            else:
                return queryset

    list_filter = ('t_type', ArticleCountryFilter,)

    def formfield_for_dbfield(self, db_field, request, **kwargs):
        formfield = super(ArticleBindingAdmin, self).formfield_for_dbfield(db_field, request, **kwargs)

        if db_field.name == 'country':
            formfield.initial = Country.RUSSIA_ID

        return formfield

admin.site.register(ArticleBinding, ArticleBindingAdmin)


class StaticTextAdmin(RaspExportModelAdmin):
    fieldsets = (
        (None, {'fields': ('code', 'title', 'sample_url', 'is_mobile')}),
        ('Localized content', {
            'fields': ['content_%s' % lang for lang in StaticText.LANGUAGES],
        }),
        ('Localized announcement', {
            'fields': ['announce_%s' % lang for lang in StaticText.LANGUAGES],
        }),
    )
    list_display = ('title', 'code')
    list_filter = ('is_mobile',)
    search_fields = ('title', 'code', 'content')
    ordering = ('title', )

admin.site.register(StaticText, StaticTextAdmin)


class PartnerForm(forms.ModelForm):
    class Meta:
        model = Partner
        exclude = []

    def clean_logo(self):
        data = self.cleaned_data.get('logo')

        allowed_extensions = ('.png', '.jpg', '.gif', '.bmp', '.jpeg', '.svg')

        if data:
            ext = os.path.splitext(data.name)[1]
            if ext not in allowed_extensions:
                raise forms.ValidationError(
                    _(u'Расширение файла %s не совпадает с допустимыми: %s')
                    % (ext, u', '.join(allowed_extensions))
                )

            if data.size > settings.MAX_UPLOADED_PARTNER_LOGO_SIZE_IN_BYTES:
                raise forms.ValidationError(
                    _(u'Файл %s превышает допустимый лимит %.2f МБ')
                    % (data.name, settings.MAX_UPLOADED_PARTNER_LOGO_SIZE_IN_BYTES / 1024. / 1024.)
                )

        return data


DOHOP_PARTNER_ID = 67


class PartnerAdmin(RaspExportModelAdmin):
    list_display = ('title', 'order', 'is_published', 'services', 'national_version')
    list_filter = ('services', 'national_version', 't_type')
    search_fields = ('title',)
    ordering = ('order', 'title')
    raw_id_fields = ('order_partner',)

    form = PartnerForm

    def save_model(self, request, obj, form, change):
        if obj.dohop_vendor_id:
            obj.order_partner_id = DOHOP_PARTNER_ID

        super(PartnerAdmin, self).save_model(request, obj, form, change)


admin.site.register(Partner, PartnerAdmin)
