# coding: utf-8

import calendar
import json
from collections import defaultdict
from datetime import date, datetime, timedelta
from functools import update_wrapper

from django.conf.urls import url
from django.conf import settings
from django.db.models import Sum
from django.contrib import admin
from django.shortcuts import render
from django.utils.translation import ugettext_lazy as _

from dateutil.relativedelta import relativedelta

from stats.models import ImportChart, ServiceMetric, FlexibleCache, DynamicCache, MinPriceStatus


class ImportChartAdmin(admin.ModelAdmin):
    list_display = ('when', 'partner_link', 'event', 'value', 'machine')
    list_filter = ('partner', 'event')

    def partner_link(self, obj):
        return '<a href="/admin/order/partner/%s/" target="_blank">%s</a>' % \
               (obj.partner.id, obj.partner.code)

    partner_link.allow_tags = True
    partner_link.short_description = _(u'Партнёр')


admin.site.register(ImportChart, ImportChartAdmin)


class ServiceMetricAdmin(admin.ModelAdmin):
    def percent(self, float1, float2):
        return int(float(float1) / float(float2) * 100) if float2 else 0

    def servicemetrics_table_view(self, request):
        def get_form_submits():
            today = date.today()
            yesterday = today - timedelta(days=1)
            fist_day_of_current_month = today.replace(day=1)

            form_submits = {}

            for service in settings.SERVICEMETRICS_SERVICES.keys():
                service_name = settings.SERVICEMETRICS_SERVICES[service]
                form_submits[service_name] = {}

                # вчера
                sql_data = ServiceMetric.objects.filter(
                    when=yesterday,
                    period='day',
                    group='form',
                    event='user_search',
                    service=service,
                )

                if sql_data:
                    sql_data = sql_data[0]
                    form_submits[service_name]['yesterday'] = int(sql_data.value)

                # сегодня
                sql_data = ServiceMetric.objects.filter(
                    when=today,
                    period='day',
                    group='form',
                    event='user_search',
                    service=service,
                )

                if sql_data:
                    sql_data = sql_data[0]
                    form_submits[service_name]['today'] = int(sql_data.value)

                # Mах за день
                sql_data = ServiceMetric.objects.filter(
                    when__gte=fist_day_of_current_month,
                    period='day',
                    group='form',
                    event='user_search',
                    service=service,
                ).values('when', 'value')

                if sql_data:
                    sql_data_dict = {}
                    for item in sql_data:
                        sql_data_dict[item['when']] = item['value']

                    max_key = max(sql_data_dict, key=sql_data_dict.get)
                    max_val = sql_data_dict[max_key]

                    form_submits[service_name]['max_day'] = {
                        'date': max_key.date(),
                        'value': int(max_val),
                    }

                # За текущий месяц
                sql_data = ServiceMetric.objects.filter(
                    when=fist_day_of_current_month,
                    period='month',
                    group='form',
                    event='user_search',
                    service=service,
                )

                if sql_data:
                    sql_data = sql_data[0]
                    form_submits[service_name]['current_month'] = int(sql_data.value)

                # Mах за месяц
                sql_data = ServiceMetric.objects.filter(
                    when__gte=fist_day_of_current_month,
                    period='month',
                    group='form',
                    event='user_search',
                    service=service,
                ).values('when', 'value')

                if sql_data:
                    sql_data_dict = {}
                    for item in sql_data:
                        sql_data_dict[item['when']] = item['value']

                    max_key = max(sql_data_dict, key=sql_data_dict.get)
                    max_val = sql_data_dict[max_key]

                    form_submits[service_name]['max_month'] = {
                        'date': max_key.date(),
                        'value': int(max_val)
                    }

            return form_submits

        def get_daemon_requests():
            today = date.today()
            yesterday = today - timedelta(days=1)
            fist_day_of_current_month = today.replace(day=1)

            form_submits = {}

            for service in settings.SERVICEMETRICS_SERVICES.keys():
                service_name = settings.SERVICEMETRICS_SERVICES[service]
                form_submits[service_name] = {}

                # сегодня
                sql_data = ServiceMetric.objects.filter(
                    when=today,
                    period='day',
                    group='daemon',
                    event__in=['cache_hit', 'cache_miss'],
                    service=service,
                )

                form_submits[service_name]['today'] = {}

                for item in sql_data:
                    form_submits[service_name]['today'][item.event] = int(item.value)

                form_submits[service_name]['today']['cache_requests'] = \
                    form_submits[service_name]['today'].get('cache_hit', 0) + \
                    form_submits[service_name]['today'].get('cache_miss', 0)

                form_submits[service_name]['today']['cache_miss_percent'] = self.percent(
                    form_submits[service_name]['today'].get('cache_miss', 0),
                    form_submits[service_name]['today']['cache_requests']
                )

                # вчера
                sql_data = ServiceMetric.objects.filter(
                    when=yesterday,
                    period='day',
                    group='daemon',
                    event__in=['cache_hit', 'cache_miss'],
                    service=service,
                )

                form_submits[service_name]['yesterday'] = {}

                for item in sql_data:
                    form_submits[service_name]['yesterday'][item.event] = int(item.value)

                form_submits[service_name]['yesterday']['cache_requests'] = \
                    form_submits[service_name]['yesterday'].get('cache_hit', 0) + \
                    form_submits[service_name]['yesterday'].get('cache_miss', 0)

                form_submits[service_name]['yesterday']['cache_miss_percent'] = self.percent(
                    form_submits[service_name]['yesterday'].get('cache_miss', 0),
                    form_submits[service_name]['yesterday']['cache_requests']
                )

                # за текущий месяц
                sql_data = ServiceMetric.objects.filter(
                    when=fist_day_of_current_month,
                    period='month',
                    group='daemon',
                    event__in=['cache_hit', 'cache_miss'],
                    service=service,
                )

                form_submits[service_name]['current_month'] = {}

                for item in sql_data:
                    form_submits[service_name]['current_month'][item.event] = int(item.value)

                form_submits[service_name]['current_month']['cache_requests'] = \
                    form_submits[service_name]['current_month'].get('cache_hit', 0) + \
                    form_submits[service_name]['current_month'].get('cache_miss', 0)

                form_submits[service_name]['current_month']['cache_miss_percent'] = self.percent(
                    form_submits[service_name]['current_month'].get('cache_miss', 0),
                    form_submits[service_name]['current_month']['cache_requests']
                )

                # Mах за день
                sql_data = ServiceMetric.objects.filter(
                    when__gte=fist_day_of_current_month,
                    period='day',
                    group='daemon',
                    event__in=['cache_hit', 'cache_miss'],
                    service=service,
                ).values('when', 'service').annotate(sum_val=Sum('value'))

                if sql_data:
                    sql_data_dict = {
                        'ticket': {},
                        'rasp': {},
                    }
                    for item in sql_data:
                        sql_data_dict[service][item['when']] = item['sum_val']

                    sql_data_service_dict = sql_data_dict[service]

                    max_key = max(sql_data_service_dict, key=sql_data_service_dict.get)
                    max_val = sql_data_service_dict[max_key]

                    form_submits[service_name]['max_day'] = {}
                    form_submits[service_name]['max_day']['cache_requests'] = {
                        'date': max_key.date(),
                        'value': int(max_val),
                    }

                # Mах за день (MISS)
                sql_data = ServiceMetric.objects.filter(
                    when__gte=fist_day_of_current_month,
                    period='day',
                    group='daemon',
                    event__in=['cache_miss'],
                    service=service,
                ).values('when', 'service').annotate(sum_val=Sum('value'))

                if sql_data:
                    sql_data_dict = {
                        'ticket': {},
                        'rasp': {},
                    }
                    for item in sql_data:
                        sql_data_dict[service][item['when']] = item['sum_val']

                    sql_data_service_dict = sql_data_dict[service]

                    max_key = max(sql_data_service_dict, key=sql_data_service_dict.get)
                    max_val = sql_data_service_dict[max_key]

                    form_submits[service_name]['max_day']['cache_miss'] = {
                        'date': max_key.date(),
                        'value': int(max_val),
                    }

                    form_submits[service_name]['max_day']['cache_miss_percent'] = {
                        'date': max_key.date(),
                        'value': self.percent(
                            form_submits[service_name]['max_day']['cache_miss']['value'],
                            form_submits[service_name]['max_day']['cache_requests']['value']
                        )
                    }

                # Mах за месяц (TOTAL)
                sql_data = ServiceMetric.objects.filter(
                    when__gte=fist_day_of_current_month,
                    period='month',
                    group='daemon',
                    event__in=['cache_hit', 'cache_miss'],
                    service=service,
                ).values('when', 'service').annotate(sum_val=Sum('value'))

                if sql_data:
                    sql_data_dict = {
                        'ticket': {},
                        'rasp': {},
                    }
                    for item in sql_data:
                        sql_data_dict[service][item['when']] = item['sum_val']

                    sql_data_service_dict = sql_data_dict[service]

                    max_key = max(sql_data_service_dict, key=sql_data_service_dict.get)
                    max_val = sql_data_service_dict[max_key]

                    form_submits[service_name]['max_month'] = {}
                    form_submits[service_name]['max_month']['cache_requests'] = {
                        'date': max_key.date(),
                        'value': int(max_val),
                    }

                # Mах за месяц (CACHE MISS)
                sql_data = ServiceMetric.objects.filter(
                    when__gte=fist_day_of_current_month,
                    period='month',
                    group='daemon',
                    event__in=['cache_miss'],
                    service=service,
                ).values('when', 'service').annotate(sum_val=Sum('value'))

                if sql_data:
                    sql_data_dict = {
                        'ticket': {},
                        'rasp': {},
                    }
                    for item in sql_data:
                        sql_data_dict[service][item['when']] = item['sum_val']

                    sql_data_service_dict = sql_data_dict[service]

                    max_key = max(sql_data_service_dict, key=sql_data_service_dict.get)
                    max_val = sql_data_service_dict[max_key]

                    form_submits[service_name]['max_month']['cache_miss'] = {
                        'date': max_key.date(),
                        'value': int(max_val),
                    }

                    form_submits[service_name]['max_month']['cache_miss_percent'] = {
                        'date': max_key.date(),
                        'value': self.percent(
                            form_submits[service_name]['max_month']['cache_miss']['value'],
                            form_submits[service_name]['max_month']['cache_requests']['value']
                        )
                    }

            return form_submits

        def get_users_search_relative_metric(metric):
            records = ServiceMetric.objects.filter(
                when__gte=date.today() - timedelta(hours=48),
                period='hour',
                group='form',
                event__in=[metric, 'user_search'],
                service__in=['ticket', 'rasp'],
            )

            records_dict = {}

            for record in records:
                if record.when not in records_dict:
                    records_dict[record.when] = {
                        'rasp': {
                            metric: 0,
                            'user_search': 0
                        },
                        'ticket': {
                            metric: 0,
                            'user_search': 0
                        },
                    }
                records_dict[record.when][record.service][record.event] += record.value

            for when, events in records_dict.items():
                for service, event in events.items():
                    records_dict[when][service] = (
                        float(event['user_search'] / float(event[metric]))
                        if event[metric]
                        else 0
                    )

            result_lists = {
                'rasp': list(),
                'ticket': list()
            }
            for key in sorted(records_dict.keys()):
                result_lists['rasp'].append([int(key.strftime('%s')) * 1000, records_dict[key]['rasp']])
                result_lists['ticket'].append([int(key.strftime('%s')) * 1000, records_dict[key]['ticket']])

            return result_lists

        context = {}

        context["form_submits"] = get_form_submits()
        context["daemon_requests"] = get_daemon_requests()
        context["yandex_uids"] = json.dumps(get_users_search_relative_metric('yandexuid_request'))
        context["ips"] = json.dumps(get_users_search_relative_metric('ip'))

        return render(request, 'servicestats_index.html', context)

    def get_urls(self):
        def wrap(view):
            def wrapper(*args, **kwargs):
                return self.admin_site.admin_view(view)(*args, **kwargs)
            return update_wrapper(wrapper, view)

        info = self.model._meta.app_label, self.model._meta.model_name

        my_urls = [
            url(r'^$', wrap(self.servicemetrics_table_view), name='%s_%s_changelist' % info)
        ]

        return my_urls

admin.site.register(ServiceMetric, ServiceMetricAdmin)


class FlexibleCacheAdmin(admin.ModelAdmin):
    def flexiblecache_table_view(self, request):
        context = {}

        flexible_context_30 = defaultdict(list)
        flexible_context_90 = defaultdict(list)
        flexible_context = defaultdict(list)

        left_border_30_days = datetime.now().date() - timedelta(days=30)
        left_border_90_days = datetime.now().date() - timedelta(days=90)

        for f in FlexibleCache.objects.all():
            item = [
                int(calendar.timegm(f.eventdate.timetuple())) * 1000,
                f.percent
            ]
            if f.eventdate >= left_border_30_days:
                flexible_context_30[str(f.national_version)].append(item)

            if f.eventdate >= left_border_90_days:
                flexible_context_90[str(f.national_version)].append(item)

            flexible_context[str(f.national_version)].append(item)

        context['flexible_30'] = dict(flexible_context_30)
        context['flexible_90'] = dict(flexible_context_90)
        context['flexible'] = dict(flexible_context)

        return render(request, 'flexiblestats_index.html', context)

    def get_urls(self):
        def wrap(view):
            def wrapper(*args, **kwargs):
                return self.admin_site.admin_view(view)(*args, **kwargs)
            return update_wrapper(wrapper, view)

        info = self.model._meta.app_label, self.model._meta.model_name

        my_urls = [
            url(r'^$', wrap(self.flexiblecache_table_view), name='%s_%s_changelist' % info)
        ]

        return my_urls


admin.site.register(FlexibleCache, FlexibleCacheAdmin)


class DynamicCacheAdmin(admin.ModelAdmin):
    def dynamic_table_view(self, request):
        context = {}
        dynamic_context = defaultdict(list)

        month_year_keys = [(datetime.now() + relativedelta(months=x)).strftime("%Y-%m") for x in range(6)]

        cache = defaultdict(list)

        for f in DynamicCache.objects.filter(month_year__in=month_year_keys):
            if f.national_version not in cache:
                cache[f.national_version] = defaultdict(list)

            cache[f.national_version][f.month_year].append(f.percent)

        for national_version in cache:
            for k, v in cache[national_version].items():
                year, month = k.split('-')
                month_year = date(int(year), int(month), 1)

                dynamic_context[str(national_version)].append([
                    int(calendar.timegm(datetime.combine(month_year, datetime.min.time()).timetuple())) * 1000,
                    int(float(sum(v)) / len(v))
                ])
            dynamic_context[str(national_version)] = sorted(dynamic_context[str(national_version)])

        context['dynamic'] = dict(dynamic_context)

        return render(request, 'stats/dynamicstats_index.html', context)

    def get_urls(self):
        def wrap(view):
            def wrapper(*args, **kwargs):
                return self.admin_site.admin_view(view)(*args, **kwargs)
            return update_wrapper(wrapper, view)

        info = self.model._meta.app_label, self.model._meta.model_name

        my_urls = [
            url(r'^$', wrap(self.dynamic_table_view), name='%s_%s_changelist' % info)
        ]

        return my_urls


admin.site.register(DynamicCache, DynamicCacheAdmin)


class MinPriceStatusAdmin(admin.ModelAdmin):
    list_display = ('national_version', 'eventdate', 'settlement_from', 'settlement_to', 'status')
    list_filter = ('national_version', 'status')
    search_fields = ('settlement_from__title',)

admin.site.register(MinPriceStatus, MinPriceStatusAdmin)
