# coding: utf-8

from django.conf import settings
from django.db import models
from django.db.models.fields import CharField
from django.utils.translation import ugettext_lazy as _

from common.models.geo import Settlement
from common.utils.fields import TrimmedCharField


if settings.ENVIRONMENT not in ['dev', 'testing', 'production']:
    raise EnvironmentError(u'Неверно указана среда выполнения')


class ImportChart(models.Model):
    partner = models.ForeignKey('order.Partner')
    when = models.DateTimeField(verbose_name=_(u'Время'), blank=False, null=False, db_index=True)
    event = CharField(_(u'Событие'), null=False, blank=False, max_length=80, db_index=True)
    machine = TrimmedCharField(_(u'Машина кластера'), max_length=40, default='', blank=True)
    value = models.FloatField(_(u'Значение'), default=0)

    def __unicode__(self):
        return u'%s - %s [%s]' % (self.when.strftime('%Y.%m.%d %H:%M:%S'),
                                  self.event, self.value)

    class Meta:
        unique_together = ('partner', 'when', 'event', 'machine')


class ServiceMetric(models.Model):
    when = models.DateTimeField(verbose_name=_(u'Время'), blank=False, null=False, db_index=True)
    period = models.CharField(
        max_length=len(max(settings.SERVICEMETRICS_PERIODS, key=len)),
        choices=[(p, p) for p in settings.SERVICEMETRICS_PERIODS],
        blank=False, null=False, verbose_name=_(u"период"))
    service = models.CharField(
        max_length=len(max(settings.SERVICEMETRICS_SERVICES.keys(), key=len)),
        choices=[(p, p) for p in settings.SERVICEMETRICS_SERVICES.keys()],
        blank=False, null=False, verbose_name=_(u"Сервис"))
    group = models.CharField(max_length=20, blank=False, null=False, verbose_name=_(u"Группа"))
    event = CharField(_(u'Событие'), null=False, blank=False, max_length=20)
    value = models.FloatField(_(u'Значение'), default=0)

    class Meta:
        unique_together = ('when', 'period', 'group', 'event', 'service')


class FlexibleCache(models.Model):
    NATIONAL_VERSIONS_CHOICES = tuple(
        [(v, v) for v in settings.AVIA_NATIONAL_VERSIONS]
    )

    eventdate = models.DateField(
        verbose_name=_(u"Дата"),
        blank=False, null=False, db_index=False
    )

    percent = models.PositiveSmallIntegerField(
        verbose_name=_(u"Процент попаданий"),
        blank=False, null=False, db_index=False
    )

    count = models.PositiveIntegerField(
        verbose_name=_(u"Кол-во"),
        blank=False, null=False, db_index=False
    )

    national_version = models.CharField(
        _(u'Нац. версия'),
        choices=NATIONAL_VERSIONS_CHOICES,
        blank=False,
        max_length=2,
        db_index=False
    )

    class Meta:
        unique_together = ('national_version', 'eventdate', 'percent')
        ordering = ['national_version', 'eventdate', 'percent', 'count']
        app_label = 'stats'


class DynamicCache(models.Model):
    NATIONAL_VERSIONS_CHOICES = tuple(
        [(v, v) for v in settings.AVIA_NATIONAL_VERSIONS]
    )

    eventdate = models.DateField(
        verbose_name=_(u"Дата"),
        blank=False, null=False
    )

    percent = models.PositiveSmallIntegerField(
        verbose_name=_(u"Процент попаданий"),
        blank=False, null=False
    )

    count = models.PositiveIntegerField(
        verbose_name=_(u"Кол-во"),
        blank=False, null=False
    )

    month_year = models.CharField(
        verbose_name=_(u"Месяц и год"),
        max_length=7,
        blank=False, null=False
    )

    national_version = models.CharField(
        _(u'Нац. версия'),
        choices=NATIONAL_VERSIONS_CHOICES,
        blank=False,
        max_length=2,
    )

    class Meta:
        unique_together = ('national_version', 'month_year', 'eventdate')
        ordering = ['national_version', 'month_year', 'eventdate', ]
        app_label = 'stats'


class MinPriceStatus(models.Model):
    NATIONAL_VERSIONS_CHOICES = tuple(
        [(v, v) for v in settings.AVIA_NATIONAL_VERSIONS]
    )

    STATUS_CHOICES = tuple(
        [(v, v) for v in ['In range', 'Lower', 'Greater']]
    )

    eventdate = models.DateField(
        verbose_name=_(u"Дата"),
        blank=False, null=False
    )

    national_version = models.CharField(
        _(u'Нац. версия'),
        choices=NATIONAL_VERSIONS_CHOICES,
        blank=False,
        max_length=2,
    )

    settlement_from = models.ForeignKey(
        Settlement,
        verbose_name=_(u'Город отправления'),
        related_name='+',
    )

    settlement_to = models.ForeignKey(
        Settlement,
        verbose_name=_(u'Город прибытия'),
        related_name='+',
    )

    status = models.CharField(
        _(u'Статус'),
        choices=STATUS_CHOICES,
        blank=False,
        max_length=8,
    )

    class Meta:
        unique_together = ('national_version', 'eventdate', 'settlement_from', 'settlement_to')
        ordering = ['national_version', 'eventdate', 'settlement_from', 'settlement_to']
        app_label = 'stats'
