# -*- coding: utf-8 -*-
import datetime
import json
from collections import Counter
from itertools import izip

from common.models.schedule import RTStation
from common.utils.date import human_duration
from travel.rasp.admin.timecorrection.correction import CorrectionFunctions
from travel.rasp.admin.timecorrection.duration_processing import ThreadDurationManipulation
from travel.rasp.admin.timecorrection.utils import n_wize
from travel.rasp.admin.www.models.geo import RoutePath

GRAPHICS_STEP = 5


class RTSCorrector(object):
    @classmethod
    def get_corrected_rtstations(cls, thread, correction_type=CorrectionFunctions.COMPARE):
        """
        Создает лист RTS c обновленным временем
        :param correction_type: Тип корректировки времени между станциями
        :type thread: common.models.schedule.RThread
        :return: RTStation list
        """
        correction_function = CorrectionFunctions(correction_type)
        diff_list = correction_function(thread)
        return list(cls.get_new_rts_from_difference_list(thread, diff_list))

    @staticmethod
    def get_new_rts_from_difference_list(thread, difference_list):
        for rts, diff in izip(thread.path, difference_list):
            diff = round(diff)
            yield RTStation(tz_arrival=None if rts.tz_arrival is None else rts.tz_arrival + diff,
                            tz_departure=None if rts.tz_departure is None else rts.tz_departure + diff,
                            time_zone=rts.time_zone,
                            thread_id=rts.thread_id,
                            station_id=rts.station_id,
                            is_fuzzy=bool(diff))


def get_geo_path_json(thread):
    return serialize_json(
        RoutePath.get(station_from, station_to).data for station_from, station_to in n_wize(thread.get_stations()))


def get_map_path_json(path_span_list):
    return serialize_json(
        path_span.map_data.json_path for path_span in path_span_list if hasattr(path_span, 'map_data'))


def serialize_json(json_path):
    try:
        return json.dumps([json.loads(p) for p in json_path if p])
    except TypeError:
        pass


def prepare_path_data(path, start_time_dt):
    """Подготавливает данные для отображения 'как в проде'
    :param start_time_dt: Время начала движения
    :type path: iterable RTStation
    """
    first_rts = path[0]
    naive_start_dt = datetime.datetime.combine(datetime.datetime.now().date(), start_time_dt)
    return [
        {
            'rtstation': rts,
            'arrival': rts.get_arrival_dt(naive_start_dt),
            'stop_time': rts.tz_departure - rts.tz_arrival if rts.tz_departure and rts.tz_arrival else None,
            'departure': rts.get_departure_dt(naive_start_dt),
            'in_trip': human_duration(
                ThreadDurationManipulation.calc_timedelta_from_start(first_rts, rts, naive_start_dt))
        } for rts in path]


def get_values_and_count(values_list):
    counted_value_list = Counter([(x // GRAPHICS_STEP) * GRAPHICS_STEP + GRAPHICS_STEP for x in values_list])
    return zip((0, 0), *sorted(counted_value_list.iteritems()))
