# -*- coding: utf-8 -*-
import re
import requests
import json
from collections import defaultdict
from time import sleep, time

from django.conf import settings

from common.utils.exceptions import SimpleUnicodeException
from travel.rasp.admin.timecorrection.utils import Constants
from travel.rasp.admin.timecorrection.models import MapData


class BaseDownloader(object):
    class BaseDownloaderError(SimpleUnicodeException):
        pass

    REQUEST_TIMEOUT = 60

    def __init__(self):
        self._request_counter = 0

    def _download_data(self, params):
        self._request_counter += 1
        try:
            return requests.get(self.url, params=params, timeout=self.REQUEST_TIMEOUT).text
        except requests.Timeout:
            raise self.BaseDownloaderError(u'Таймаут ответа')

    @property
    def url(self):
        raise NotImplementedError

    @property
    def total_requests(self):
        return self._request_counter


class MapsDataDownloader(BaseDownloader):
    class MapsDataDownloaderError(SimpleUnicodeException):
        pass

    ROUTE_MAPS_URL = defaultdict(
        lambda: u'http://route-net.int01e.tst.maps.yandex.ru/1.x/',
        production=u'http://route.maps.yandex.net/1.x/',
    )
    MAX_ALLOWED_RPS = 10

    def __init__(self):
        super(MapsDataDownloader, self).__init__()
        self.__last_request_time = 0

    def get_data(self, station_from, station_to):
        """
        :return: timecorrection.models.MapData
        """
        stations = (station_from, station_to)
        point_list = self.get_point_list(stations)
        self.__make_time_delay()
        yamapsml_data = self.get_data_from_maps(point_list)
        parse_data = self.parse_maps_data(yamapsml_data)
        return MapData(raw=yamapsml_data,
                       time=parse_data['time'],
                       distance=parse_data['length'],
                       json_path=parse_data['path'])

    def __make_time_delay(self):
        """Задержка для выполнения условия MAX RPS к маршрутизатору"""
        time_delta = time() - self.__last_request_time
        if time_delta < self.time_delay_between_requests:
            sleep(self.time_delay_between_requests - time_delta)
        self.__last_request_time = time()

    def get_data_from_maps(self, points):
        """Каждая точка задается долготой и широтой. Координаты точек разделяются символом «~».
            Пример: rll=37.614042,55.756685~37.628855,55.759101"""

        coordinates = u'~'.join(u'{},{}'.format(point[0], point[1]) for point in points)
        params = {
            'rll': coordinates,
            'origin': 'rasp',
        }

        return self._download_data(params=params)

    @classmethod
    def parse_maps_data(cls, yamapsml_data):
        """
        Парсер для YMapsML-документа
        формат файла https://doc.yandex-team.ru/maps/router/desc/concepts/get_response.xml
        :param yamapsml_data: YMapsML-документ
        :return: dict('time':value,'length':value)
        """
        data = dict()
        for data_type in ['time', 'length']:
            data_match = re.search('<r:{0}>(.*?)</r:{0}>'.format(data_type), yamapsml_data)

            if data_match:
                data[data_type] = data_match.group(1)
            else:
                raise cls.MapsDataDownloaderError(u'Поле {} не найдено'.format(data_type))

        # TODO преределать на отдельный метод
        path = re.findall('<gml:posList>(.*?)</gml:posList>', yamapsml_data)
        path = ' '.join(path)
        path = (float(x) for x in path.split())
        data['path'] = json.dumps(zip(path, path))

        try:
            if 'time' in data:
                data['time'] = round(float(data['time']) / Constants.SECONDS_IN_MINUTE, 2)
            if 'length' in data:
                data['length'] = round(float(data['length']) / Constants.METERS_IN_KM, 2)
        except (TypeError, ValueError):
            raise cls.MapsDataDownloaderError(u'Ошибка преобразования данных')

        return data

    @staticmethod
    def get_point_list(stations):
        return [(station.longitude, station.latitude) for station in stations]

    @property
    def time_delay_between_requests(self):
        return 1. / self.MAX_ALLOWED_RPS

    @property
    def url(self):
        config = getattr(settings, 'APPLIED_CONFIG', None)
        return self.ROUTE_MAPS_URL[config]


class GeoCoderDataDownloader(BaseDownloader):
    GEOCODE_URL = defaultdict(
        lambda: u'http://geocode-net.int01e.tst.maps.yandex.ru/1.x/',
        production=u'http://geocode.maps.yandex.net/1.x/',
    )

    def get_country_code(self, station):
        geo_data = self.get_data_from_geocoder(station.as_point())
        return self.parse_geo_data(geo_data)

    def get_data_from_geocoder(self, point):
        params = {
            'geocode': ','.join(str(c) for c in point),
            'sco': 'latlong',
            'origin': 'rasp',
        }
        return self._download_data(params=params)

    @staticmethod
    def parse_geo_data(data):
        try:
            return re.search('<CountryNameCode>([A-Z]*)</CountryNameCode>', data).group(1)
        except AttributeError:
            return None

    @property
    def url(self):
        config = getattr(settings, 'APPLIED_CONFIG', None)
        return self.GEOCODE_URL[config]
