# -*- coding: utf-8 -*-
import datetime
import hashlib

from django.db import models
from django.utils.translation import ugettext_lazy as _

from common.utils.date import human_duration
from travel.rasp.admin.timecorrection.utils import Constants, mean_speed_function


class MapData(models.Model):
    """Данные маршрутизатора"""
    pathspan = models.OneToOneField(
        'timecorrection.PathSpan',
        on_delete=models.CASCADE,
        null=True,
        blank=True,
        related_name='map_data',
    )
    raw = models.TextField(
        null=True,
        default=None,
        blank=True,
        editable=False,
    )
    time = models.FloatField(verbose_name=_(u'время пути в минутах'), default=None)
    distance = models.FloatField(verbose_name=_(u'расстояние между станциями в километрах'), default=None)
    json_path = models.TextField(verbose_name=_(u'координаты пути'),
                                 help_text=_(u'json-данные геокодирования маршрута'), null=True, blank=True)

    def __unicode__(self):
        return u'id %s время: %s расстояние %s' % (self.pk, self.time, self.distance)

    class Meta:
        app_label = 'timecorrection'
        verbose_name = _(u'Данные маршрутизатора')


class PathSpan(models.Model):
    """Характеристики участка пути между двумя автобусными стациями."""

    SPEED_DEVIATION = 30
    MAX_BUS_SPEED = 90

    station_from = models.ForeignKey(
        'www.Station',
        null=False,
        blank=False,
        verbose_name=_(u'от какой станции'),
        on_delete=models.CASCADE,
        related_name='pathspan_station_from'
    )
    station_to = models.ForeignKey(
        'www.Station',
        null=False,
        blank=False,
        verbose_name=_(u'до какой станции'),
        on_delete=models.CASCADE,
        related_name='pathspan_station_to'
    )
    duration = models.FloatField(verbose_name=_(u'время пути в минутах'), default=None)
    distance = models.FloatField(verbose_name=_(u'расстояние между станциями в километрах'), default=None)

    is_data_calculated = models.BooleanField(verbose_name=_(u'данные были рассчитаны вручную'), default=False,
                                             blank=True)

    is_one_country_path = models.BooleanField(default=False, blank=True)

    manual_max_drive_speed = models.FloatField(verbose_name=_(u'максимальная скорость движения в км/ч'), default=0)
    manual_min_drive_speed = models.FloatField(verbose_name=_(u'минимальная скорость движения в км/ч'), default=0)

    @property
    def max_drive_time(self):
        return round(Constants.MINUTES_IN_HOUR * self.distance / self.min_drive_speed, 2)

    @property
    def min_drive_time(self):
        return round(Constants.MINUTES_IN_HOUR * self.distance / self.max_drive_speed, 2)

    @property
    def min_drive_speed(self):
        if self.manual_min_drive_speed:
            return self.manual_min_drive_speed
        return self.mean_drive_speed - self.SPEED_DEVIATION

    @property
    def max_drive_speed(self):
        if self.manual_max_drive_speed:
            return self.manual_max_drive_speed
        return self.MAX_BUS_SPEED

    @property
    def mean_drive_speed(self):
        return round(mean_speed_function(self.distance), 2)

    def is_duration_need_correct(self, duration):
        return self.is_one_country_path and not self.min_drive_time <= duration <= self.max_drive_time

    def get_min_duration_to_valid(self, duration):
        """возвращает время которое необходимо добавить чтобы участок стал валидным
           Время отрицательное если участок необходимо уменьшить
        """
        if self.max_drive_time < duration:  # время больше максимального
            return self.max_drive_time - duration  # надо уменьшить
        else:
            return self.min_drive_time - duration  # надо увеличить

    class Meta:
        app_label = 'timecorrection'
        verbose_name = _(u'время пути между двумя станциями')
        verbose_name_plural = _(u'времена в пути между двумя станциями')

    def __unicode__(self):
        return u'%s -> %s время: %s расстояние %s' % (
            self.station_from.title, self.station_to.title, self.duration, self.distance)


# Временные таблицы для ускорения отображения

class PathSpanDataCache(models.Model):
    rtstation_from = models.ForeignKey(
        'www.RTStation',
        null=False,
        blank=False,
        verbose_name=_(u'от какой станции'),
        on_delete=models.CASCADE,
        related_name='pathspan_station_from'
    )

    rtstation_to = models.ForeignKey(
        'www.RTStation',
        null=False,
        blank=False,
        verbose_name=_(u'до какой станции'),
        on_delete=models.CASCADE,
        related_name='pathspan_station_to'
    )

    path_span = models.ForeignKey(
        'timecorrection.PathSpan',
        null=True,
        blank=True,
        on_delete=models.CASCADE,
        related_name='calculated_data'
    )

    stop_time = models.FloatField(verbose_name=_(u'время остановки на станции "от"'), default=None)
    time_shift = models.FloatField(verbose_name=_(u'время участка от поставщика'), default=None)
    line_distance = models.FloatField(verbose_name=_(u'расстояние по прямой'), default=None)
    line_with_div = models.FloatField(verbose_name=_(u'расстояние по прямой с дорожным дополнением'), default=None)
    geo_dist = models.FloatField(verbose_name=_(u'расстояние по гео админу'), null=True, blank=True)
    is_corrected = models.BooleanField(verbose_name=_(u'время участка было скорректировано'), default=False, blank=True,
                                       db_index=True)

    class Meta:
        app_label = 'timecorrection'


class ThreadCalcDataCache(models.Model):
    MIN_SPEED = 40 / Constants.MINUTES_IN_HOUR  # в минуту
    MAX_SPEED = 90 / Constants.MINUTES_IN_HOUR

    thread = models.OneToOneField(
        'www.RThread',
        null=False,
        blank=False,
        on_delete=models.CASCADE,
        related_name='calc_data'
    )

    permanent_path_data = models.ForeignKey(
        'timecorrection.PermanentPathData',
        null=True,
        blank=True,
        db_constraint=False,
        related_name='+',
    )

    corrected = models.BooleanField(verbose_name=_(u'нитка скорректирована'), default=False, blank=True, db_index=True)
    path_approved = models.BooleanField(verbose_name=_(u'нитка подтверждена'), default=False, blank=True, db_index=True)
    thread_distance = models.FloatField(verbose_name=_(u'длинна нитки'), default=0, blank=True, db_index=True)
    line_distance = models.FloatField(verbose_name=_(u'длинна нитки по прямой'), default=0, blank=True, db_index=True)
    duration_sum = models.FloatField(verbose_name=_(u'время нитки поставщика'), default=0, blank=True,
                                     db_index=True)
    duration_sum_correct = models.FloatField(verbose_name=_(u'время нитки корректора'), default=0, blank=True,
                                             db_index=True)
    correct_sum = models.FloatField(verbose_name=_(u'сумма по модулю всех корректировок'), default=0, blank=True,
                                    db_index=True)
    correct_percent = models.FloatField(verbose_name=_(u'процент корректировки'), default=0, blank=True, db_index=True)
    correct_to_valid_sum = models.FloatField(
        verbose_name=_(u'сумма минимального времени, на которое нужно изменить участки нитки, '
                       u'чтобы она стала валидной'), default=0, blank=True, db_index=True)
    number_of_invalid_parts = models.IntegerField(verbose_name=_(u'количество невалидных участков нитки'), default=0,
                                                  blank=True, db_index=True)

    @property
    def path_correct(self):
        if self.permanent_path_data_id is None:
            permanent_path_data = PermanentPathData.get_or_create_from_rtstations(self.thread.path)
            self.permanent_path_data = permanent_path_data
            self.save()
        return self.permanent_path_data.path_correct

    @path_correct.setter
    def path_correct(self, value):
        if self.permanent_path_data_id is None:
            permanent_path_data = PermanentPathData.get_or_create_from_rtstations(self.thread.path)
            self.permanent_path_data = permanent_path_data
        self.permanent_path_data.path_correct = value
        self.permanent_path_data.save()

    @property
    def human_duration_sum(self):
        return human_duration(datetime.timedelta(minutes=self.duration_sum))

    @property
    def human_duration_sum_correct(self):
        return human_duration(datetime.timedelta(minutes=self.duration_sum_correct))

    @property
    def is_duration_valid(self):
        if self.duration_sum == 0:
            return self.thread_distance == 0

        current_speed = self.thread_distance / float(self.duration_sum)
        return self.MIN_SPEED <= current_speed <= self.MAX_SPEED

    @property
    def ordered_correction_results(self):
        return self.correction_results.order_by('algorithm_type')

    @property
    def correct_to_valid_sum_percent(self):
        return round(100 * abs(self.correct_to_valid_sum) / float(self.duration_sum), 2)

    class Meta:
        app_label = 'timecorrection'


class CorrectionResult(models.Model):
    thread_data_cache = models.ForeignKey(
        'timecorrection.ThreadCalcDataCache',
        null=True,
        on_delete=models.CASCADE,
        related_name='correction_results'
    )

    algorithm_type = models.CharField(verbose_name=_(u'тип корректировки'), max_length=30, db_index=True)
    is_path_valid = models.BooleanField(verbose_name=_(u'нитка верна после корректировки'), default=False, blank=True,
                                        db_index=True)
    number_of_moved_parts = models.IntegerField(verbose_name=_(u'количество измененных участков нитки'), default=None,
                                                blank=True, db_index=True, null=True)

    @classmethod
    def algorithm_types(cls):
        if not hasattr(cls, '__algorithm_type_list') or not cls.__algorithm_type_list:
            cls.__algorithm_type_list = cls.objects.values_list('algorithm_type', flat=True).distinct().order_by(
                'algorithm_type')
        return cls.__algorithm_type_list

    class Meta:
        app_label = 'timecorrection'


class PermanentPathData(models.Model):
    """
    Данные маршрута независящие от времени поставщика.
    Записи не удаляются при удалении нитки.
    Поиск происходит по хеш-сумме всех координат станций расположенных в порядке следования.
    Чтобы избежать коллизии также сравниваются координаты пути.
    """

    stations_coordinates_hash = models.CharField(verbose_name=_(u'Хеш из координат станций нитки'),
                                                 max_length=40, db_index=True)
    station_coordinates_string = models.TextField(verbose_name=_(u'координаты станций'), default='')

    path_correct = models.BooleanField(verbose_name=_(u'нитка не вызывает подозрений'), default=True, blank=True)

    @classmethod
    def create_station_coordinates_string(cls, rtstations):
        """
        Возвращает строку из координат станций
        :type rtstations: iterable(common.models.schedule.RTStation)
        :return longitude0,latitude0;longitude1,latitude0; ...
        """
        return ';'.join('{},{}'.format(rts.station.longitude, rts.station.latitude) for rts in rtstations)

    @classmethod
    def make_hash_from_coordinates_string(cls, coordinates_string):
        """
        Возвращает хеш-сумму SHA1
        :param coordinates_string: string or buffer
        :return: string containing only hexadecimal digits
        """
        return hashlib.sha1(coordinates_string).hexdigest()

    @classmethod
    def get_from_rtstations(cls, rtstations):
        """
        Реализует метод доступа к данным модели.
        При отсутствии записи возбуждает исключение PermanentPathData.DoesNotExist
        :param rtstations: iterable(common.models.schedule.RTStation)
        :return: timecorrection.models.PermanentPathData
        """
        station_coordinates_string = cls.create_station_coordinates_string(rtstations=rtstations)
        coordinates_hash = cls.make_hash_from_coordinates_string(station_coordinates_string)
        return cls.objects.get(stations_coordinates_hash=coordinates_hash,
                               station_coordinates_string=station_coordinates_string)

    @classmethod
    def create_from_rtstations(cls, rtstations, **kwargs):
        """
        Реализует метод создания записи в модели.
        :type rtstations: iterable(common.models.schedule.RTStation)
        :return: timecorrection.models.PermanentPathData
        """
        station_coordinates_string = cls.create_station_coordinates_string(rtstations=rtstations)
        coordinates_hash = cls.make_hash_from_coordinates_string(station_coordinates_string)
        return cls.objects.create(station_coordinates_string=station_coordinates_string,
                                  stations_coordinates_hash=coordinates_hash, **kwargs)

    @classmethod
    def get_or_create_from_rtstations(cls, rtstations, defaults=None, update_data=False):
        """
        Удобный метод для поиска данных. В случае отсутствия запись создается.
        Параметр defaults используются только для сознания новой записи в БД. В поиске не участвует.
        :param rtstations: iterable(common.models.schedule.RTStation)
        :param defaults: {'foo': 'bar'}
        :param update_data: обновлять существующие записи значениями из defaults
        :return: timecorrection.models.PermanentPathData
        """
        station_coordinates_string = cls.create_station_coordinates_string(rtstations=rtstations)
        coordinates_hash = cls.make_hash_from_coordinates_string(station_coordinates_string)
        try:
            permanent_path_data = cls.objects.get(stations_coordinates_hash=coordinates_hash,
                                                  station_coordinates_string=station_coordinates_string)
        except cls.DoesNotExist:
            defaults = defaults or {}
            return cls.objects.create(station_coordinates_string=station_coordinates_string,
                                      stations_coordinates_hash=coordinates_hash, **defaults)

        if update_data and defaults:
            for key, value in defaults.iteritems():
                setattr(permanent_path_data, key, value)
            permanent_path_data.save()

        return permanent_path_data

    class Meta:
        app_label = 'timecorrection'
