# coding: utf-8

import json
import logging
import math
from itertools import izip

from django.core.exceptions import FieldDoesNotExist

from common.utils.exceptions import SimpleUnicodeException
from common.utils.geo import great_circle_distance_km, GeoPoint
from travel.rasp.admin.timecorrection.duration_processing import ThreadDurationManipulation
from travel.rasp.admin.timecorrection.path_checkers import is_local_path, is_path_approved, GeometryCheck
from travel.rasp.admin.timecorrection.models import PathSpan, ThreadCalcDataCache, PathSpanDataCache, PermanentPathData
from travel.rasp.admin.timecorrection.utils import Constants, mean_speed_function, n_wize
from travel.rasp.admin.timecorrection.data_downloaders import MapsDataDownloader
from travel.rasp.admin.www.models.geo import RoutePath


log = logging.getLogger(__name__)


class PathSpanDataPreparer(object):
    def __call__(self, station_from, station_to, map_data=None):
        """
        Подготавливает данные для записи в модель PathSpan
        :type station_from: common.models.geo.Station
        :type station_to: common.models.geo.Station
        :type map_data: timecorrection.models.MapData
        :return: path distance in km, path time in minutes
        """
        if map_data is not None and not all([map_data.distance, map_data.time]):
            map_data = None

        distance_by_line = self.distance_km_with_route_deviation(station_from, station_to)
        distance_by_geo_path = self.distance_by_path_or_none(station_from, station_to)

        result_distance = self._get_path_distance(map_data, distance_by_line, distance_by_geo_path)
        time_in_minutes = self._get_path_duration(map_data, result_distance, mean_speed_function)

        return round(result_distance, 2), round(time_in_minutes, 2)

    @classmethod
    def distance_km_with_route_deviation(cls, station_from, station_to):
        """:return расстояние по прямой + средняя криволинейность дорог =]"""
        geo_distance = great_circle_distance_km(station_from, station_to)
        return cls.add_route_deviation(geo_distance)

    @staticmethod
    def _get_path_duration(map_data, result_distance, speed_function):
        if map_data and result_distance == map_data.distance:
            return map_data.time
        elif map_data and abs(result_distance - map_data.distance) < 5:
            return result_distance * map_data.time / map_data.distance
        else:
            return Constants.MINUTES_IN_HOUR * result_distance / speed_function(result_distance)

    @staticmethod
    def _get_path_distance(map_data, distance_by_line, distance_by_geo_path):
        distance_array = []

        # TODO проверять что map.distance >= distance_by_line
        if map_data is not None:
            distance_array.append(map_data.distance)

        if distance_by_geo_path is not None:
            distance_array.append(distance_by_geo_path)

        if distance_array:
            return min(distance_array)

        return distance_by_line

    @staticmethod
    def add_route_deviation(geo_distance, k=0.21):
        """дополнительное расстояние вноситься на криволинейность дорог.
           По формуле Гюйгенса для длинны дуги.
           :param geo_distance: расстояние между двумя точками в километрах
           :param k: коэффициент для расчета высоты дуги
        """
        chord_height = geo_distance * k
        chord_length = math.sqrt(chord_height ** 2 + (geo_distance / 2.) ** 2)
        arc_length = 2 * chord_length + (2 * chord_length - geo_distance) / 3.
        return arc_length

    @staticmethod
    def get_geo_path_data_or_none(station_from, station_to):
        """Возвращает данные геокодирования маршрута между станциями station_from и station_to
        :return [[longitude0, latitude0], [longitude1, latitude1], ... ] or None
        """
        route_path = RoutePath.get(station_from, station_to)
        if route_path.status == RoutePath.STATUS_REMOVED:
            log.error(
                u'RoutePath not found. Station from id {} -> station to id {}.'.format(station_from.pk, station_to.pk))
        else:
            try:
                return json.loads(route_path.data)
            except TypeError:
                log.error(
                    u'RoutePath not found. Station from id {} -> station to id {}.'.format(station_from.pk,
                                                                                           station_to.pk))

    @classmethod
    def distance_by_path_or_none(cls, station_from, station_to):
        """расстояние по точкам в RoutePath"""
        geo_path_data = cls.get_geo_path_data_or_none(station_from, station_to)
        if geo_path_data is None:
            return
        geo_point_list = [GeoPoint(longitude=x[0], latitude=x[1]) for x in geo_path_data]
        distance_to_first_point = great_circle_distance_km(station_from, geo_point_list[0])
        distance_to_last_point = great_circle_distance_km(station_from, geo_point_list[-1])

        if distance_to_first_point > distance_to_last_point:
            distance_to_first_point = great_circle_distance_km(station_to, geo_point_list[0])
        else:
            distance_to_last_point = great_circle_distance_km(station_to, geo_point_list[-1])

        distance_between_point = sum(great_circle_distance_km(*point_pair) for point_pair in n_wize(geo_point_list, 2))
        return distance_to_first_point + distance_between_point + distance_to_last_point


class PathSpanProxy(object):
    def __init__(self):
        self._pathspan_data_preparer = PathSpanDataPreparer()
        self._data_downloader = None

    def get_pathspan(self, station_from, station_to, update=False):
        """
        Проверяет наличие PathSpan в БД. Если нет то создает  PathSpan и MapData
        :param update: указывает необходимость обновления существующих данных
        :param station_from:
        :param station_to:
        :rtype: timecorrection.models.PathSpan
        """
        try:
            path_span = PathSpan.objects.get(station_from=station_from, station_to=station_to)
            if not update:
                return path_span
        except PathSpan.DoesNotExist:
            path_span = PathSpan(station_from=station_from, station_to=station_to)

        map_data = self.get_maps_data_or_none(station_from, station_to)
        distance, moving_time = self._pathspan_data_preparer(station_from, station_to, map_data)
        local_path = is_local_path(station_from, station_to)

        path_span.duration = moving_time
        path_span.distance = distance
        path_span.is_one_country_path = local_path
        path_span.save()

        if map_data is not None:
            if hasattr(path_span, 'map_data'):
                path_span.map_data.delete()
            map_data.pathspan = path_span
            map_data.save()

        return path_span

    def get_maps_data_or_none(self, station_from, station_to):
        """
        :rtype: timecorrection.models.MapData
        :return MapData | None
        """
        if self._data_downloader is None:
            self._data_downloader = MapsDataDownloader()
        try:
            return self._data_downloader.get_data(station_from, station_to)
        except (MapsDataDownloader.MapsDataDownloaderError, TypeError, KeyError) as e:
            log.error(u'{}. station from id {} -> station to id {}.'.format(e.msg, station_from.pk, station_to.pk))

    @property
    def total_map_requests(self):
        return self._data_downloader.total_requests if self._data_downloader else 0

    @classmethod
    def get_pathspan_list(cls, rtstations):
        proxy = cls()
        return [proxy.get_pathspan(rts_from.station, rts_to.station) for rts_from, rts_to in n_wize(rtstations, 2)]


class ThreadCalcDataCacheProxy(object):
    class UpdateFieldError(SimpleUnicodeException):
        pass

    def __init__(self):
        self._path_span_cache_proxy = PathSpanDataCacheProxy()

    def get_thread_data_cache(self, thread):
        """
        Прокси ThreadCalcDataCache.
        :type thread: common.models.schedule.RThread
        """
        try:
            return ThreadCalcDataCache.objects.get(thread=thread)
        except ThreadCalcDataCache.DoesNotExist:
            pass

        stations = thread.get_stations()
        path_approved = is_path_approved(stations)
        path_correct = GeometryCheck.is_path_correct(stations)
        permanent_path_data = PermanentPathData.get_or_create_from_rtstations(rtstations=thread.path,
                                                                              defaults={'path_correct': path_correct})
        temp_thread_correct = ThreadCalcDataCache(thread=thread, path_approved=path_approved,
                                                  permanent_path_data=permanent_path_data)
        if path_approved and path_correct:
            self.add_data_to_thread_cache(temp_thread_correct)

        temp_thread_correct.save()
        return temp_thread_correct

    def add_data_to_thread_cache(self, thread_data_cache):
        """
        Заполняет поля класса ThreadCalcDataCache расчетными значениями
        Изменения необходимо сохранять отдельно
        :param thread_data_cache: timecorrection.models.ThreadCalcDataCache
        """
        correct_count = 0
        for rts_from, rts_to in n_wize(thread_data_cache.thread.path, 2):
            path_span_cache = self._path_span_cache_proxy.get_path_span_cache(rts_from=rts_from, rts_to=rts_to)

            if path_span_cache.is_corrected:
                thread_data_cache.correct_to_valid_sum += path_span_cache.path_span.get_min_duration_to_valid(
                    path_span_cache.time_shift)
                thread_data_cache.duration_sum_correct += path_span_cache.path_span.duration + path_span_cache.stop_time
                thread_data_cache.correct_sum += abs(path_span_cache.path_span.duration - path_span_cache.time_shift)
                correct_count += 1
            else:
                thread_data_cache.duration_sum_correct += path_span_cache.time_shift + path_span_cache.stop_time
            # TODO время должно быть без остановок
            thread_data_cache.duration_sum += path_span_cache.time_shift + path_span_cache.stop_time
            thread_data_cache.thread_distance += path_span_cache.path_span.distance
            thread_data_cache.line_distance += path_span_cache.line_distance

        thread_data_cache.corrected = bool(correct_count)
        thread_data_cache.number_of_invalid_parts = correct_count
        thread_data_cache.correct_percent = round(
            abs(100 * (1 - thread_data_cache.duration_sum_correct / thread_data_cache.duration_sum)), 2)

    @classmethod
    def _update_number_of_invalid_parts(cls, thread_data_cache):
        supplier_duration_list = ThreadDurationManipulation.get_supplier_duration_list(thread_data_cache.thread)
        path_span_cache_list = PathSpanDataCacheProxy.get_path_span_cache_list(thread_data_cache.thread)
        need_correction_flags = [path_span_cache.path_span.is_duration_need_correct(supplier_duration) for
                                 path_span_cache, supplier_duration
                                 in izip(path_span_cache_list, supplier_duration_list)]
        return sum(need_correction_flags)

    _update_functions = {
        'number_of_invalid_parts': _update_number_of_invalid_parts,
    }

    @classmethod
    def update_field(cls, thread_data_cache_qs, field):
        try:
            ThreadCalcDataCache._meta.get_field(field)
        except FieldDoesNotExist:
            raise cls.UpdateFieldError(
                u'поля {field} не существует в модели {model}'.format(field=field, model=ThreadCalcDataCache))

        if field in cls._update_functions:
            update_function = cls._update_functions[field]
        else:
            raise cls.UpdateFieldError(u'нет метода для обновления поля {field}'.format(field=field))

        for thread_data_cache in thread_data_cache_qs:
            if not thread_data_cache.path_approved or not thread_data_cache.path_correct:
                continue
            setattr(thread_data_cache, field, update_function(thread_data_cache))
            thread_data_cache.save()


class PathSpanDataCacheProxy(object):
    def __init__(self):
        self.thread_duration = ThreadDurationManipulation
        self._path_span_proxy = PathSpanProxy()

    def get_path_span_cache(self, rts_from, rts_to):
        """
        Прокси для PathSpanDataCache
        :type rts_to: common.models.schedule.RTStation
        :type rts_from: common.models.schedule.RTStation
        :rtype timecorrection.models.PathSpanDataCache
        """
        try:
            return PathSpanDataCache.objects.get(rtstation_from=rts_from, rtstation_to=rts_to)
        except PathSpanDataCache.DoesNotExist:
            pass

        path_span = self._path_span_proxy.get_pathspan(station_from=rts_from.station, station_to=rts_to.station)
        local_time_shift, stop_time = self.thread_duration(rts_from.thread).get_shift_and_stop_between_rts(rts_from,
                                                                                                           rts_to)
        corrected = path_span.is_duration_need_correct(local_time_shift)
        geo_dist = PathSpanDataPreparer.distance_by_path_or_none(rts_from.station, rts_to.station)

        line_distance = round(great_circle_distance_km(rts_from.station, rts_to.station), 2)
        line_with_div = PathSpanDataPreparer.distance_km_with_route_deviation(rts_from.station, rts_to.station)

        return PathSpanDataCache.objects.create(rtstation_from=rts_from,
                                                rtstation_to=rts_to,
                                                path_span=path_span,
                                                time_shift=local_time_shift,
                                                stop_time=stop_time,
                                                line_distance=line_distance,
                                                line_with_div=line_with_div,
                                                geo_dist=geo_dist,
                                                is_corrected=corrected)

    @classmethod
    def get_path_span_cache_list(cls, thread):
        proxy = cls()
        return [proxy.get_path_span_cache(rts_from, rts_to) for rts_from, rts_to in n_wize(thread.path)]
