# -*- coding: utf-8 -*-
import logging
import math

from common.utils.exceptions import SimpleUnicodeException
from common.utils.geo import great_circle_angular_distance
from travel.rasp.admin.timecorrection.data_downloaders import GeoCoderDataDownloader
from travel.rasp.admin.timecorrection.utils import n_wize
from travel.rasp.admin.www.models.geo import RoutePath

log = logging.getLogger(__name__)


class GeometryCheck(object):
    MIN_ANGLE_FOR_MIDDLE_STATION = 15
    MIN_ANGLE_FOR_LAST_STATION = 5
    MIN_ANGLE_BETWEEN_TWO_STATION_RAD = 0.000001

    class GeometryCheckError(SimpleUnicodeException):
        BAD_ANGLE_ID = 1
        DOUBLE_POINTS_ID = 2
        NONE_POINT_COORDINATE_ID = 3

        def __init__(self, msg, code):
            self.code = code
            self.msg = msg

    @classmethod
    def is_path_correct(cls, stations):
        try:
            cls.check_coord_not_none(stations)
            cls.check_angles_between_stations(stations)
        except cls.GeometryCheckError as e:
            log.error(e.msg)
            return False
        else:
            return True

    @classmethod
    def check_angles_between_stations(cls, stations):
        """
        Определяет для станций стоящих подряд угол меньше MIN_ANGLE_FOR_MIDDLE_STATION
        """
        if len(stations) < 3:
            return

        previous_angle = None

        for trio in n_wize(stations, n=3):
            angle_between_station = cls.get_angle_between_station(*trio)
            if max(angle_between_station, previous_angle) < cls.MIN_ANGLE_FOR_MIDDLE_STATION:
                raise cls.GeometryCheckError(u'возможно перепутаны точки {}'.format(trio[1].title),
                                             cls.GeometryCheckError.BAD_ANGLE_ID)
            previous_angle = angle_between_station
        else:
            if previous_angle < cls.MIN_ANGLE_FOR_LAST_STATION:
                raise cls.GeometryCheckError(u'возможно перепутаны точки {}'.format(stations[~0].title),
                                             cls.GeometryCheckError.BAD_ANGLE_ID)

    @classmethod
    def check_coord_not_none(cls, stations):
        for station in stations:
            if station.longitude is None or station.latitude is None:
                raise GeometryCheck.GeometryCheckError(u'у точки {} не заполнены координаты'.format(station.title),
                                                       cls.GeometryCheckError.NONE_POINT_COORDINATE_ID)

    @classmethod
    def get_angle_between_station(cls, s1, s2, s3):
        """
        Вычисляет угол между s1-s2 и s2-s3
        У всех станций должны быть атрибуты longitude, latitude и они не None.
        :param s1 s2 s3: Station
        :return угол в градусах
        """
        s1_s2_len = great_circle_angular_distance(s1, s2)
        s2_s3_len = great_circle_angular_distance(s2, s3)
        s1_s3_len = great_circle_angular_distance(s1, s3)

        # TODO добавить проверку s1_s3_len. для определения сильного отклонения станции от нитки
        # TODO добавить проверку станций расположенных на одной 'прямой' через сравнение углов или расстояний.
        if s1_s2_len < cls.MIN_ANGLE_BETWEEN_TWO_STATION_RAD or s2_s3_len < cls.MIN_ANGLE_BETWEEN_TWO_STATION_RAD:
            raise cls.GeometryCheckError(u'дублирование станций стоящий друг за другом',
                                         cls.GeometryCheckError.DOUBLE_POINTS_ID)

        angle_rad = cls.three_sides_given(s1_s2_len, s2_s3_len, s1_s3_len)

        return math.degrees(angle_rad)

    @staticmethod
    def three_sides_given(a, b, c):
        """
        Вычисляет угол между сторонами a,b сферического треугольника.
        угол определяются из сферической теоремы косинусов.
        :param a, b, c: стороны сферического треугольника в радианах
        :rtype: угол между сторонами a,b в радианах
        """
        numerator = math.cos(c) - math.cos(a) * math.cos(b)
        denominator = math.sin(a) * math.sin(b)

        angle = round(numerator / denominator, 6)

        return math.acos(angle)


def is_path_approved(stations):
    return all(x.status for x in RoutePath.get_route_paths(stations))


def is_local_path(first_station, second_station):
    if first_station.country_id is not None and second_station.country_id is not None:
        return first_station.country_id == second_station.country_id
    else:
        geo_coder = GeoCoderDataDownloader()
        return geo_coder.get_country_code(first_station) == geo_coder.get_country_code(second_station)


def is_path_in_single_tz(thread):
    return all(rts.time_zone == thread.time_zone for rts in thread.path)
