# coding: utf-8
import json

from django.contrib.auth.decorators import login_required
from django.http import HttpResponse, Http404
from django.shortcuts import get_object_or_404
from django.db.models import Count
from django.template.response import TemplateResponse

from common.models.schedule import RThread
from travel.rasp.admin.timecorrection.utils import undo_accumulate
from travel.rasp.admin.timecorrection.models import ThreadCalcDataCache
from travel.rasp.admin.timecorrection.models_data_access import PathSpanDataCacheProxy
from travel.rasp.admin.timecorrection.correction import CorrectionFunctions
from travel.rasp.admin.timecorrection.compare_and_processing import (get_geo_path_json, get_map_path_json, RTSCorrector,
                                                   prepare_path_data, get_values_and_count, GRAPHICS_STEP)


def test(request):
    return HttpResponse("test", content_type="text/plain")


@login_required
def index(request, supplier_id):
    context = {
        'corrected': ThreadCalcDataCache.objects.filter(permanent_path_data__path_correct=True, path_approved=True,
                                                        corrected=True, thread__supplier_id=supplier_id)
                                                .annotate(path_length=Count('thread__rtstation'))
                                                .filter(path_length__gte=3)
                                                .order_by('-correct_percent')[:300]
    }
    return TemplateResponse(request, 'timecorrection/index.html', context)


@login_required
def by_uid(request, uid):
    thread = get_object_or_404(RThread, uid=uid)

    correction_type = request.GET.get('diff') or CorrectionFunctions.select_correction_algorithm_for_view(thread)

    show_correction_types = request.GET.get('show_correction_types', False)

    if 'valid_flag' in request.POST and hasattr(thread, 'calc_data'):
        thread.calc_data.path_correct = not thread.calc_data.path_correct
        thread.calc_data.refresh_from_db()

    path_span_cache_list = PathSpanDataCacheProxy.get_path_span_cache_list(thread)

    context = {
        'thread': thread,
        'path_span_cache_list': path_span_cache_list,
        'geo_path': get_geo_path_json(thread),
        'map_path': get_map_path_json(path_span_cache.path_span for path_span_cache in path_span_cache_list),
        'show_correction_types': show_correction_types,
    }

    new_thread_path = RTSCorrector.get_corrected_rtstations(thread, correction_type=correction_type)
    context['time_table'] = prepare_path_data(thread.path, thread.tz_start_time)
    context['new_time_table'] = prepare_path_data(new_thread_path, thread.tz_start_time)

    context['diff_array'] = json.dumps(list(undo_accumulate(CorrectionFunctions(correction_type)(thread))))
    context['correction_function_type'] = correction_type
    context['correction_types'] = CorrectionFunctions.get_all_correction_function_types()

    return TemplateResponse(request, 'timecorrection/uid.html', context)


@login_required
def two_station(request, supplier_id):
    context = {
        'corrected': ThreadCalcDataCache.objects
                                        .filter(thread__supplier_id=supplier_id, path_approved=True,
                                                corrected=True)
                                        .annotate(path_length=Count('thread__rtstation'))
                                        .filter(path_length__lt=3)
                                        .order_by('-correct_percent'),
    }
    return TemplateResponse(request, 'timecorrection/index.html', context)


@login_required
def package(request, package_id):
    context = {
        'corrected': ThreadCalcDataCache.objects.filter(thread__route__two_stage_package_id=package_id,
                                                        path_approved=True, corrected=True,
                                                        permanent_path_data__path_correct=True)
                                                .order_by('-correct_percent'),
    }
    return TemplateResponse(request, 'timecorrection/index.html', context)


@login_required
def new(request):
    context = {'packages': list()}
    two_stage_package_list = ThreadCalcDataCache.objects.values_list('thread__route__two_stage_package',
                                                                     flat=True).distinct()
    for two_stage_package_id in two_stage_package_list:
        tread_qs = ThreadCalcDataCache.objects.filter(thread__route__two_stage_package=two_stage_package_id)
        data = {
            'title': str(tread_qs.first().thread.route.two_stage_package),
            'package_id': two_stage_package_id,
            'total_threads': tread_qs.count(),
            'need_to_correct': tread_qs.filter(corrected=True,
                                               permanent_path_data__path_correct=True,
                                               correct_to_valid_sum__gte=1).count(),
            'path_not_correct': tread_qs.filter(path_approved=True,
                                                permanent_path_data__path_correct=False).count(),
            'path_not_approved': tread_qs.filter(path_approved=False).count(),
        }
        data['good_thread'] = data['total_threads'] - data['need_to_correct'] - data['path_not_correct'] - data[
            'path_not_approved']
        context['packages'].append(data)

    return TemplateResponse(request, 'timecorrection/new_index.html', context)


@login_required
def new_package_view(request, package_id):
    thread_data_cache_qs = ThreadCalcDataCache.objects.filter(thread__route__two_stage_package_id=package_id)

    context = {
        'package_id': package_id,
        'title': str(thread_data_cache_qs.first().thread.route.two_stage_package),
        'total_threads': thread_data_cache_qs.count(),
        'need_to_correct': thread_data_cache_qs.filter(corrected=True,
                                                       permanent_path_data__path_correct=True).count(),
        'path_not_correct': thread_data_cache_qs.filter(path_approved=True,
                                                        permanent_path_data__path_correct=False).count(),
        'path_not_approved': thread_data_cache_qs.filter(path_approved=False).count(),
        'supplier_id': thread_data_cache_qs.first().thread.supplier_id,
    }

    correction_sum_list = thread_data_cache_qs.filter(corrected=True,
                                                      permanent_path_data__path_correct=True) \
                                              .values_list('correct_sum', flat=True)
    x_value, y_value = get_values_and_count(correction_sum_list)
    context['x_value'] = json.dumps(x_value)
    context['y_value'] = json.dumps(y_value)

    thread_distance_list = thread_data_cache_qs.filter(path_approved=True,
                                                       permanent_path_data__path_correct=True) \
                                               .values_list('thread_distance', flat=True)
    x_duration_value, y_duration_value = get_values_and_count(thread_distance_list)
    context['x_duration_value'] = json.dumps(x_duration_value)
    context['y_duration_value'] = json.dumps(y_duration_value)

    delta_to_correct_list = thread_data_cache_qs.filter(corrected=True,
                                                        permanent_path_data__path_correct=True) \
                                                .values_list('correct_to_valid_sum', flat=True)
    x_delta_to_correct, y_delta_to_correct = get_values_and_count(delta_to_correct_list)
    context['x_delta_to_correct'] = json.dumps(x_delta_to_correct)
    context['y_delta_to_correct'] = json.dumps(y_delta_to_correct)

    return TemplateResponse(request, 'timecorrection/new_package_view.html', context)


@login_required
def new_thread_list(request):
    if 'package_id' in request.GET:
        package_id = request.GET['package_id']
    else:
        raise Http404

    context = {'package_id': package_id}

    thread_data_cache_qs = ThreadCalcDataCache.objects.filter(thread__route__two_stage_package_id=package_id)

    if 'duration' in request.GET:
        duration = int(request.GET['duration'])
        thread_data_cache_qs = thread_data_cache_qs.filter(thread_distance__range=(duration - GRAPHICS_STEP, duration))

    if 'correction' in request.GET:
        correction = int(request.GET['correction'])
        thread_data_cache_qs = thread_data_cache_qs.filter(corrected=True, permanent_path_data__path_correct=True,
                                                           correct_sum__range=(correction - GRAPHICS_STEP, correction))
    if 'duration_to_valid' in request.GET:
        duration_to_valid = int(request.GET['duration_to_valid'])
        thread_data_cache_qs = thread_data_cache_qs.filter(corrected=True, permanent_path_data__path_correct=True,
                                                           correct_to_valid_sum__range=(
                                                               duration_to_valid - GRAPHICS_STEP, duration_to_valid))
    if 'is_valid' in request.GET:
        is_valid = int(request.GET['is_valid'])
        thread_data_cache_qs = thread_data_cache_qs.filter(permanent_path_data__path_correct=bool(is_valid))

    if 'is_corrected' in request.GET:
        is_corrected = int(request.GET['is_corrected'])
        thread_data_cache_qs = thread_data_cache_qs.filter(corrected=bool(is_corrected))

    if 'not_approved' in request.GET:
        thread_data_cache_qs = thread_data_cache_qs.filter(path_approved=False)

    context['thread_cache_qs'] = thread_data_cache_qs

    return TemplateResponse(request, 'timecorrection/new_thread_list.html', context)
