# coding: utf-8

from __future__ import unicode_literals, division

import codecs
import json
import os
from datetime import datetime
from functools import update_wrapper

import six
from django import forms
from django.conf import settings
from django.conf.urls import url
from django.contrib import admin, messages
from django.contrib.admin import SimpleListFilter
from django.contrib.admin.widgets import FilteredSelectMultiple, AdminBigIntegerFieldWidget
from django.contrib.auth.decorators import permission_required
from django.db import transaction, models
from django.db.models import Q
from django.forms import ModelForm, modelform_factory
from django.http import HttpResponse, HttpResponseRedirect
from django.http.response import HttpResponseBadRequest
from django.shortcuts import get_object_or_404, redirect
from django.template.response import TemplateResponse
from django.utils.decorators import method_decorator
from django.utils.html import format_html
from django.utils.safestring import mark_safe
from django.utils.translation import ugettext_lazy as _

from common.apps.facility.models import SuburbanThreadFacility
from common.models.geo import (
    CityMajority, StationMajority, Country, StationType, CompanyOffice, SettlementProperty, SettlementRelatedStations,
    SettlementNearest, StationProperty, StationPassage, StationPhone, CodeSystem, StationCode, StationExpressAlias,
    SuburbanZone, ExternalDirection, ExternalDirectionMarker, PointSynonym, PseudoRegion, ReplaceException,
    GortransCityLink, StationTerminal, Station2Settlement, WayToAirport,
    Region, Direction, DirectionMarker, Settlement, Station, DirectionTranslate, District, DirectionFromTranslate,
)
from common.models_utils.i18n import L_field
from common.models.schedule import (
    VehicleProducer, ExpressTypeLite, PlatformTranslation, AviaAlliance, RThreadType, Supplier, DeLuxeTrain, Company,
    RThread, RTStation, Route, TrainSchedulePlan, TrainTurnover, CompanyMarker, TrainPurchaseNumber,
    PlatformRepresentation
)

from common.models.special_tariffs import SpecialOffer
from common.models.tariffs import SuburbanTariff, AeroexTariff, TariffType, NotTariffTrain, ThreadTariff, Setting, \
    TariffGroup
from common.models.teasers import Teaser, Page
from common.models.transport import TransportType, TransportSubtype, TrainPseudoStationMap, TransportSubtypeColor, \
    TransportModel
from travel.rasp.library.python.common23.date import environment
from common.utils.date import RunMask
from common.utils.fields import UUID64Field
from common.models.pathfinder_maps import PathfinderMapsNearestSettlement
from travel.rasp.admin.lib.admin_options import (RaspExportModelAdmin, OrderingOnlyOnSmallQuerysetModelAdminMixin,
                               model_form_time_zone_mixin)
from travel.rasp.admin.lib.adminutils import get_rasp_admin_url
from mapping.generators.utils import INT_PRECISION
from mapping.models import RouteMapBlacklist
from geosearch.models import DefaultPoint
from travel.rasp.admin.www.models import Redirect
from travel.rasp.admin.www.models.geo import RoutePath, RegionWhiteList
from travel.rasp.admin.www.models.precalc import Suggest
from travel.rasp.admin.www.models.schedule import (TransportScheme, Route2Company, RouteImportInfo,
                                 ExpressNumber, ThreadSchedule, StationTeleportMap, FaceDirection)
from travel.rasp.admin.www.models.suggests import DefaultSuggest
from travel.rasp.admin.www.models.tariffs import TicketOffice
from www_extra.admin import CompanyExtraInlineAdmin
from www_extra.models import CompanyExtra


class CommentListFilter(SimpleListFilter):
    title = (_('Комментарий'))
    parameter_name = 'has_comment'

    def lookups(self, request, model_admin):
        return [
            ('y', _('Да')),
            ('n', _('Нет')),
        ]

    def queryset(self, request, queryset):

        if self.value():
            if self.value() == 'n':
                return queryset.filter(comment__exact='')
            elif self.value() == 'y':
                return queryset.exclude(comment__exact='')

        return queryset


class CityMajorityAdmin(RaspExportModelAdmin):
    fieldsets = (
        ('Profile field', {
            'fields': ('title',)
        }),
    )
    list_display = ('title',)

admin.site.register(CityMajority, CityMajorityAdmin)


class StationMajorityAdmin(RaspExportModelAdmin):
    fieldsets = (
        ('Profile field', {
            'fields': L_field.admin_fields(StationMajority, ['title'])
        }),
    )
    list_display = ('title_ru', 'code', 'id')

admin.site.register(StationMajority, StationMajorityAdmin)


class TransportSubtypeInline(admin.TabularInline):
    model = TransportSubtype
    extra = 1
    fieldsets = (
        ('Profile field', {
            'fields': [
                'code', 't_type', 'icon', 'color',
                L_field.admin_fields(TransportSubtype, [
                    'title', 'title_plural', 'title_partner_page', 'station_name', 'station_name_plural'
                ]),
                'comment'
            ]
        }),
    )


class TransportTypeAdmin(RaspExportModelAdmin):
    inlines = [TransportSubtypeInline]
    fieldsets = (
        ('Profiled field', {
            'fields': [
                'code',
            ]
        }),

        (_('Переводы'), {
            'fields': L_field.admin_fields(TransportType, ['title', 'title_plural', 'title_partner_page',
                                                           'station_name', 'station_name_plural'])
        }),
    )

    list_display = ('title_ru', 'code', 'station_name_plural_ru')

admin.site.register(TransportType, TransportTypeAdmin)


class VehicleProducerAdmin(RaspExportModelAdmin):
    fieldsets = (
        ('Profile field', {
            'fields': ('title', 't_type')
        }),
    )
    list_filter = ('t_type',)
    list_display = ('title', 't_type')
    search_fields = ('title',)

admin.site.register(VehicleProducer, VehicleProducerAdmin)


class TransportModelForm(forms.ModelForm):
    def clean(self):
        cleaned_data = super(TransportModelForm, self).clean()
        t_type = cleaned_data.get('t_type')
        t_subtype = cleaned_data.get('t_subtype')
        if t_type and t_subtype and t_subtype.t_type != t_type:
            raise forms.ValidationError(_('Подтипу транспорта {} соответствует тип транспорта {}')
                                        .format(t_subtype, t_subtype.t_type))
        return cleaned_data


class TransportModelAdmin(RaspExportModelAdmin):
    form = TransportModelForm

    fieldsets = (('Profile field', {'fields': (
        'title', 'title_en', 't_type', 't_subtype', 'producer', 'img',
        ('plane_body_type', 'is_propeller_flight'), 'code', 'code_en',
        'background', 'ttx', 'descr', 'template_page', 'is_cargo',
        'get_transportmodelmapping_urls',
    )}),)

    readonly_fields = ('get_transportmodelmapping_urls',)
    list_display = ('title', 'title_en', 'code', 'code_en',
                    'producer', 't_type', 'template_page')
    list_filter = ('t_type', 't_subtype', 'producer')
    search_fields = ('title', 'title_en', 'code', 'ttx', 'descr')

    def get_transportmodelmapping_urls(self, t_model):
        from travel.rasp.admin.importinfo.models.mappings import TransportModelMapping

        tmms = TransportModelMapping.objects.filter(t_model=t_model)

        data = [(get_rasp_admin_url(tmm, relative_only=True), six.text_type(tmm)) for tmm in tmms]

        urls = [format_html('<a href="{}">{}</a><br />', *d) for d in data]

        return ''.join(urls)

    get_transportmodelmapping_urls.allow_tags = True
    get_transportmodelmapping_urls.short_description = _('Ссылки на привязанные модели транспорта')

admin.site.register(TransportModel, TransportModelAdmin)


class ShowUrlOnDublicateGeoIdForm(forms.ModelForm):
    def clean__geo_id(self):
        geo_id = self.cleaned_data['_geo_id']

        if geo_id is None:
            return geo_id

        objs = self._meta.model.objects.filter(_geo_id=geo_id).exclude(pk=self.instance.pk)

        if objs:
            title = objs[0].L_title()
            url = get_rasp_admin_url(objs[0], relative_only=False)

            url = format_html('<a href="{url}" target="_blank">{title}</a>', url=url, title=title)

            model_name = self._meta.model._meta.verbose_name

            error = _('{model_name} с таким Geo ID уже есть в базе: {obj_url}.')\
                .format(obj_url=url, model_name=model_name.capitalize())

            raise forms.ValidationError(mark_safe(error))

        return geo_id


class CountryForm(ShowUrlOnDublicateGeoIdForm):
    class Meta:
        model = Country
        exclude = []


class CountryAdmin(RaspExportModelAdmin):
    form = CountryForm

    list_display = ('id', '_geo_id', 'title', 'title_en', 'code', 'code3', 'currency', 'language')
    search_fields = ('title', 'title_en')
    readonly_fields = ('modified_at',)

    fieldsets = (
        (_('Последнее обновление'), {'fields': ('modified_at',)}),
        (_('Название'), {
            'fields': [
                'title'
            ]
        }),

        (_('Переводы'), {
            'fields': list(L_field.admin_fields(Country, ['title']))
        }),

        (_('Коды'), {
            'fields': [
                '_geo_id', 'code', 'code3', 'domain_zone'
            ]
        }),

        (_('Другое'), {
            'fields': [
                'language', 'currency'
            ]
        }),
    )

admin.site.register(Country, CountryAdmin)


class RegionForm(ShowUrlOnDublicateGeoIdForm,
                 model_form_time_zone_mixin('time_zone', additonal_choices=[('', '!!!Не указана :(')])):
    class Meta:
        model = Region
        exclude = []


class RegionAdmin(RaspExportModelAdmin):
    form = RegionForm

    raw_id_fields = ('country',)
    list_display = ('id', '_geo_id', 'title', 'title_en', 'time_zone', 'country', 'hidden')
    search_fields = ('title', 'title_en')
    readonly_fields = ('modified_at', '_kladr_id')

    fieldsets = (
        (_('Последнее обновление'), {'fields': ('modified_at',)}),
        (_('Название'), {'fields': ('title',)}),
        (_('Переводы'), {'fields': list(L_field.admin_fields(Region, ['title']))}),
        (_('Основные'), {
            'fields': [
                'country', 'time_zone', 'hidden', 'disputed_territory'
            ]
        }),
        (_('Коды'), {
            'fields': [
                '_geo_id', '_kladr_id', 'koatuu', 'agent_geo_id'
            ]
        }),
        (_('Несогласовнные остановки'), {
            'fields': [
                'not_show_not_agreed', 'not_show_not_agreed_in_yabus', 'show_is_from_bus_station'
            ]
        }),
    )

admin.site.register(Region, RegionAdmin)


class SettlementPropertyAdmin(RaspExportModelAdmin):
    fieldsets = (
        ('Profile field', {
            'fields': ('title', 'description')
        }),
    )
    list_display = ('title', 'description')

admin.site.register(SettlementProperty, SettlementPropertyAdmin)


class SettlementRelatedStationsAdmin(RaspExportModelAdmin):
    raw_id_fields = ('station', 'settlement')

    fieldsets = (
        ('Profile field', {
            'fields': ('station', 'settlement')
        }),
    )
    list_display = ('station', 'settlement')
    search_fields = ('station__title', 'settlement__title')

admin.site.register(SettlementRelatedStations, SettlementRelatedStationsAdmin)


class SettlementNearestAdmin(RaspExportModelAdmin):
    raw_id_fields = ('settlement', 'nearest')
    list_display = ('settlement', 'nearest')

admin.site.register(SettlementNearest, SettlementNearestAdmin)


class SettlementNearestInline(admin.TabularInline):
    model = SettlementNearest
    fk_name = 'settlement'
    raw_id_fields = ('nearest',)
    extra = 1


class SettlementForm(ShowUrlOnDublicateGeoIdForm,
                     model_form_time_zone_mixin('time_zone', additonal_choices=[(u'', u'!!!Не указана :(')])):
    class Meta:
        model = Settlement
        exclude = []


class SettlementAdmin(RaspExportModelAdmin):
    form = SettlementForm

    raw_id_fields = ('country', 'region', 'district')

    fieldsets = (
        (_('Последнее обновление'), {'fields': ('modified_at',)}),
        (_('Локальные названия'), {
            'fields': [
                'title',
                'abbr_title',
            ]
        }),

        (_('Переводы'), {
            'fields': L_field.admin_fields(Settlement, ['title', 'abbr_title'])
        }),

        ('Profile field', {

            'fields': (
                'country', 'region', 'district', 'majority', 'suggest_order',
                'time_zone',
                'phone_info', 'phone_info_short',
                'big_city', 'hidden', 'has_urban_transport', 'has_tablo', '_geo_id', 'properties',
                'sirena_id', 'iata', 'agent_geo_id', 'koatuu', 'suburban_zone', 'type_choices',
            )
        }),

        (_('Электрички'), {'fields': ('use_in_suburban_app_suggests',)}),

        ('Additional', {'fields': ('human_url', '_disputed_territory', 'slug'), 'classes': ('collapse',)}),
        (_('Координаты'), {
            'classes': ('ymaps-point',),
            'fields': ('longitude', 'latitude'),
        }),
    )
    readonly_fields = ('type_choices', 'modified_at')
    list_display = ('id', '_geo_id', 'title', 'title_en', 'majority', 'suggest_order', 'country',
                    'region', 'time_zone', 'sirena_id', 'big_city',)
    list_display_links = ('title',)
    list_filter = ('majority', 'big_city')
    search_fields = ('title', 'title_en', 'sirena_id', 'iata')
    filter_horizontal = ('properties',)
    inlines = (SettlementNearestInline,)

admin.site.register(Settlement, SettlementAdmin)


class StationPropertyAdmin(RaspExportModelAdmin):
    fieldsets = (
        ('Profile field', {
            'fields': ('title', 'description',)
        }),
    )
    list_display = ('title',)

admin.site.register(StationProperty, StationPropertyAdmin)


class StationTypeAdmin(RaspExportModelAdmin):
    list_display = ('name_ru', 'prefix_ru')


admin.site.register(StationType, StationTypeAdmin)


class StationCodeInline(admin.TabularInline):
    model = StationCode


class StationTerminalInline(admin.TabularInline):
    model = StationTerminal


class StationPhoneInline(admin.TabularInline):
    model = StationPhone


StationTimezoneFormMixIn = model_form_time_zone_mixin('time_zone', additonal_choices=[('', '!!!Не указана :(')],
                                                      required=True)


# https://st.yandex-team.ru/RASPFRONT-7917#5ed3f6ab6cb06a7bf9fa766a
# Виджет, который отображает значения через repr
class ReprInput(forms.NumberInput):
    def format_value(self, value):
        if isinstance(value, float):
            value = repr(value)
        return super(ReprInput, self).format_value(value)


class StationForm(StationTimezoneFormMixIn):
    class Meta:
        model = Station
        exclude = []
        widgets = {'longitude': ReprInput, 'latitude': ReprInput}


class StationAdmin(RaspExportModelAdmin):
    form = StationForm
    raw_id_fields = ('settlement', 'region', 'country', 'district')
    inlines = [StationTerminalInline, StationCodeInline, StationPhoneInline]

    fieldsets = (
        (_('Последнее обновление'), {'fields': ('modified_at',)}),
        (_('Локальные названия'), {
            'fields': [
                'title',
                'popular_title',
                'short_title'
            ]
        }),
        (_('Переводы'), {
            'fields': L_field.admin_fields(Station, ['title', 'popular_title', 'short_title'], group_by='lang')
        }),
        (_('Расположение'), {
            'fields': ('country', 'region', 'settlement', 'district', 'time_zone', 'time_zone_not_check')
        }),
        (_('Адрес'), {
            'fields': L_field.admin_fields(Station, ['address'], show_local_field=True)
        }),
        (_('Тип'), {
            'fields': ('t_type', 'station_type')
        }),
        (_('Базовый автовокзал'), {
            'fields': ('is_base', 'is_base_modified_at')
        }),
        (_('Нечеткие признаки'), {
            'fields': ('is_fuzzy', 'is_searchable_from', 'is_searchable_to',
                       'in_station_schedule', 'in_thread')
        }),
        (_('Признаки для опозданий электричек'), {
            'fields': ('use_in_departure_forecast', 'use_in_forecast')
        }),
        (_('Как добраться до города'), {
            'fields': L_field.admin_fields(Station, ['how_to_get_to_city'])
        }),
        (_('Не сортированные настройки'), {
            'fields': (
                'hidden', 'use_direction',
                'majority',
                'show_settlement', 'site_url',
                'show_mode',
                'not_generalize', 'suburban_zone', 'has_aeroexpress', 'near_metro',
                'map_zoom', 'tablo_state', 'tablo_state_prev', 'virtual_end', 'type_choices',
                'show_tablo_stat', 'slug', 'agreed_stop_type'
            )
        }),
        (_('Панорама и схема'), {
            'fields': ('photo', 'panorama_url', 'schema_image'),
        }),
        (_('Координаты'), {
            'classes': ('ymaps-point',),
            'fields': ('longitude', 'latitude'),
        }),
        ('Meta', {
            'fields': ('meta_title', 'meta_description')
        })
    )
    list_display = ('id', 'title', 'station_type', 'majority', 'full_path_with_links', 't_type',
                    'express_id', 'sirena_id', 'hidden')
    list_filter = ('majority', 't_type', 'station_type', 'suburban_zone',)
    search_fields = ('title', 'title_en', 'code_set__code')
    filter_horizontal = ('properties',)
    ordering = ('-id',)
    readonly_fields = ('type_choices', 'tablo_state_prev', 'modified_at', 'is_base_modified_at')

    def full_path_with_links(self, station):
        return self.station_full_path(station)

    full_path_with_links.allow_tags = True
    full_path_with_links.short_description = _('Путь к станции')

    @classmethod
    def station_full_path(cls, station, include_station=False):
        parts = filter(None, [
            station,
            station.settlement,
            station.district or (station.settlement.district if station.settlement is not None else None),
            station.region,
            station.country
        ])

        if not include_station:
            parts = parts[1:]

        def make_url(object, bold=False):
            if bold:
                return format_html(
                    '<a href="{}" style="font-size: 110%%; font-weight: bold;">{}</a>',
                    get_rasp_admin_url(object, relative_only=True), object.title
                )
            return format_html('<a href="{}">{}</a>', get_rasp_admin_url(object, relative_only=True), object.title)

        if not parts:
            return ''

        parts = [make_url(parts[0], True)] + map(make_url, parts[1:])
        return ' &lt; '.join(parts)

    def change_view(self, request, object_id, form_url='', extra_context=None):
        import locale

        extra_context = extra_context or {}
        extra_context['MODEL_LANGUAGES'] = {}
        for lang in settings.MODEL_LANGUAGES:
            extra_context['MODEL_LANGUAGES'][lang] = locale.normalize(lang).split('.')[0].replace('_', '-')

        return super(StationAdmin, self).change_view(request, object_id,
                                                     form_url, extra_context=extra_context)

    def save_model(self, request, obj, form, change):
        if obj.id:
            old_station = Station.objects.get(pk=obj.id)
            if old_station.is_base != obj.is_base:
                obj.is_base_modified_at = environment.now()
        else:
            obj.is_base_modified_at = environment.now()
        obj.save()


admin.site.register(Station, StationAdmin)


class PointSynonymAdmin(RaspExportModelAdmin):
    list_display = ('title', 'point', 'language')
    search_fields = ('title',)

admin.site.register(PointSynonym, PointSynonymAdmin)


class Station2SettlementAdmin(RaspExportModelAdmin):
    raw_id_fields = ('station', 'settlement')
    fieldsets = (
        ('Profile field', {
            'fields': ('station', 'settlement')
        }),
    )
    list_display = ('station', 'settlement')
    search_fields = ('station__title', 'settlement__title')

admin.site.register(Station2Settlement, Station2SettlementAdmin)


class CompanyMarkerInline(admin.TabularInline):
    model = CompanyMarker
    raw_id_fields = ('station', )
    extra = 0


class CompanyMarkerAdmin(RaspExportModelAdmin):
    list_filter = ('company',)
    list_display = ('company', 'station')
    raw_id_fields = ('company', 'station')

admin.site.register(CompanyMarker, CompanyMarkerAdmin)


class PlatformPresentationAdmin(RaspExportModelAdmin):
    list_display = ('station', 'reg_exp', 'representation')
    raw_id_fields = ('station',)


admin.site.register(PlatformRepresentation, PlatformPresentationAdmin)


@admin.register(Company)
class CompanyAdmin(RaspExportModelAdmin):
    raw_id_fields = ('home_station', 'country')
    fieldsets = (
        (_('Локальные названия'), {
            'fields': [
                'title',
                'short_title'
            ]
        }),
        (_('Переводы'), {
            'fields': L_field.admin_fields(Company, ['title', 'short_title'])
        }),
        (_('Коды'), {
            'fields': ('sirena_id', 'iata', 'icao', 'icao_ru', 'tis_code', 'express_code', 'yandex_avia_code')
        }),
        (_('Лого'), {
            'fields': ('logo', 'icon', 'logo_mono', 'logo_bgcolor', 'logo_complex', 'svg_logo')
        }),
        ('Profile field', {
            'fields': ('address', 't_type', 'country',
                       'home_station', 'url', 'priority', 'email',
                       'contact_info', 'phone', 'phone_booking', 'description', 'hidden', 'strange', 'is_freight',
                       'meta_title', 'meta_description')
        }),
        (_('Контактные данные для регистрации'), {
            'fields': [
                'registration_url',
                'registration_phone',
            ]
        }),
        (_('Переводы контактных данных для регистрации'), {
            'fields': L_field.admin_fields(Company, [
                'registration_url',
                'registration_phone',
            ])
        }),
    )
    list_display = ('title', 'short_title', 'title_en', 'address', 'country',
                    'home_station', 'sirena_id', 'iata', 'icao', 'url')
    list_filter = ('t_type', 'extra__allow_sirena_import')
    search_fields = ('title', 'title_en', 'address', 'url', 'iata',
                     'sirena_id', 'icao', 'contact_info', 'phone', 'phone_booking', 'description')
    readonly_fields = ('logo_complex',)
    inlines = [CompanyExtraInlineAdmin, CompanyMarkerInline]
    ordering = ('title',)

    def logo_complex(self, obj):
        if obj.logo_mono:
            return format_html('<img src="{}" style="background-color: {}"/>',
                               obj.logo_mono.url, obj.logo_bgcolor or '#000000')
        else:
            return ''

    logo_complex.allow_tags = True
    logo_complex.short_description = _('Предпросмотр составного логотипа')

    def get_urls(self):
        urlpatterns = super(CompanyAdmin, self).get_urls()

        urlpatterns = [
            url(r'^(.+)/remove-routes/$',
                self.admin_site.admin_view(self.remove_routes),
                name='{}_{}_remove_routes'.format(self.model._meta.app_label, self.model._meta.model_name)),
        ] + urlpatterns

        return urlpatterns

    @method_decorator(permission_required('www.can_delete_company_routes'))
    @transaction.atomic
    def remove_routes(self, request, company_id):
        from travel.rasp.admin.www.utils.mysql import fast_delete_threads

        company = get_object_or_404(Company, pk=company_id)

        if request.method == 'POST':
            if not request.POST.get('post') == 'yes':
                return HttpResponseBadRequest('')

            threads_qs = RThread.objects.filter(route__script_protected=False, company=company)
            affected_route_ids = {t.route_id for t in threads_qs}
            threads_count = threads_qs.count()
            fast_delete_threads(RThread.objects.filter(route__script_protected=False, company=company))

            routes_qs = Route.objects.filter(script_protected=False, rthread__isnull=True,
                                             pk__in=affected_route_ids)
            route_count = routes_qs.count()
            routes_qs.delete()

            messages.success(request,
                             'Успешно удалили {threads_count} ниток и {route_count} маршрутов'.format(**locals()))

            return redirect('admin:www_company_change', company.id)

        return TemplateResponse(request, 'admin/www/company/remove_routes_confirmation.html', {
            'title': 'Удаление рейсов компании {}'.format(company.L_title()),
            'object': company,
            'opts': Company._meta,
            'rthread_count': RThread.objects.filter(company=company).count()
        })

    def save_related(self, request, form, formsets, change):
        super(CompanyAdmin, self).save_related(request, form, formsets, change)
        CompanyExtra.objects.get_or_create(company=form.instance)


class AllianceCompanyInline(admin.TabularInline):
    """ Нужен для отображения партнёров внутри альянса """

    model = Company
    extra = 0
    can_delete = False
    fields = ('title', 'iata', 'hidden')
    readonly_fields = ('title', 'iata', 'hidden')

    def has_add_permission(self, request):
        return False


class AviaAllianceAdmin(admin.ModelAdmin):
    list_display = ('title', 'enabled')
    list_filter = ('enabled',)
    fieldsets = (
        (_('Основное'), {'fields': [
            'title',
            'description',
            'logo_svg',
            'enabled',
        ]}),
        (_('Переводы'), {'fields': L_field.admin_fields(AviaAlliance, ['title', 'description'])})
    )
    inlines = [AllianceCompanyInline]

admin.site.register(AviaAlliance, AviaAllianceAdmin)


class TransportSchemeAdmin(RaspExportModelAdmin):
    raw_id_fields = ('transport',)
    fieldsets = (
        ('Profile field', {
            'fields': ('img', 'title', 'company', 'transport', 'transport_img')
        }),
    )
    list_display = ('img', 'title', 'company', 'transport', 'transport_img')
    search_fields = ('title', 'company')

admin.site.register(TransportScheme, TransportSchemeAdmin)


class RouteAdmin(RaspExportModelAdmin):
    def mark_for_recount(self, request, queryset):
        RThread.objects.filter(route__in=queryset).update(changed=True)
        messages.add_message(request, messages.SUCCESS, _('Маршруты помечены как измененные'))
    mark_for_recount.short_description = 'Mark as changed'

    actions = ['mark_for_recount']

    fieldsets = (
        (_('Последнее обновление'), {'fields': ('modified_at',)}),
        (_('Параметры'), {
            'fields': (
                'route_uid',
                'route_number',
                'route_title',
                't_type',
                'script_protected',
                'comment',
                'hidden',
                'style',
                'red_metaroute_link'
            )
        }),
    )
    list_display = ('route_uid', 'route_number', 'route_title', 't_type', 'supplier')
    list_display_links = ('route_uid',)
    list_filter = ('t_type', CommentListFilter, 'supplier')
    search_fields = ('route_uid', 'rthread__title', 'rthread__number')
    readonly_fields = ('red_metaroute_link', 'route_uid', 'route_title', 'route_number', 'modified_at')

    def red_metaroute_link(self, obj):
        if not obj.red_metaroute:
            return ''
        return format_html('<a href="{}" target="_blank">{}: {}</a>',
                           obj.red_metaroute.redadmin_link,
                           obj.red_metaroute.id,
                           obj.red_metaroute.full_name)
    red_metaroute_link.short_description = _('Рейс красного админа')
    red_metaroute_link.allow_tags = True

    def route_title(self, obj):
        return obj.route_title
    route_title.short_description = _('Название')

    def route_number(self, obj):
        return obj.route_number
    route_number.short_description = _('Номер')


admin.site.register(Route, RouteAdmin)


class SuburbanThreadFacilityInlineAdmin(admin.StackedInline):
    model = SuburbanThreadFacility
    min_num = 0
    extra = 0
    show_change_link = True
    template = 'admin/edit_inline/stacked_collapsed.html'


class RThreadTypeAdmin(RaspExportModelAdmin):
    list_display = ('title', 'code')

admin.site.register(RThreadType, RThreadTypeAdmin)


class RThreadAdminForm(forms.ModelForm):
    class Meta:
        model = RThread
        exclude = []

    def clean_template_code(self):
        self.cleaned_data['template_code'] = (self.cleaned_data['template_code'] or u'').strip()

        if self.cleaned_data['template_code']:
            from travel.rasp.admin.scripts.schedule.utils.afmasktext import (AFTextBuilder, AFTextBuildError,
                                                           AFTextMatchError)

            try:
                AFTextBuilder().build_range_day_texts(
                    self.cleaned_data['template_code'].split(u'#')[-1]
                )
            except (AFTextBuildError, AFTextMatchError) as e:
                raise forms.ValidationError(
                    _('Ошибка разбора шабона: "{template}": {error}').format(
                        template=self.cleaned_data['template_code'], error=six.text_type(e)
                    )
                )

        return self.cleaned_data['template_code'].strip()

    def clean_t_subtype(self):
        t_subtype = self.cleaned_data['t_subtype']
        t_type = self.cleaned_data['t_type']
        if t_subtype and t_type != t_subtype.t_type:
            raise forms.ValidationError(
                _('Неверный подтип транспорта "{}" для типа транспорта "{}"').format(t_subtype, t_type))

        return t_subtype


@admin.register(RThread)
class RThreadAdmin(OrderingOnlyOnSmallQuerysetModelAdminMixin, RaspExportModelAdmin):
    form = RThreadAdminForm
    raw_id_fields = ('route', 'company', 't_model', 'basic_thread')
    readonly_fields = ('translated_manual_days_texts', 'pseudo_data', 'modified_at')
    fieldsets = (
        (_('Последнее обновление'), {'fields': ('modified_at',)}),
        (_('Названия'), {
            'fields': (
                'title',
                'title_short',
                'title_tr',
                'title_uk',
                'is_manual_title'
            )
        }),
        (_('Основные параметры'), {
            'fields': (
                'route',
                't_type',
                't_subtype',
                'number',
                'hidden_number',
                'ordinal_number',
                'old_thread_number',
                'uid',
                'canonical_uid',
                'type',
                'tariff_type',
                'company',
                't_model',
                'basic_thread',
                'express_type',
                'express_lite',
                'schedule_plan',
                'supplier',
                'changed',
                'path_and_time_unchanged',
                'is_circular',
                'is_combined',
                'hidden',
            )
        }),
        (_('Время и дни хождения'), {
            'fields': (
                'tz_start_time',
                'time_zone',
                'begin_time',
                'end_time',
                'period_int',
                'density',
                'comment',
                'pseudo_data',
                'year_days'
            )
        }),
        (_('Тексты дней хождений'), {
            'fields': ('template_code', 'template_start', 'template_end', 'template_text')
        })
    )
    list_display = ('number', 'title', 'time_zone', 'tz_start_time', 'get_type', 'get_model', 'uid')
    list_display_links = ('number', 'title', 'uid')
    list_filter = ('type', 't_type', CommentListFilter, 'supplier', 'schedule_plan')
    search_fields = ('number', 'uid', 'title')
    ordering = ('number', 'title')
    actions = ['restore_changes', 'mark_as_changed']
    inlines = [SuburbanThreadFacilityInlineAdmin]

    def restore_changes(self, request, queryset):
        basic = RThreadType.objects.get(code='basic')
        for thread in queryset:
            if thread.type == basic:
                thread_mask = RunMask(thread.year_days)
                for date_ in thread.get_change_date_list():
                    thread_mask[date_] = True
                for change in thread.thread_changes.all():
                    change.delete()
                thread.year_days = str(thread_mask)
                thread.changed = True
                thread.save()
        messages.add_message(request, messages.SUCCESS, _('Изменения и отмены ниток были восстановлены.'))

    restore_changes.short_description = _('Восстановить изменения')

    def mark_as_changed(self, request, queryset):
        queryset.update(changed=True)
        messages.add_message(request, messages.SUCCESS, _('Нитки помечены как измененные'))

    mark_as_changed.short_description = _('Пометить как изменённые')

    def extrapolate_and_calc_cysix_template_texts(self, thread):
        from cysix.two_stage_import.factory import CysixTSIFactory

        if not thread.route_id:
            return

        if not thread.route.two_stage_package_id:
            return

        tsi_package = thread.route.two_stage_package

        factory = tsi_package.get_two_stage_factory()
        if not factory or not isinstance(factory, CysixTSIFactory):
            # Пакет не является общим
            return

        filter_ = factory.get_filter('text_schedule_and_extrapolation', max_forward_days=factory.max_forward_days)

        if filter_ and filter_.enabled:
            filter_.apply(thread, save=False, extrapolate=False)

    def save_model(self, request, obj, form, change):
        from travel.rasp.admin.scripts.schedule.utils import afmasktext

        if obj.id:
            old_thread = RThread.objects.get(pk=obj.id)
            if old_thread.year_days != obj.year_days:
                self.extrapolate_and_calc_cysix_template_texts(obj)

        obj.template_code = obj.template_code.strip()

        obj.translated_manual_days_texts = ''

        if obj.template_code:
            if '#' in obj.template_code:
                tz, obj.template_code = obj.template_code.split('#')
                tz = tz or 'local'

                texts = afmasktext.shift_and_build_days_texts(obj, template_timezone=tz)
            else:
                texts = afmasktext.shift_and_build_days_texts(obj)

            if texts:
                obj.translated_manual_days_texts = json.dumps(texts)

        obj.save()

        # RASP-3317
        for station_id in request.POST.getlist('station'):
            station_admin = StationAdmin(Station, self.admin_site)
            station = Station._default_manager.get(id=station_id)

            approx = request.POST['station_%s_approx' % station_id]

            if not approx:
                try:
                    lat = float(request.POST['station_%s_lat' % station_id])
                    lng = float(request.POST['station_%s_lng' % station_id])
                except ValueError:
                    continue

                old_lat = station.latitude
                old_lng = station.longitude

                if abs(old_lat - lat) * INT_PRECISION > 1 or abs(old_lng - lng)*INT_PRECISION > 1:
                    station.latitude = lat
                    station.longitude = lng
                    message = _('Сменили координаты при редактировании из нитки '
                                '%(uid)s, c %(old_lat)s %(old_lng)s на %(lat)s %(lng)s') % {
                        'uid': obj.uid, 'old_lat': old_lat, 'old_lng': old_lng, 'lat': lat, 'lng': lng
                    }

                    station.save()
                    station_admin.log_change(request, station, message)

    def fill_coords(self, s0, s1, unknown):
        # FIXME: использует старые поля
        def not_none(*args):
            for arg in args:
                if arg is not None:
                    return arg

        lng0 = s0.station.longitude
        lat0 = s0.station.latitude
        t0 = not_none(s0.departure, s0.arrival)

        lng1 = s1.station.longitude
        lat1 = s1.station.latitude
        t1 = not_none(s1.departure, s1.arrival)

        if t1 == t0:
            klng = lng1
            klat = lat1
        else:
            klng = (lng1 - lng0) / (t1 - t0)
            klat = (lat1 - lat0) / (t1 - t0)

        for rtstation in unknown:
            t = not_none(rtstation.departure, rtstation.arrival)

            rtstation.station.longitude = lng0 + klng * (t - t0)
            rtstation.station.latitude = lat0 + klat * (t - t0)
            rtstation.station.approx = True

    def get_interpolated_stations(self, rtstations):
        railway_interpolated_stations = {}

        filepath = os.path.join(settings.LOG_PATH, 'railway_interpolated_stations.txt')
        if os.path.exists(filepath):
            with codecs.open(filepath, 'rt', encoding='utf-8') as f:
                for line in f:
                    parts = line.replace('\n', '').split('\t')
                    railway_interpolated_stations[int(parts[0])] = ', '.join(parts[1:])

        interpolated_stations = {}
        for rtstation in rtstations:
            if rtstation.station_id in railway_interpolated_stations:
                interpolated_stations[rtstation.station_id] = railway_interpolated_stations[rtstation.station_id]

        return interpolated_stations

    def get_far_stations(self, rtstations):
        railway_far_stations = {}
        filepath = os.path.join(settings.LOG_PATH, 'railway_far_stations.txt')
        if os.path.exists(filepath):
            with codecs.open(filepath, 'rt', encoding='utf-8') as f:
                for line in f:
                    parts = line.replace('\n', '').split('\t')
                    railway_far_stations[int(parts[0])] = ', '.join(parts[1:])

        far_stations = {}
        for rtstation in rtstations:
            if rtstation.station_id in railway_far_stations:
                far_stations[rtstation.station_id] = railway_far_stations[rtstation.station_id]

    def fill_arrival_and_departure(self, rtstations, naive_start_dt):
        if rtstations[0].tz_departure is None:
            return rtstations

        start_dt = rtstations[0].get_departure_loc_dt(naive_start_dt)

        for rtstation in rtstations:
            arr_dt = rtstation.get_arrival_loc_dt(naive_start_dt)
            arrival = (arr_dt - start_dt).total_seconds() // 60 if arr_dt else None

            dep_dt = rtstation.get_departure_loc_dt(naive_start_dt)
            departure = (dep_dt - start_dt).total_seconds() // 60 if dep_dt else None

            rtstation.arrival, rtstation.departure = arrival, departure

        return rtstations

    def change_view(self, request, object_id, form_url='', extra_context=None):
        today = environment.today()
        thread = get_object_or_404(RThread, id=object_id)
        naive_start_dt = datetime.combine(today, thread.tz_start_time)

        rtstations = list(thread.rtstation_set.select_related('station'))

        preprev = None
        prev = None
        unknown = []

        has_thread_tariffs = ThreadTariff.objects.filter(thread_uid=thread.uid).exists()

        rtstations = self.fill_arrival_and_departure(rtstations, naive_start_dt)

        for rtstation in rtstations:
            if rtstation.station.latitude and (rtstation.arrival is not None or
                                               rtstation.departure is not None):
                if prev:
                    self.fill_coords(prev, rtstation, unknown)
                    preprev = prev
                    unknown = []

                prev = rtstation
            else:
                unknown.append(rtstation)

        if unknown and preprev and prev:
            self.fill_coords(preprev, prev, unknown)

        json_fields = []
        for json_field in ['translated_manual_days_texts',
                           'translated_days_texts',
                           'translated_except_texts']:

            value = getattr(thread, json_field)
            if value:
                try:
                    value = json.loads(value)
                except ValueError:
                    pass

            json_fields.append({
                'name': json_field,
                'verbose_name': RThread._meta.get_field(json_field).verbose_name,
                'value': json.dumps(value, ensure_ascii=False, encoding='utf8', indent=2)
            })

        return super(RThreadAdmin, self).change_view(request, object_id, extra_context={
            'rtstations': rtstations,
            'interpolated_stations': self.get_interpolated_stations(rtstations),
            'far_stations': self.get_far_stations(rtstations),
            'json_fields': json_fields,
            'has_thread_tariffs': has_thread_tariffs,
        })


class RTStationAdmin(RaspExportModelAdmin):
    raw_id_fields = ('thread', 'station', 'arrival_t_model', 'departure_t_model', 'terminal')
    fieldsets = (
        (_('Смешанные поля'), {
            'fields': (
                'thread',
                'station',
                'is_technical_stop',
                'terminal',
                'arrival_t_model',
                'departure_t_model',
                'is_virtual_end',
                'is_combined',
                'platform',
                'train_number'
            )
        }),
        (_('Новая информация об остановке'), {
            'fields': ('time_zone', 'tz_arrival', 'tz_departure')
        }),
        (_('Нечеткие признаки'), {
            'fields': ('is_fuzzy', 'is_searchable_from', 'is_searchable_to',
                       'in_station_schedule', 'in_thread')
        }),
        (_('Информация о направлениях'), {
            'fields': ('arrival_direction',
                       'arrival_subdir',
                       'departure_direction',
                       'departure_subdir',
                       'is_from_subdir')
        }),
        ('Code sharing', {
            'fields': ('arrival_code_sharing',
                       'departure_code_sharing')
        }),
    )
    list_display = ('get_thread_number', 'get_station_title', 'terminal_letter', 'time_zone', 'tz_arrival',
                    'tz_departure', 'is_technical_stop', 'get_directions', 'arrival_subdir', 'departure_subdir',
                    'get_is_fuzzy', 'get_is_combined', 'train_number')
    list_filter = ()
    search_fields = ('thread__number',)

    def get_is_fuzzy(self, rtstation):
        return rtstation.is_fuzzy

    get_is_fuzzy.allow_tags = True
    get_is_fuzzy.short_description = 'fuzzy'
    get_is_fuzzy.boolean = True

    def get_is_combined(self, rtstation):
        return rtstation.is_combined

    get_is_combined.allow_tags = True
    get_is_combined.short_description = _('Согл. Перес.')
    get_is_combined.boolean = True

    def get_station_title(self, rtstation):
        return StationAdmin.station_full_path(rtstation.station, include_station=True)

    get_station_title.allow_tags = True
    get_station_title.short_description = _('Станция')

    def terminal_letter(self, rtstation):
        if rtstation.terminal_id:
            return rtstation.terminal.name
    terminal_letter.short_description = 'term'

    def get_thread_number(self, rtstation):
        return rtstation.thread.number

    get_thread_number.short_description = _('Нитка')

    def get_directions(self, rtstation):
        if not rtstation.arrival_direction and not rtstation.departure_direction:
            return None
        if rtstation.arrival_direction != rtstation.departure_direction:
            directions = filter(None, [rtstation.arrival_direction and rtstation.arrival_direction.title,
                                       rtstation.departure_direction and rtstation.departure_direction.title])
            return ' - '.join(directions)
        else:
            return rtstation.arrival_direction.title

    get_directions.short_description = _('Направление')

admin.site.register(RTStation, RTStationAdmin)


class RoutePathAdmin(RaspExportModelAdmin):
    raw_id_fields = ('station_from', 'station_to')
    list_display = ('station_from', 'station_to', 'status_direct', 'status_back')

admin.site.register(RoutePath, RoutePathAdmin)


class PlatformTranslationAdmin(RaspExportModelAdmin):
    list_display = ('__unicode__', ) + \
        tuple(L_field.admin_fields(PlatformTranslation, ['platform'], show_local_field=False))

    fieldsets = (
        (_('Смешанные поля'), {
            'fields': ('description',),
        }),
        (_('Платформа'), {
            'fields': L_field.admin_fields(PlatformTranslation, ['platform'], show_local_field=True),
        }),
    )

admin.site.register(PlatformTranslation, PlatformTranslationAdmin)


class StationPhoneAdmin(RaspExportModelAdmin):
    raw_id_fields = ('station',)
    fieldsets = (
        ('Profile field', {
            'fields': ('station', 'phone', 'note')
        }),
    )
    list_display = ('station', 'phone', 'note')
    search_fields = ('phone', 'station__title', 'note')

admin.site.register(StationPhone, StationPhoneAdmin)


admin.site.register(RouteMapBlacklist)


class Route2CompanyAdmin(RaspExportModelAdmin):
    raw_id_fields = ('company',)
    fieldsets = (
        ('Profile field', {
            'fields': ('number', 'company')
        }),
    )
    list_display = ('number', 'company')
    search_fields = ('number', 'company__title')

admin.site.register(Route2Company, Route2CompanyAdmin)


class SupplierAdmin(RaspExportModelAdmin):
    fieldsets = (
        ('Profile field', {'fields': (
            'title', 'code', 'filename', 'hide_in_filters',
            'add_to_route_changes_report', 'exclude_from_external_api',
            'logo', 'displace_yabus',
        )}),
        (_('Продажи'), {'fields': (
            'can_buy_ru', 'can_buy_ua', 'can_buy_tr',
            'sale_url_template', 'sale_start_days', 'sale_stop_hours',
        )}),
    )
    list_display = ('id', 'title', 'code', 'can_buy_ru', 'can_buy_ua', 'can_buy_tr')
    search_fields = ('title', 'code')
    ordering = ('id',)

admin.site.register(Supplier, SupplierAdmin)


class StationCodeAdmin(RaspExportModelAdmin):
    fieldsets = (
        ('Profile field', {
            'fields': ('station', 'code', 'system')
        }),
    )
    raw_id_fields = ('station',)
    list_filter = ('system',)
    search_fields = ('station__title', 'code')
    list_display = ('station', 'code', 'system')
    ordering = ('station_id',)

admin.site.register(StationCode, StationCodeAdmin)


class SuggestAdmin(RaspExportModelAdmin):
    list_filter = ('type', 'ttype')
    search_fields = ('title',)
    list_display = ('title', 'ttype', 'object_id', 'type',
                    'full_title_ru', 'full_title_tr')
    raw_id_fields = ('region', 'country')

admin.site.register(Suggest, SuggestAdmin)


class DirectionMarkerInline(admin.TabularInline):
    model = DirectionMarker
    raw_id_fields = ('station', 'direction', 'prev', 'next')
    extra = 0


class DirectionAdmin(RaspExportModelAdmin):
    list_filter = ('suburban_zone', )
    search_fields = ('title', 'code')
    list_display = ('title', 'title_from', 'title_to', 'code')
    inlines = (DirectionMarkerInline,)

admin.site.register(Direction, DirectionAdmin)


class TranslateAdmin(RaspExportModelAdmin):
    search_fields = ('value',)
    list_display = ('value',)

admin.site.register(DirectionTranslate, TranslateAdmin)
admin.site.register(DirectionFromTranslate, TranslateAdmin)


class DirectionMarkerAdmin(RaspExportModelAdmin):
    raw_id_fields = ('station', 'direction', 'prev', 'next')
    list_filter = ('direction',)
    list_display = ('order', 'direction_name', '__unicode__',
                    'title_from', 'title_to',
                    'arrow_type')

    @transaction.atomic
    def shift_down(self, request, queryset):
        directions = set([dm.direction for dm in queryset])
        if len(directions) > 1:
            messages.add_message(request, messages.ERROR, _(u"Нужно выбирать маркеры из одного направления"))
        else:
            direction = list(directions)[0]
            # Начиная с минимального номера увеличиваем всем порядок на 1
            min_order = queryset.order_by('order')[0].order
            for dm in direction.directionmarker_set.filter(order__gte=min_order).order_by('-order'):
                dm.order += 1
                dm.save()

        messages.add_message(request, messages.SUCCESS, _('Порядковые номера маркеров были увеличены'))
        return HttpResponseRedirect('./?direction=%d' % direction.id)

    shift_down.short_description = _('Сдвинуть вниз')

    actions = ['shift_down']

admin.site.register(DirectionMarker, DirectionMarkerAdmin)


class SuburbanZoneAdmin(RaspExportModelAdmin):
    raw_id_fields = ('settlement',)
    list_display = ('title', 'code')
    ordering = ('title',)

    def get_urls(self):
        from django.conf.urls import url

        def wrap(view):
            def wrapper(*args, **kwargs):
                return self.admin_site.admin_view(view)(*args, **kwargs)
            return update_wrapper(wrapper, view)

        urlpatterns = super(SuburbanZoneAdmin, self).get_urls()

        info = self.model._meta.app_label, self.model._meta.model_name

        urlpatterns = [
            url(r'^(.+)/notices/$', wrap(self.notices), name='%s_%s_notices' % info),
        ] + urlpatterns

        return urlpatterns

    def notices(self, request, object_id):
        from suburban_change_notices.models import Notice

        current_date = environment.today()
        if request.GET.get('date'):
            current_date = datetime.strptime(request.GET.get('date'), '%Y-%m-%d').date()

        zone = SuburbanZone.objects.get(id=int(object_id))

        filter_ = (
            Q(start_date__gte=current_date) |
            Q(start_date__lt=current_date, end_date__gte=current_date))

        notices = (Notice.objects
                         .filter(filter_, directions__suburban_zone=zone)
                         .distinct()
                         .order_by('start_date', 'end_date'))

        json_notices = []

        for notice in notices:
            json_notices.append({
                'title': notice.title,
                'text': notice.text,
                'mobile_text': notice.mobile_text,
                'start_date': six.text_type(notice.start_date),
                'end_date': six.text_type(notice.end_date),
                'type': notice.type,
                'type_display': six.text_type(notice.get_type_display()),
                'directions': [{'code': d.code, 'title': d.title} for d in notice.directions.all()]
            })

        return HttpResponse(json.dumps({'data': json_notices}, ensure_ascii=False, indent=2),
                            content_type='application/json; charset=utf-8')


admin.site.register(SuburbanZone, SuburbanZoneAdmin)


class TicketOfficeAdmin(RaspExportModelAdmin):
    raw_id_fields = ('settlement',)
    search_fields = ('settlement__title', 'title', 'address', 'postal_code')
    list_display = ('title', 'settlement', 'plane', 'train', 'postal_code', 'address', 'home_number')
    fieldsets = (
        ('Profile field', {
            'fields': ('title', 'settlement', 'postal_code', 'address',
                       'home_number', 'address2', 'phone', 'fax',
                       'home_site', 'email', 'plane',
                       'train', 'companies')
        }),
        (_('Координаты'), {
            'classes': ('ymaps-point',),
            'fields': ('longitude', 'latitude'),
        })
    )

admin.site.register(TicketOffice, TicketOfficeAdmin)


class ExternalDirectionMarkerAdmin(RaspExportModelAdmin):
    list_filter = ('external_direction',)
    list_display = ('order', 'direction_name', '__unicode__')
    raw_id_fields = ('station', 'external_direction')

    def direction_name(self, obj):
        return obj.external_direction.title

    direction_name.short_description = _(u"направление")

    @transaction.atomic
    def shift_down(self, request, queryset):
        external_directions = set([edm.external_direction for edm in queryset])
        if len(external_directions) > 1:
            messages.add_message(request, messages.ERROR, _('Нужно выбирать маркеры из одного направления'))
        else:
            external_direction = list(external_directions)[0]
            # Начиная с минимального номера увеличиваем всем порядок на 1
            min_order = queryset.order_by('order')[0].order
            for edm in external_direction.externaldirectionmarker_set.filter(order__gte=min_order).order_by('-order'):
                edm.order += 1
                edm.save()

        messages.add_message(request, messages.SUCCESS, _(u"Порядковые номера маркеров были увеличены"))
        return HttpResponseRedirect('./?external_direction=%d' % external_direction.id)

    shift_down.short_description = _('Сдвинуть вниз')

    actions = ['shift_down']

admin.site.register(ExternalDirectionMarker, ExternalDirectionMarkerAdmin)


class ExternalDirectionMarkerInline(admin.TabularInline):
    model = ExternalDirectionMarker
    raw_id_fields = ('station', )
    extra = 0


class ExternalDirectionAdmin(RaspExportModelAdmin):
    search_fields = ('title', 'code')
    list_filter = ('suburban_zone', )
    list_display = ('title', 'code')
    inlines = (ExternalDirectionMarkerInline,)
    raw_id_fields = ('base_station', )

admin.site.register(ExternalDirection, ExternalDirectionAdmin)


class SuburbanTariffAdmin(RaspExportModelAdmin):
    list_display = ('code', 'tariff')


admin.site.register(SuburbanTariff, SuburbanTariffAdmin)


class RouteImportInfoAdmin(RaspExportModelAdmin):
    list_display = ('number', 'supplier', 't_type')
    fieldsets = (
        ('Identification fields', {
            'fields': ('number', 'supplier', 'route_uid')
        }),
        ('Add. info fields', {
            'fields': ('t_type',)
        }),
    )

admin.site.register(RouteImportInfo, RouteImportInfoAdmin)


class PageAdmin(RaspExportModelAdmin):
    list_display = ('title', 'code')

admin.site.register(Page, PageAdmin)


class TeaserCountryInline(admin.TabularInline):
    model = Teaser.countries.through
    raw_id_fields = ('country',)
    extra = 1


class TeaserSettlementInline(admin.TabularInline):
    model = Teaser.settlements.through
    raw_id_fields = ('settlement',)
    extra = 1


class TeaserStationInline(admin.TabularInline):
    model = Teaser.stations.through
    raw_id_fields = ('station',)
    extra = 1


class TeaserDirectionInline(admin.TabularInline):
    model = Teaser.directions.through
    raw_id_fields = ('externaldirection',)
    extra = 1


class TeaserThreadInline(admin.TabularInline):
    model = Teaser.threads.through
    raw_id_fields = ('rthread',)
    extra = 1


class TeaserCompanyInline(admin.TabularInline):
    model = Teaser.companies.through
    raw_id_fields = ('company',)
    extra = 1


class TeaserPageInline(admin.TabularInline):
    model = Teaser.pages.through
    raw_id_fields = ('page',)
    extra = 1


class UUID64FieldWidget(AdminBigIntegerFieldWidget):
    @property
    def media(self):
        return forms.Media(js=('rasp/js/teaser_uid.js',))

    def render(self, name, value, attrs=None, renderer=None):
        widget = super(UUID64FieldWidget, self).render(name, value, attrs=attrs, renderer=renderer)
        return mark_safe(u'{}<button type="button" class="js-regenerate-uid">Перегенерировать uid</button>'.
                         format(widget))


class TeaserAdmin(RaspExportModelAdmin):
    fieldsets = (
        (
            None,
            {'fields': ('title', 'content', 'mobile_content',
                        'url', 'image', 'template',
                        'mode', 'importance',
                        'date_start', 'date_finish',
                        'uid64')}
        ),
        (
            _('Где показывать'),
            {'fields': ('lang', 'national_version',
                        'is_active_rasp', 'is_active_ticket', 'is_active_export')}
        ),
    )

    list_display = ('title', 'content', 'lang', 'national_version',
                    'is_active_rasp', 'is_active_ticket', 'is_active_export',
                    'mode', 'importance',
                    'date_start', 'date_finish')
    list_filter = ('is_active_rasp', 'is_active_ticket', 'is_active_export',
                   'importance', 'mode',
                   'lang', 'national_version')
    inlines = [TeaserPageInline, TeaserCountryInline, TeaserSettlementInline, TeaserStationInline,
               TeaserDirectionInline, TeaserThreadInline, TeaserCompanyInline]
    ordering = ('-is_active_rasp',)

    formfield_overrides = {
        UUID64Field: {'widget': UUID64FieldWidget},
    }

    def queryset(self, request):
        return Teaser.base_objects.all()


admin.site.register(Teaser, TeaserAdmin)


class DeLuxeTrainForm(ModelForm):
    class Meta:
        widgets = {'numbers': forms.Textarea}


class DeLuxeTrainAdmin(RaspExportModelAdmin):
    form = DeLuxeTrainForm

    list_display = (
        'title_admin', 'numbers', 'deluxe', 'high_speed',
        'im_title', 'need_check', 'need_reverse_check'
    )
    search_fields = ('numbers', 'title', 'im_title')
    list_filter = ('deluxe', 'high_speed', 'need_check', 'need_reverse_check')

    fieldsets = (
        (None, {
            'fields': (
                'numbers',
                'deluxe',
                'high_speed',
                'page_path',
                't_subtype'
            )
        }),
        (_('Название'), {
            'fields': L_field.admin_fields(DeLuxeTrain, ['title'], show_local_field=True)
        }),
        (_('Проверка'), {
            'fields': (
                'wagon_types',
                'im_title',
                'need_check',
                'need_reverse_check',
                'min_im_records_count'
            )
        }),

    )

    def title_admin(self, obj):
        return obj.L_title_short() or _(u'название не указано')

admin.site.register(DeLuxeTrain, DeLuxeTrainAdmin)


class AeroexTariffAdmin(RaspExportModelAdmin):
    list_filter = ('type', 'precalc')
    list_display = (
        'station_from', 'station_to', 'reverse', 'type', 'tariff', 'currency',
        'suburban_search', 'precalc', 'replace_tariff_type'
    )
    raw_id_fields = ('station_from', 'station_to')
    search_fields = ('station_from__title', 'station_to__title')

    def get_changelist(self, request, **kwargs):
        """
        Returns the ChangeList class for use on the changelist page.
        """
        from travel.rasp.admin.lib.admin_options import ChangeListNoDefaultOrdering
        return ChangeListNoDefaultOrdering

admin.site.register(AeroexTariff, AeroexTariffAdmin)


class NotTariffTrainAdmin(RaspExportModelAdmin):
    list_display = ('number',)

admin.site.register(NotTariffTrain, NotTariffTrainAdmin)

admin.site.register(ExpressTypeLite)


class ExpressNumberAdmin(RaspExportModelAdmin):
    list_display = ('number', 'supplier', 'express_type')

admin.site.register(ExpressNumber, ExpressNumberAdmin)


class RedirectAdmin(RaspExportModelAdmin):
    list_display = ('old_url', 'new_url')
    search_fields = ('old_url', 'new_url')

admin.site.register(Redirect, RedirectAdmin)


class StationTerminalAdmin(RaspExportModelAdmin):
    raw_id_fields = ('station',)
    list_display = ('station', 'name', 'is_domestic', 'is_international')

admin.site.register(StationTerminal, StationTerminalAdmin)


class WayToAirportAdmin(RaspExportModelAdmin):
    raw_id_fields = ('station', 'from_station', 'to_station')
    list_display = ('station', 'title_ru', 'way_type')

admin.site.register(WayToAirport, WayToAirportAdmin)


class ThreadTariffAdmin(RaspExportModelAdmin):
    raw_id_fields = ('station_from', 'station_to', 'settlement_from', 'settlement_to')
    list_display = ('thread_uid', 'station_from', 'station_to', 'tariff')
    search_fields = ('thread_uid', 'station_from__title', 'station_to__title')

    def get_changelist(self, request, **kwargs):
        """
        Returns the ChangeList class for use on the changelist page.
        """
        from travel.rasp.admin.lib.admin_options import ChangeListNoDefaultOrdering
        return ChangeListNoDefaultOrdering


admin.site.register(ThreadTariff, ThreadTariffAdmin)


class TariffTypeAdmin(RaspExportModelAdmin):
    list_display = ('title', 'category', 'code', 'order', 'is_main')
    list_filter = ['category', 'tariff_groups__title']
    filter_horizontal = ['tariff_groups']


admin.site.register(TariffType, TariffTypeAdmin)


@admin.register(TariffGroup)
class TariffGroupAdmin(RaspExportModelAdmin):
    list_display = ['title']


class TrainSchedulePlanAdmin(RaspExportModelAdmin):
    list_display = ('title', 'start_date', 'end_date')
    fieldsets = (
        ('', {
            'fields': ('title', 'code', 'start_date', 'end_date', 'appendix_type')
        }),
    )

admin.site.register(TrainSchedulePlan, TrainSchedulePlanAdmin)


class DefaultPointAdmin(RaspExportModelAdmin):
    list_display = ('title', 'settlement', 'station')
    raw_id_fields = ('station', 'settlement')

admin.site.register(DefaultPoint, DefaultPointAdmin)


class ReplaceExceptionAdmin(RaspExportModelAdmin):
    list_display = ('city_from', 'city_to', 'station_to')
    raw_id_fields = ('city_from', 'city_to', 'station_to')

admin.site.register(ReplaceException, ReplaceExceptionAdmin)


class ThreadScheduleAdmin(RaspExportModelAdmin):
    list_display = ('thread', 'start_time', 'end_time', 'week_days', 'frequency', 'times')
    raw_id_fields = ('thread',)
    search_fields = ('thread__route__number', 'thread__number')

admin.site.register(ThreadSchedule, ThreadScheduleAdmin)


class StationPassageAdmin(RaspExportModelAdmin):
    list_display = ('station_from', 'terminal_from', 'station_to', 'terminal_to', 'duration')
    raw_id_fields = ('station_from', 'terminal_from', 'station_to', 'terminal_to')


admin.site.register(StationPassage, StationPassageAdmin)

admin.site.register(CodeSystem)


class TrainPseudoStationMapAdmin(RaspExportModelAdmin):
    search_fields = ('number', 'pseudo_station__title', 'station__title')
    list_display = ('number', 'pseudo_station', 'station')
    raw_id_fields = ('station', 'pseudo_station')

admin.site.register(TrainPseudoStationMap, TrainPseudoStationMapAdmin)


class DistrictAdmin(RaspExportModelAdmin):
    search_fields = ('title', 'settlement__title', 'region__title')
    list_display = ('title', 'settlement', 'region', 'country')
    raw_id_fields = ('region', 'settlement')

    def country(self, obj):
        return (
            obj.settlement and obj.settlement.country
        ) or (
            obj.region and obj.region.country
        )

    country.short_description = _('страна')

admin.site.register(District, DistrictAdmin)


class PseudoRegionCountriesInline(admin.TabularInline):
    model = PseudoRegion.countries.through
    raw_id_fields = ('country',)
    extra = 1


class PseudoRegionAdmin(RaspExportModelAdmin):
    list_display = ('title',)
    fields = ('title',)
    inlines = [PseudoRegionCountriesInline]

admin.site.register(PseudoRegion, PseudoRegionAdmin)


class RegionWhiteListForm(ModelForm):
    regions = forms.ModelMultipleChoiceField(Region.objects.all(),
                                             label=_('Области'),
                                             widget=FilteredSelectMultiple(_('Области'),
                                                                           False,
                                                                           attrs={'rows': '10'}))

    class Meta:
        model = RegionWhiteList
        exclude = []


class RegionWhiteListAdmin(RaspExportModelAdmin):
    form = RegionWhiteListForm

admin.site.register(RegionWhiteList, RegionWhiteListAdmin)


class StationExpressAliasAdmin(RaspExportModelAdmin):
    list_display = ('station', 'alias')
    raw_id_fields = ('station',)


admin.site.register(StationExpressAlias, StationExpressAliasAdmin)


class GortransCityLinkAdmin(RaspExportModelAdmin):
    list_display = ('city', 'url_a')
    raw_id_fields = ('city',)

admin.site.register(GortransCityLink, GortransCityLinkAdmin)


class StationTeleportMapAdmin(RaspExportModelAdmin):
    list_display = ('station_from', 'settlement_from', 'station_to', 'settlement_to', 'teleport_time')
    raw_id_fields = ('station_from', 'settlement_from', 'station_to', 'settlement_to')

admin.site.register(StationTeleportMap, StationTeleportMapAdmin)


class SettingsAdmin(RaspExportModelAdmin):
    list_display = ('name', 'code', 'value', 'help_text')
    readonly_fields = ('type', 'code')

    def has_delete_permission(self, request, obj=None):
        return False

    def has_add_permission(self, request):
        return False

admin.site.register(Setting, SettingsAdmin)


class FaceDirectionAdmin(RaspExportModelAdmin):
    list_display = ('departure_settlement', 'arrival_settlement', 'ttype')
    list_filter = ('ttype',)
    raw_id_fields = ('arrival_settlement', 'departure_settlement')

admin.site.register(FaceDirection, FaceDirectionAdmin)


class CustomSpecialOfferSettlementFromFilter(SimpleListFilter):
    title = _('Город отправления')
    parameter_name = 'settlement_from__id__exact'

    def lookups(self, request, model_admin):
        settlement_from_ids = SpecialOffer.objects.values_list('settlement_from', flat=True).distinct()
        settlements_from = Settlement.objects.in_bulk(settlement_from_ids).items()

        return settlements_from

    def queryset(self, request, queryset):
        if self.value():
            return queryset.filter(settlement_from=self.value())

        else:
            return queryset


class SpecialOfferAdmin(RaspExportModelAdmin):
    def company_icon_img(self, obj):
        if obj.company.icon:
            content = ('<img src="%s" width="16" height="16" title="%s" alt="%s"/>' %
                       (obj.company.icon.url, obj.company.L_title(), obj.company.iata))
        else:
            content = obj.company.iata

        return format_html('<a href="/admin/www/company/{}/" target="_blank">{}</a>', obj.company.id, content)

    company_icon_img.allow_tags = True
    company_icon_img.short_description = _(u'А/к')

    list_display = ('settlement_from', 'settlement_to', 'company_icon_img', 'tariff',
                    'date_start', 'date_end')
    raw_id_fields = ('settlement_from', 'settlement_to', 'airport_from', 'airport_to', 'company')
    search_fields = (
        'company__iata',
        'settlement_from__iata',
        'settlement_to__iata',
        'settlement_to__title_ru',
    )
    list_filter = (
        'reverse', 'service_class', 'online_booking',
        CustomSpecialOfferSettlementFromFilter,
    )

admin.site.register(SpecialOffer, SpecialOfferAdmin)


class TransportSubtypeAdmin(RaspExportModelAdmin):
    list_filter = ('t_type',)
    list_display = ('code', 'title_ru', 't_type', 'color', 'use_in_suburban_search')

admin.site.register(TransportSubtype, TransportSubtypeAdmin)


class TransportSubtypeColorAdmin(RaspExportModelAdmin):
    pass

admin.site.register(TransportSubtypeColor, TransportSubtypeColorAdmin)


class DefaultSuggestAdmin(RaspExportModelAdmin):
    fieldsets = (
        ('', {
            'fields': ('suggest_type_id', 'national_version', 'priority')
        }),
        ('Значения', {
            'fields': L_field.admin_fields(DefaultSuggest, ['value'], show_local_field=True)
        }),
    )
    list_display = ('suggest_type_id', 'national_version', 'value', 'priority')

admin.site.register(DefaultSuggest, DefaultSuggestAdmin)


class CompanyOfficeAdmin(RaspExportModelAdmin):
    search_fields = ('title',
                     'settlement__country__title_ru', 'settlement__country__title_en',
                     'settlement__country__title_uk', 'settlement__country__title_tr',
                     'settlement__title_ru', 'settlement__title_en',
                     'settlement__title_tr', 'settlement__title_uk',
                     'company__title_ru', 'company__title_tr',
                     'company__title_en', 'company__title_uk')

    list_display = ('title', 'country', 'settlement', 'main_station')

    raw_id_fields = ('company', 'settlement', 'main_station')
    list_filter = ('is_main',)

    fieldsets = (
        ('', {
            'fields': ('title', 'company', 'settlement', 'main_station', 'address', 'contact_info',
                       'phone', 'phone_booking', 'description', 'is_main')
        }),
        (_('Координаты'), {
            'classes': ('ymaps-point',),
            'fields': ('longitude', 'latitude'),
        }),
    )

    def country(self, obj):
        return six.text_type(obj.settlement.country)

admin.site.register(CompanyOffice, CompanyOfficeAdmin)


class TrainTurnoverAdmin(RaspExportModelAdmin):
    raw_id_fields = ('station', 'graph')
    list_display = ('number_before', 'station', 'template_code', 'graph')

admin.site.register(TrainTurnover, TrainTurnoverAdmin)


class TrainPurchaseNumberAdmin(RaspExportModelAdmin):
    raw_id_fields = ('thread',)
    list_display = ('thread', 'number')

admin.site.register(TrainPurchaseNumber, TrainPurchaseNumberAdmin)


class PathfinderMapsNearestSettlementAdmin(RaspExportModelAdmin):
    raw_id_fields = ('settlement_from', 'settlement_to')
    list_display = ('settlement_from', 'settlement_to')

admin.site.register(PathfinderMapsNearestSettlement, PathfinderMapsNearestSettlementAdmin)
