# -*- coding: utf-8 -*-

import logging
import contextlib

from django.core.files.base import ContentFile
from django.db import models
from django.conf import settings
from django.db.models.signals import pre_save, post_save
from django.dispatch import receiver
from django.utils.translation import ugettext_lazy as _

import travel.rasp.admin.www.models.geo
import travel.rasp.admin.www.models.precalc
import travel.rasp.admin.www.models.schedule
import travel.rasp.admin.www.models.tariffs
import travel.rasp.admin.www.models.suggests
from common.models.geo import Station
from common.models.schedule import Company
from travel.rasp.admin.lib.image import get_pilimage_content
from travel.rasp.admin.lib.image import svg2image


log = logging.getLogger(__name__)


class Redirect(models.Model):
    old_url = models.CharField(max_length=100, verbose_name=_(u"Старый адрес"),
                               unique=True)
    new_url = models.CharField(max_length=100, verbose_name=_(u"Новый адрес"))

    def get_new_url(self):
        if self.new_url.startswith('http://'):
            return self.new_url
        else:
            return "https://%s%s" % (settings.DOMAIN_NAME, self.new_url)

    class Meta:
        verbose_name = _(u"Редирект")
        verbose_name_plural = _(u"Редиректы")
        app_label = 'www'


@receiver(pre_save, sender=Company)
def company_pre_save(instance, **kwargs):
    try:
        orig = Company.objects.get(id=instance.id)

    except Company.DoesNotExist:
        return

    svg_logo_changed = instance.svg_logo != orig.svg_logo

    if svg_logo_changed or not instance.svg_logo:
        instance.svg2png_logo = None


@receiver(post_save, sender=Company)
def company_post_save(instance, **kwargs):
    c = instance

    if c.svg_logo and not c.svg2png_logo:
        try:
            with contextlib.closing(c.svg_logo) as f:
                svg = f.read()

            img = svg2image(svg, size=(132, 132))
            img_content = get_pilimage_content(img)
            c.svg2png_logo.save('unused_name', ContentFile(img_content))

        except Exception:
            log.exception('Saving svg2png_logo error')


@receiver(pre_save, sender=Station)
def station_change_coordinates_handler(sender, **kwargs):
    instance = kwargs['instance']

    if not instance.id:
        return

    try:
        old_value = Station.objects.get(id=instance.id)

        if not (instance.longitude == old_value.longitude and instance.latitude == old_value.latitude):
            travel.rasp.admin.www.models.geo.RoutePath.set_changed(instance)

    except Station.DoesNotExist:
        pass
