# coding: utf-8
import re
import logging

from lxml import etree
from django.db import models
from django.utils import translation
from django.utils.http import urlencode
from django.utils.translation import ugettext_lazy as _

from common.db.mds.s3_storage import mds_s3_media_storage
from common.models_abstract.schedule import EXPRESS_TYPE_CHOICES
from common.models_admin.schedule import StationTeleportMap  # noqa
from travel.rasp.library.python.common23.date import environment
from common.utils.fields import RegExpField
from common.xgettext.i18n import gettext, xformat, xgettext

from travel.rasp.admin.www.templatetags.common import url
from travel.rasp.admin.www.utils.common import RegExpSet


log = logging.getLogger(__name__)


class TransportScheme(models.Model):
    title = models.CharField(verbose_name=_(u'название'), max_length=100,
                             null=True, blank=True, default=None)
    img = models.ImageField(verbose_name=_(u'схема'),
                            storage=mds_s3_media_storage,
                            upload_to='data/transportscheme/img')
    transport = models.ForeignKey('www.TransportModel', verbose_name=_(u'модель ТС'),
                                  null=True, blank=True, default=None)
    transport_img = models.ImageField(verbose_name=_(u'фото ТС'),
                                      storage=mds_s3_media_storage,
                                      upload_to='data/transportscheme/transport_img',
                                      null=True, blank=True, default=None)
    company = models.ForeignKey('www.Company', verbose_name=_(u'компания-перевозчик'),
                                null=True, blank=True, default=None)

    def __unicode__(self):
        return self.title

    class Meta:
        verbose_name = _(u'схема салона ТС')
        verbose_name_plural = _(u'схемы салонов ТС')
        ordering = ('title',)
        app_label = 'www'


class ThreadSchedule(models.Model):
    thread = models.ForeignKey('RThread', verbose_name=_(u'нитка'))
    schedule_oid = models.IntegerField(verbose_name=_(u'номер расписания внутри нитки'))
    start_date = models.DateField(verbose_name=_(u'начало периода'))
    end_date = models.DateField(verbose_name=_(u'конец периода'))
    start_time = models.TimeField(verbose_name=_(u'время начала промежутка'))
    end_time = models.TimeField(verbose_name=_(u'время окончания промежутка'))
    frequency = models.IntegerField(verbose_name=_(u'интервал хождения'),
                                    null=True, blank=True, default=None)
    times = models.CharField(verbose_name=_(u'время хождения вне интервалов'),
                             max_length=255, null=True, blank=True, default=None)
    week_days = models.CharField(verbose_name=_(u'дни недели'),
                                 default='1234567',
                                 max_length=20, null=True, blank=True)
    is_cancel = models.BooleanField(verbose_name=_(u'является ли отменой'),
                                    default=False)
    time_zone = models.CharField(verbose_name=_(u'временная зона'),
                                 max_length=30)

    def schedule_id(self):
        return '%s_%s' % (self.thread.uid, self.schedule_oid)

    @classmethod
    def scoped_by_date(cls, _date=None):
        u"""Расписания, действующие на сегодняшнюю или любую другую дату"""
        if _date is None:
            _date = environment.today()
        return cls.objects.filter(start_date__lte=_date, end_date__gte=_date)

    def as_xml(self):
        element = etree.Element('schedule')

        element.set('period_start_date', self.start_date.strftime('%Y-%m-%d'))
        element.set('period_end_date', self.end_date.strftime('%Y-%m-%d'))
        element.set('period_start_time', self.start_time.strftime('%H:%M:%S'))
        element.set('period_end_time', self.end_time.strftime('%H:%M:%S'))

        if self.frequency is not None:
            element.set('period_int', str(self.frequency))
        elif self.times is not None:
            element.set('times', self.times.replace(',', ';'))

        element.set('days', self.week_days)

        if self.is_cancel:
            element.set('canceled', '1')

        element.set('timezone', self.time_zone)

        return element

    class Meta:
        verbose_name = _(u'расписание движения транспорта')
        verbose_name_plural = _(u'расписания движения транспорта')
        app_label = 'www'


class Route2Company(models.Model):
    """ Список ЖД рейсов принадлежащих не РЖД """

    number = models.CharField(verbose_name=_(u'номер рейса'), max_length=100,
                              blank=False)
    company = models.ForeignKey('Company', verbose_name=_(u'компания-перевозчик'),
                                null=False, blank=False)

    def __unicode__(self):
        return self.number + ': ' + self.company.title

    def save(self, **kwargs):
        self.number = self.number.strip()
        super(Route2Company, self).save(**kwargs)

    class Meta:
        verbose_name = _(u'соответствие номера рейса и компании')
        verbose_name_plural = _(u'соответствия номеров ЖД рейсов и компаний')
        ordering = ('number', 'company')
        unique_together = (('number', 'company'),)
        app_label = 'www'


class RouteImportInfo(models.Model):
    u"""Содержить информацию для рейса основываясь на номере и поставщике.
    Можно так же идентифицировать рейс по route_uid, если есть колизии."""
    number = models.CharField(verbose_name=_(u'номер'), max_length=100, db_index=True, null=False)
    supplier = models.ForeignKey('Supplier', verbose_name=_(u'поставщик'),
                                 null=False)
    route_uid = models.CharField(verbose_name=_(u"идентификатор"),
                                 help_text=_(u"идентификатор для однозначного "
                                             u"определения рейса из импорта"),
                                 max_length=100, null=True, default=None,
                                 blank=True, unique=True)
    t_type = models.ForeignKey('TransportType', verbose_name=_(u'тип транспорта'),
                               null=False, blank=False)

    def save(self, *args, **kwargs):
        self.route_uid = self.route_uid or None
        super(RouteImportInfo, self).save(*args, **kwargs)

    def __unicode__(self):
        return self.number + u" " + self.supplier.title

    class Meta:
        ordering = ('supplier__id', 'number')
        verbose_name = _(u'доп. информация о рейсе')
        verbose_name_plural = _(u'доп. информация о рейсе')
        unique_together = (('number', 'supplier'))
        app_label = 'www'


class ExpressNumber(models.Model):
    u"""Список номеров экспрессов
    """
    number = RegExpField(verbose_name=_(u'номер рейса'), max_length=255, unique=True,
                         blank=False, help_text=_(u'номер рейса можно задавать регулярым выражением'))
    supplier = models.ForeignKey('Supplier', verbose_name=_(u'поставщик'),
                                 null=False, blank=False)
    express_type = models.CharField(verbose_name=_(u"тип экспресса"),
                                    choices=EXPRESS_TYPE_CHOICES,
                                    max_length=20, null=False, blank=False)

    class Meta:
        verbose_name = _(u"номер экспресса")
        verbose_name_plural = _(u"номера экспрессов")
        app_label = 'www'

    _number_sets = {}

    @classmethod
    def number_set(cls, supplier, not_cache=False):
        # TODO: init in lib.precache
        if supplier not in cls._number_sets or not_cache:
            cls._number_sets[supplier] = RegExpSet([en.number for en in cls.objects.filter(supplier=supplier)])

        return cls._number_sets[supplier]

    @classmethod
    def get_express_type(cls, number, supplier):
        if number in cls.number_set(supplier):
            for en in cls.objects.filter(supplier=supplier):
                if re.match(en.number, number, re.U):
                    return en.express_type
            raise cls.DoesNotExist()
        return None

    def __unicode__(self):
        return self.number + u" " + self.supplier.title + u" - " + self.express_type


class FaceDirection(models.Model):
    u"""Дополнительные направления на морде города"""

    departure_settlement = models.ForeignKey('Settlement', verbose_name=_(u'Город отправления'),
                                             related_name='departure_facedirections')
    arrival_settlement = models.ForeignKey('Settlement', verbose_name=_(u'Город прибытия'),
                                           related_name='arrival_facedirections')
    ttype = models.ForeignKey('TransportType', verbose_name=_(u'Тип транспорта'))
    text = models.CharField(max_length=255, verbose_name=_(u'Тескт'), null=True, blank=True)
    url = models.TextField(_(u'Ссылка'), null=True, blank=True)

    def __unicode__(self):
        return u'%s - %s (%s)' % (self.departure_settlement.title, self.arrival_settlement.title, self.ttype.L_title())

    def L_title(self):
        if self.text:
            return self.text

        if translation.get_language() == 'ru':
            return xformat(u'на <settlement-to-title case="accusative"/>',
                           settlement_to_title=self.arrival_settlement.L_title)

        return xgettext(
            u'<settlement-from-title/>&nbsp;&mdash; <settlement-to-title/>',
            settlement_from_title=self.departure_settlement.L_title,
            settlement_to_title=self.arrival_settlement.L_title
        )

    def title(self):
        return self.L_title()

    def link(self):
        if self.url:
            return self.url

        if self.ttype.code == 'urban':
            return '#'

        params = urlencode({
            'fromName': self.departure_settlement.title,
            'fromId': self.departure_settlement.point_key,
            'toName': self.arrival_settlement.title,
            'toId': self.arrival_settlement.point_key,
            'when': gettext(u'на все дни')
        })

        try:
            u = url('search', self.ttype.code)
        except:
            u = url('search')

        link = u + '?' + params

        return link

    class Meta:
        verbose_name = _(u'дополнительное направление')
        verbose_name_plural = _(u'дополнительные направления на морде города')
        app_label = 'www'
