# -*- coding: utf-8 -*-
from __future__ import absolute_import

import logging
from datetime import timedelta

from django.conf import settings
from django.utils.translation import gettext_noop as N_

from common.utils.caching import cache_method_result
from common.utils.date import RunMask, daterange
from common.utils.calendar_matcher import commaseparated_datelist, get_generator
from common.xgettext.common import xgettext_weekday_short, get_datetemplate_translation
from common.xgettext.i18n import xformat, stringify

from travel.rasp.admin.lib.exceptions import SimpleUnicodeException


log = logging.getLogger(__name__)


class TemplateMatcherException(SimpleUnicodeException):
    pass


class ThreadDataForExtrapolation(object):
    """
    Объект для хранения данных нитки, необходимых для экстраполяции
    """

    def __init__(self, run_days_in_half_year, last_date_of_full_mask, mask_is_extrapolatable,
                 dst_date, extrapolate_from, extrapolate_to):
        self.run_days_in_half_year = run_days_in_half_year
        self.last_date_of_full_mask = last_date_of_full_mask
        self.mask_is_extrapolatable = mask_is_extrapolatable
        self.dst_date = dst_date
        self.extrapolate_from = extrapolate_from
        self.extrapolate_to = extrapolate_to

    @property
    def key(self):
        key_parts = [
            str(RunMask(days=self.run_days_in_half_year)),
            unicode(self.last_date_of_full_mask),
            unicode(self.mask_is_extrapolatable),
            unicode(self.dst_date),
            unicode(self.extrapolate_from),
            unicode(self.extrapolate_to),
        ]

        return u'_'.join(key_parts)


class CysixTemplateMatcher(object):
    def __init__(self, templates, today, schedule_length, extrapolation_limit, min_match_days, max_mismatch_days):
        self.today = today
        self.templates = templates
        self.schedule_length = schedule_length
        self.extrapolation_limit = extrapolation_limit
        self.min_match_days = min_match_days
        self.max_mismatch_days = max_mismatch_days

        self.min_schedule_length_in_range = 8
        self.find_template_cache = {}
        self.find_template_exception_cache = {}

    def find_template(self, thread_data):
        """
        кэшируем результат выполнения метода, реальная работа в _find_template
        """

        key = thread_data.key
        try:
            return self.find_template_cache[key]
        except KeyError:
            pass

        if key in self.find_template_exception_cache:
            raise self.find_template_exception_cache[key]

        try:
            self.find_template_cache[key] = self._find_template(thread_data)
            return self.find_template_cache[key]
        except Exception as e:
            self.find_template_exception_cache[key] = e
            raise

    def _find_template(self, thread_data):
        """
        thread_data - объект класса ThreadDataForExtrapolation
        """

        template, mismatch, m_type = self._find_best_template(thread_data)

        extrapolable = self._can_extrapolate(thread_data, template, mismatch)

        template_params = self._fill_template_params(thread_data, template, mismatch, m_type, extrapolable)

        texts = self._generate_texts(template, template_params)

        return texts, template, extrapolable

    def _find_best_template(self, thread_data):
        # Последняя дата, учитываемая при наложении шаблонов
        last = thread_data.run_days_in_half_year[-1]

        range_days = self._get_cached_range_days(thread_data.extrapolate_from, last, include_end=True)
        run_days_in_range = [d for d in thread_data.run_days_in_half_year if d in range_days]
        if len(run_days_in_range) < self.min_match_days:
            raise TemplateMatcherException(N_(u'Слишком мало дней хождения'))

        schedule_length_in_range = (run_days_in_range[-1] - run_days_in_range[0]).days + 1  # + 1 т.к. включительно
        if schedule_length_in_range < self.min_schedule_length_in_range:
            raise TemplateMatcherException(N_(u'Слишком мало дней хождения'))

        template, mismatch, m_type = self._best_match(tuple(run_days_in_range), thread_data.extrapolate_from, last)

        if not template:
            raise TemplateMatcherException(N_(u'Не нашли подходящий шабон дней хождения'))

        if len(mismatch) > self.max_mismatch_days:
            raise TemplateMatcherException(N_(u'Слишком много выпадений из шаблона'))

        return template, mismatch, m_type

    def _can_extrapolate(self, thread_data, template, mismatch):
        if not thread_data.mask_is_extrapolatable:
            return False

        if mismatch:
            return False

        if thread_data.run_days_in_half_year[-1] < self.today + timedelta(self.schedule_length):
            post_days = set(daterange(thread_data.run_days_in_half_year[-1] + timedelta(1),
                                      self.today + timedelta(self.schedule_length)))
            # У маски шаблона есть дни хождения в диапазоне
            # (последний день хождения нитки, последний день достоверного расписания)
            if template.days & post_days:
                return False

        return True

    def _fill_template_params(self, thread_data, template, mismatch, m_type, extrapolable):
        params = {}

        if thread_data.extrapolate_from > self.today:
            pre_days = set(daterange(self.today, thread_data.extrapolate_from))

            # У маски шаблона есть дни хождения в диапазоне [сегодня, первый день хождения нитки)
            if template.days & pre_days:
                params['start'] = '%02d.%02d' % (thread_data.extrapolate_from.day, thread_data.extrapolate_from.month)

        if mismatch:
            if m_type == 'except':
                params['except'] = commaseparated_datelist(mismatch)
            else:
                params['extra'] = commaseparated_datelist(mismatch)

        if thread_data.mask_is_extrapolatable:
            if not extrapolable:
                params['end'] = '%02d.%02d' % (thread_data.last_date_of_full_mask.day,
                                               thread_data.last_date_of_full_mask.month)
            # еще может быть ситуация, когда уперлись в лимит экстраполяции
            if thread_data.dst_date is not None and thread_data.dst_date < self.extrapolation_limit:
                params['end'] = '%02d.%02d' % (thread_data.dst_date.day, thread_data.dst_date.month)
        elif thread_data.last_date_of_full_mask < thread_data.extrapolate_to:
            params['end'] = '%02d.%02d' % (thread_data.last_date_of_full_mask.day,
                                           thread_data.last_date_of_full_mask.month)

        return params

    def _generate_texts(self, template, params):
        texts = {}

        if params:
            key_tokens = []

            for key in ['start', 'end', 'except', 'extra']:
                if key in params:
                    key_tokens.append(key)

            template_key = "Matcher_%s" % ("_".join(key_tokens))

            for lang in settings.FRONTEND_LANGUAGES:
                params['main'] = template.get_text(lang)

                texts[lang] = stringify(xformat(get_datetemplate_translation(lang, template_key), **params))

        else:
            for lang in settings.FRONTEND_LANGUAGES:
                texts[lang] = template.get_text(lang)

        return texts

    @cache_method_result
    def _best_match(self, run_days, first, last):
        best = None
        best_mismatch = None
        best_m_type = None

        range_days = self._get_cached_range_days(first, last, include_end=True)
        run_days_set = set(run_days)
        for t in self.templates:
            # Дни маски, в которые рейсы не ходит
            except_ = (t.days - run_days_set).intersection(range_days)

            # Дни хождения, дополнительные к маске
            extra = (run_days_set - t.days).intersection(range_days)

            # Точное совпадение
            if not except_ and not extra:
                return t, set(), None

            # Есть и те и другие, нельзя сформировать строку
            if except_ and extra:
                continue

            # "кроме" уже есть в маске
            if t.has_except:
                continue

            if except_:
                mismatch = except_
                m_type = 'except'
            elif extra:
                mismatch = extra
                m_type = 'extra'

            if not best_mismatch or len(mismatch) < len(best_mismatch):
                best = t
                best_mismatch = mismatch
                best_m_type = m_type

        return best, best_mismatch, best_m_type

    @cache_method_result
    def _get_cached_range_days(self, first, last, include_end):
        return set(daterange(first, last, include_end=include_end))


def get_cysix_template_matcher(generator_name, today, start_date, length,
                               schedule_length, extrapolation_limit, min_match_days, max_mismatch_days):
    generator = get_generator(generator_name)

    return CysixTemplateMatcher(
        templates=generator(start_date, length),
        today=today,
        schedule_length=schedule_length,
        extrapolation_limit=extrapolation_limit,
        min_match_days=min_match_days,
        max_mismatch_days=max_mismatch_days,
    )
