# coding: utf8

from __future__ import absolute_import, unicode_literals

import logging
from django.db import connection, transaction

from common.apps.facility.models import SuburbanThreadFacility
from common.models.schedule import Route, RThread
from common.utils.iterrecipes import chunker
from travel.rasp.admin.lib.exceptions import FormattingException


log = logging.getLogger(__name__)


ID_CHUNK_SIZE = 10000
UID_CHUNK_SIZE = 3000


@transaction.atomic
def fast_delete_threads(qs, log=None):
    assert qs.model == RThread

    thread_ids = list(qs.values_list('id', flat=True))

    assert_has_no_live_changes(thread_ids)

    if not thread_ids:
        return

    if log is not None:
        log.info('Удаляем нитки')

    for chunk_ids in chunker(thread_ids, ID_CHUNK_SIZE):
        delete_threads_from_z_tables(chunk_ids)
        _delete_thread_and_rtstation(chunk_ids)

    if log is not None:
        log.info('Нитки удалены')


@transaction.atomic
def fast_delete_threads_with_routes(thread_qs, keep_empty_script_protected_routes=True):
    affected_route_ids = {t.route_id for t in thread_qs}
    fast_delete_threads(thread_qs)

    routes_qs = Route.objects.filter(rthread__isnull=True, pk__in=affected_route_ids)
    if keep_empty_script_protected_routes:
        routes_qs = routes_qs.exclude(script_protected=True)

    routes_qs.delete()


@transaction.atomic
def fast_delete_package_threads_and_routes(two_stage_package, log=None):
    routes = list(Route.objects.filter(two_stage_package_id=two_stage_package.id).values_list('id', 'script_protected'))
    affected_route_ids = [route_id for route_id, script_protected in routes if not script_protected]
    excluded_route_ids = [route_id for route_id, script_protected in routes if script_protected]

    affected_thread_ids = []
    for chunk_route_ids in chunker(affected_route_ids, ID_CHUNK_SIZE):
        affected_thread_ids += list(RThread.objects.filter(route_id__in=chunk_route_ids).values_list('id', flat=True))

    if not affected_thread_ids:
        return

    if log is not None:
        log.info('Удаляем нитки')

    cursor = connection.cursor()

    # Z tables
    del_znoderoute_sql = 'DELETE FROM www_znoderoute2 WHERE two_stage_package_id = %s'
    if excluded_route_ids:
        del_znoderoute_sql += 'AND route_id NOT IN {}'.format(_make_in_params(excluded_route_ids))
    cursor.execute(del_znoderoute_sql, [two_stage_package.id] + excluded_route_ids)

    for thread_ids_chunk in chunker(affected_thread_ids, ID_CHUNK_SIZE):
        SuburbanThreadFacility.objects.filter(thread__id__in=thread_ids_chunk).delete()

        in_params = _make_in_params(thread_ids_chunk)
        cursor.execute('DELETE FROM www_stationschedule WHERE thread_id IN {}'.format(in_params), thread_ids_chunk)

    # threads
    for thread_ids_chunk in chunker(affected_thread_ids, ID_CHUNK_SIZE):
        _delete_thread_and_rtstation(thread_ids_chunk)

    # routes
    del_route_sql = 'DELETE FROM www_route WHERE script_protected = 0 AND two_stage_package_id = %s'
    cursor.execute(del_route_sql, [two_stage_package.id])


@transaction.atomic
def fast_delete_supplier_threads_and_routes(supplier, log=None):
    routes = list(Route.objects.filter(supplier_id=supplier.id, two_stage_package_id=None)
                               .values_list('id', 'script_protected'))
    affected_route_ids = [route_id for route_id, script_protected in routes if not script_protected]
    excluded_route_ids = [route_id for route_id, script_protected in routes if script_protected]

    affected_thread_ids = []
    for chunk_route_ids in chunker(affected_route_ids, ID_CHUNK_SIZE):
        affected_thread_ids += list(RThread.objects.filter(route_id__in=chunk_route_ids).values_list('id', flat=True))

    if not affected_thread_ids:
        return

    if log is not None:
        log.info('Удаляем нитки')

    cursor = connection.cursor()

    # Z tables
    del_znoderoute_sql = 'DELETE FROM www_znoderoute2 WHERE supplier_id = %s AND two_stage_package_id is NULL'
    if excluded_route_ids:
        del_znoderoute_sql += 'AND route_id NOT IN {}'.format(_make_in_params(excluded_route_ids))
    cursor.execute(del_znoderoute_sql, [supplier.id] + excluded_route_ids)

    for thread_ids_chunk in chunker(affected_thread_ids, ID_CHUNK_SIZE):
        SuburbanThreadFacility.objects.filter(thread__id__in=thread_ids_chunk).delete()

        in_params = _make_in_params(thread_ids_chunk)
        cursor.execute('DELETE FROM www_stationschedule WHERE thread_id IN {}'.format(in_params), thread_ids_chunk)

    # threads
    for thread_ids_chunk in chunker(affected_thread_ids, ID_CHUNK_SIZE):
        _delete_thread_and_rtstation(thread_ids_chunk)

    # routes
    del_route_sql = ('DELETE FROM www_route WHERE script_protected = 0 AND'
                     ' supplier_id = %s AND two_stage_package_id is NULL')
    cursor.execute(del_route_sql, [supplier.id])

    if log is not None:
        log.info(u'Нитки удалены')


@transaction.atomic
def fast_delete_routes(qs, log=None):
    assert qs.model == Route
    route_ids = list(set(qs.values_list('id', flat=True)))

    if not route_ids:
        return

    if log is not None:
        log.info(u'Удаляем маршруты')

    cursor = connection.cursor()

    for chunk_route_ids in chunker(route_ids, ID_CHUNK_SIZE):
        fast_delete_threads(RThread.objects.filter(route__id__in=chunk_route_ids))

        del_sql = 'DELETE FROM www_route WHERE id IN {}'.format(_make_in_params(chunk_route_ids))

        cursor.execute(del_sql, chunk_route_ids)

    if log is not None:
        log.info(u'Маршруты удалены')


@transaction.atomic
def delete_threads_from_z_tables(thread_ids):
    in_params = _make_in_params(thread_ids)

    with connection.cursor() as cursor:
        cursor.execute('DELETE FROM www_znoderoute2 WHERE thread_id IN {}'.format(in_params), thread_ids)
        cursor.execute('DELETE FROM www_stationschedule WHERE thread_id IN {}'.format(in_params), thread_ids)

    SuburbanThreadFacility.objects.filter(thread__id__in=thread_ids).delete()


@transaction.atomic
def correct_z_tables(cursor):
    cursor.execute("""DELETE z FROM www_znoderoute2 z LEFT JOIN www_rthread t ON t.id = z.thread_id
                        WHERE t.id IS NULL""")
    cursor.execute("""DELETE z FROM www_znoderoute2 z LEFT JOIN www_route r ON r.id = z.route_id
                    WHERE r.hidden = 1 OR r.id IS NULL""")

    cursor.execute("""DELETE ss FROM www_stationschedule ss LEFT JOIN www_route r ON r.id = ss.route_id
                    WHERE r.hidden=1 OR r.id IS NULL
                    """)
    cursor.execute("""DELETE ss FROM www_stationschedule ss LEFT JOIN www_rthread t ON t.id = ss.thread_id
                    WHERE t.id IS NULL""")

    cursor.execute("""DELETE rni FROM www_routenumberindex rni
                    LEFT JOIN www_route r ON r.id = rni.route_id
                    WHERE r.id IS NULL""")


@transaction.atomic
def fast_delete_tariffs_by_uids(thread_uids):
    thread_uids = list(thread_uids)

    with connection.cursor() as cursor:
        for chunk_uids in chunker(thread_uids, UID_CHUNK_SIZE):
            del_sql = 'DELETE FROM www_threadtariff WHERE thread_uid IN {}'.format(_make_in_params(chunk_uids))
            cursor.execute(del_sql, chunk_uids)


def _delete_thread_and_rtstation(thread_ids):
    with connection.cursor() as cursor:
        cursor.execute(
            'DELETE FROM www_rtstation WHERE thread_id IN {}'.format(_make_in_params(thread_ids)),
            thread_ids
        )
        cursor.execute(
            'DELETE FROM www_trainpurchasenumber WHERE thread_id IN {}'.format(_make_in_params(thread_ids)),
            thread_ids
        )
        cursor.execute(
            'DELETE FROM www_rthread WHERE id IN {}'.format(_make_in_params(thread_ids)),
            thread_ids
        )


def _make_in_params(elements):
    return '({})'.format(','.join(['%s'] * len(elements)))


class HasLiveChangesError(FormattingException):
    pass


def assert_has_no_live_changes(thread_ids):
    changes_ids = set()

    for chunk in chunker(thread_ids, ID_CHUNK_SIZE):
        changes_ids.update(RThread.objects.filter(basic_thread__in=chunk).values_list('pk', flat=True))

    if changes_ids - set(thread_ids):
        raise HasLiveChangesError('Остануться нитки изменения ссылающиеся на удаленные базовые нитки')
