# -*- coding: utf-8 -*-

import pytz
import logging
from datetime import datetime, timedelta, date

from django.conf import settings
from django.conf.urls import url
from django.core.exceptions import ValidationError
from django.db import models
from django.utils.functional import cached_property
from django.utils.translation import ugettext_lazy as _

from common.models_abstract.schedule import BaseRThread, BaseSupplier, BaseRoute
from travel.rasp.library.python.common23.date import environment
from common.utils.date import RunMask, get_local_time

from travel.rasp.admin.lib.adminutils import get_rasp_admin_url


log = logging.getLogger(__name__)


class Supplier(BaseSupplier):
    add_to_route_changes_report = models.BooleanField(_(u'Добавлять в статистику по измененным рейсам'), default=True)

    class Meta(BaseSupplier.Meta):
        app_label = 'www'
        db_table = 'www_supplier'


class RThread(BaseRThread):
    """ Нитка - группа вагонов поезда, следующая неразрывно или
        Другая модель самолёта.
    """
    # при размере max_allow_packet 32MB это оптимальный вариант,
    # учитывая, что длина import_uid ~ 100-200.
    # Итого запрос будет порядке 20Мб
    IMPORT_UID_CHUNK_SIZE = 100000

    path_key = models.TextField(_(u"Ключ пути"), default=None, null=True, blank=True)

    def get_basic(self):
        from common.models.schedule import RThreadType

        if self.type_id == RThreadType.get_by_code(code="basic").id:
            pass

    def gen_title(self, strategy=None):
        u"""
        Генерация title, short_title на основе маршрута следования
        """

        from common.utils.title_generator import TitleGenerator

        generator = TitleGenerator(self, strategy)
        generator.generate()

        self.title = generator.title
        self.is_manual_title = False
        self.title_short = generator.title_short
        self.title_common = generator.title_common

        self.title_rtstations = generator.title_rtstations

    def gen_uid(self):
        from travel.rasp.admin.www.utils.uids import gen_thread_uid

        self.uid = gen_thread_uid(self)

        return self.uid

    def gen_route_uid(self, **kwargs):
        from travel.rasp.admin.www.utils.uids import gen_route_uid

        return gen_route_uid(self, **kwargs)

    def gen_import_uid(self):
        from travel.rasp.admin.www.utils.uids import gen_thread_uuid

        self.import_uid = gen_thread_uuid(self)

        return self.import_uid

    def get_type(self):
        return self.type.title

    get_type.short_description = _(u"Тип нитки")

    def get_model(self):
        return self.t_model.title if self.t_model else None

    get_model.short_description = _(u"Модель")

    def clean(self, *args, **kwargs):
        # Убедимся что для интервальной нитки прописаны время начала и конца интервала
        if self.type_id == settings.INTERVAL_RTHREAD_TYPE_ID:
            if self.begin_time is None or self.end_time is None:
                raise ValidationError(_(u'У интервальной нитки должно быть указано время начала и конца интервала'))

        # Убедимся что для интервальной нитки начало интервала не больше конца интервала
        if self.type_id == settings.INTERVAL_RTHREAD_TYPE_ID and self.begin_time > self.end_time:
            raise ValidationError(_(u'Интервал должен быть положительным (время начала меньше времени конца)'))

        super(RThread, self).clean(*args, **kwargs)

    def save(self, **kwargs):
        if not self.import_uid:
            raise ValueError('RThread must have import_uid')

        if not self.uid:
            raise ValueError('RThread must have uid')

        if self.id and not self.title and self.rtstation_set.count() > 2:
            self.gen_title()

        # сохраняем перевенутый номер, используется для поиска по подстроке
        self.reversed_number = self.number[::-1]

        super(RThread, self).save(**kwargs)

    def runs_at(self, dt):
        if not self.year_days:
            return False

        if len(self.year_days) != 372:
            log.error('Bad runmask %r' % self.year_days)
            return False

        return self.year_days[(dt.day - 1) + (dt.month - 1) * 31] == '1'

    def first_run(self, today):
        u"""Вычисляет первый или последний день хождения"""

        year_days = self.year_days

        if len(year_days) != 372:
            log.error('Bad runmask %r' % year_days)
            return None

        def safe_date(year, month, day):
            try:
                return date(year, month, day)
            except ValueError:
                assert (month == 2 and day == 29)

                return date(year, 3, 1)

        # Ищем первую дату хождения после сегодняшнего дня
        def first():
            day_position = (today.month - 1) * 31 + (today.day - 1)

            year = today.year

            try:
                first_day = year_days.index('1', day_position)
            except ValueError:
                year += 1

                try:
                    first_day = year_days.index('1')
                except ValueError:
                    return None

            month0, day0 = divmod(first_day, 31)

            return safe_date(year, month0 + 1, day0 + 1)

        def last():
            day_position = (today.month - 1) * 31 + (today.day - 1)

            year = today.year

            try:
                last_day = year_days.rindex('1', 0, day_position)
            except ValueError:
                year -= 1

                try:
                    last_day = year_days.rindex('1')
                except ValueError:
                    return None

            month0, day0 = divmod(last_day, 31)

            return safe_date(year, month0 + 1, day0 + 1)

        first_run = first()

        if first_run is None:
            return None

        if first_run - today > timedelta(365 - settings.DAYS_TO_PAST):
            first_run = last()

        return first_run

    def get_mask(self, today=None):
        mask = RunMask(self.year_days, today=today)

        return mask

    def get_rtstations(self):
        try:
            # при импортах этот аттрибут заполняется
            return self.rtstations
        except AttributeError:
            return list(self.rtstation_set.all().select_related('station'))

    def get_stations(self):
        from common.models.schedule import RTStation

        return [rs.station for rs in RTStation.objects.filter(thread__id=self.id).order_by('id')]

    def calc_mask_shift(self, timezone, start_date=None):
        """
        На сколько нужно сдвинуть маску вправо, чтобы получить маску в зоне timezone.
        """

        if not start_date:
            today = environment.today()
            mask_start_date = self.first_run(today) or today

        else:
            mask_start_date = start_date

        naive_start_dt = datetime.combine(mask_start_date, self.tz_start_time)
        start_dt = self.pytz.localize(naive_start_dt)

        timezone_start_dt = start_dt.astimezone(pytz.timezone(timezone))

        return (timezone_start_dt.date() - naive_start_dt.date()).days

    def get_short_title(self):
        return self.title_short or self.title

    def get_change_date_list(self):
        u""" Список дней по которым пришли нитки изменения"""
        changes = set()
        for t in self.thread_changes.all():
            changes.update(set(t.get_run_date_list()))
        changes = list(changes)
        changes.sort()
        return changes

    def get_original_mask(self):
        u'Маска нитки включая изменения'
        mask = RunMask(self.year_days)
        for t in self.thread_changes.all():
            mask |= RunMask(t.year_days)
        return mask

    def __unicode__(self):
        return self.number

    class BadStations(Exception):
        pass

    def get_extreme_stations(self):
        u"""Возвращает первую и последнюю станции маршрута"""
        return self.rtstation_set.all()[0].station, self.rtstation_set.all().reverse()[0].station

    def get_admin_url(self):
        return get_rasp_admin_url(self)

    def get_absolute_url(self, departure=None):
        rv = url('thread', self.uid)

        if departure:
            rv += departure.strftime("?departure=%Y-%m-%d")

        return rv

    class Meta(BaseRThread.Meta):
        app_label = 'www'


class Route(BaseRoute):
    """ Рейс """

    triangle_package = models.ForeignKey('importinfo.TriangleBusImportPackage', null=True,
                                         default=None, blank=True)
    two_stage_package = models.ForeignKey('importinfo.TwoStageImportPackage',
                                          null=True, default=None, blank=True)
    red_metaroute = models.ForeignKey('red.MetaRoute', null=True, default=None, blank=True)

    @property
    def deluxe(self):
        return self.deluxe_train.deluxe if self.deluxe_train else False

    @property
    def express(self):  # FIXME: назвать свойство is_express
        return self.express_type == 'express'

    @property
    def aeroexpress(self):  # FIXME: назвать свойство is_aeroexpress
        return self.express_type == 'aeroexpress'

    def save(self, **kwargs):
        if self.supplier_id is None:
            raise ValueError('Route must have supplier_id')

        if not self.route_uid:
            raise ValueError('Route must have route_uid')

        super(Route, self).save(**kwargs)

    def get_thread_by_rundate(self, local_date=None, msk_date=None):
        u"""
        @param self: маршрут.
        @param local_date: дата старта местная.
        @param msk_date: дата старта московская.
        По маршруту и дате отправления определяет первую попавшуюся нитку,
        которая ходит в этот день.
        local_date и msk_date не могут быть одновременно None.
        """
        assert local_date is not None or msk_date is not None, \
            "local_date and msk_date can not be None simultaneously"

        def get_local_datetime(run_date):
            msk_datetime = datetime.combine(run_date, thread.start_time)
            local_datetime = get_local_time(first_station, msk_datetime).replace(tzinfo=None)
            return local_datetime

        for thread in self.rthread_set.order_by('type__id'):
            from common.models.schedule import RTStation
            try:
                first_station = thread.rtstation_set.get(arrival=None,
                                                         departure=0).station
            except RTStation.DoesNotExist:
                log.error(u"У нитки %s нет начальной станции берем первую в списке станций",
                          thread.uid)
                first_station = thread.rtstation_set.all()[0].station

            if msk_date:
                if thread.runs_at(msk_date):
                    return thread
                continue

            run_date = local_date
            assumed_local_date = get_local_datetime(run_date).date()
            if assumed_local_date > local_date:
                # Часовое смещение не может быть больше чем сутки
                run_date = local_date - timedelta(1)

            elif assumed_local_date < local_date:
                # Часовое смещение не может быть больше чем сутки
                run_date = local_date + timedelta(1)

            # Проверяем хождение
            if thread.runs_at(run_date):
                return thread

    def get_all_stored_thread_ordinal_numbers(self):
        return self.rthread_set.order_by().values_list('ordinal_number', flat=True)

    def has_script_protected_copy_in_base(self):
        return bool(Route.objects.filter(route_uid=self.route_uid,
                                         script_protected=True))

    @cached_property
    def any_thread(self):
        try:
            return RThread.objects.filter(route=self)[0]
        except IndexError:
            return None

    @property
    def route_number(self):
        return self.any_thread.number if self.any_thread else u''

    @property
    def route_title(self):
        return self.any_thread.title if self.any_thread else u''

    class Meta(BaseRoute.Meta):
        app_label = 'www'
        db_table = 'www_route'
