# -*- coding: utf-8 -*-

from __future__ import absolute_import

import logging

from django.utils import translation

from common.models_utils.i18n import RouteLTitle
from route_search.by_number.helpers import number_variants
from stationschedule.models_abstract import BaseZTablo2


log = logging.getLogger(__name__)


class ZTablo2(BaseZTablo2):
    class Meta(BaseZTablo2.Meta):
        db_table = 'z_tablo2'
        app_label = 'www'

    def fill_search_strings(self, opposite_station=None):
        RouteLTitle.fetch([self.L_tablo_title] + [self.thread.L_title] if self.thread_id else [])

        opposite_station = opposite_station or self.next_station

        for lang in self.FOR_SEARCH_LANGS:
            with translation.override(lang):
                setattr(self, 'for_search_' + lang, self.gen_search_string(
                    opposite_station=opposite_station
                ))

    def gen_search_string(self, opposite_station):
        return u'|'.join(self.gen_search_fields(opposite_station))

    def gen_search_fields(self, opposite_station):
        from common.models.transport import TransportType

        title = self.L_title()
        number = self.get_number()
        company = self.get_company()

        if number and title:
            yield u'%s %s' % (number, title)
        elif number:
            yield number
        elif title:
            yield title

        if company:
            yield company.L_title()

        if self.t_type_id == TransportType.PLANE_ID:
            if opposite_station:
                yield opposite_station.L_title()

                if opposite_station.country_id:
                    yield opposite_station.country.L_title()

        yield u'|'  # Эта палка влияет на логику отображения

        if number:
            for number in number_variants(number):
                yield number

    def get_opposite_station(self, thread):
        """
        Если это начальная точка рейса, вернуть конечную, иначе - начальную
        """
        from common.models.transport import TransportType
        from common.models.schedule import RTStation

        thread = thread or self.thread

        if not (thread and thread.id):
            return

        if self.t_type_id != TransportType.PLANE_ID:
            return

        try:
            if self.departure:

                return RTStation.objects.get(thread=thread, tz_departure__isnull=True).station
            else:
                return RTStation.objects.get(thread=thread, tz_arrival__isnull=True).station
        except RTStation.DoesNotExist:
            return

    def get_number(self):
        return self.thread.number if self.thread_id else self.number

    def get_company(self):
        return self.thread.company if self.thread_id else self.company
