# -*- coding: utf-8 -*-

import json
import logging
from datetime import datetime
from collections import namedtuple

from django.utils.encoding import smart_unicode
from django.utils.translation import ugettext_lazy as _

from common.models_abstract.tariffs import BaseThreadTariff
from common.utils.date import timedelta2minutes


log = logging.getLogger(__name__)


class ThreadTariff(BaseThreadTariff):
    class Meta(BaseThreadTariff.Meta):
        app_label = 'www'
        verbose_name = _(u'тариф нитка-точка-точка')
        verbose_name_plural = _(u'тарифы нитка-точка-точка')

    def set_order_data(self, value):
        self.data = self.clean_order_data(value)

    def set_cleaned_order_data(self, value):
        self.data = value

    @classmethod
    def clean_order_data(cls, data):
        if isinstance(data, basestring):
            data = json.loads(data)
        else:
            data = data or {}

        return smart_unicode(
            json.dumps(data, separators=(',', ':'), sort_keys=True, ensure_ascii=False)
        )

    @classmethod
    def create_tariff_from_fare(cls, thread, first_run, rts_from, rts_to, tariff_mask, fare):
        thread_tariff = ThreadTariff(
            thread_uid=thread.uid,
            station_from_id=rts_from.station_id,
            station_to_id=rts_to.station_id,
            settlement_from_id=rts_from.station.settlement_id,
            settlement_to_id=rts_to.station.settlement_id,
            supplier_id=thread.supplier_id,
            number=thread.number or thread.hidden_number,
            t_type_id=thread.t_type_id,
            year_days=str(tariff_mask),
            time_zone_from=rts_from.time_zone,
            time_zone_to=rts_to.time_zone,
            tariff=fare.tariff,
            currency=fare.currency,
            url=fare.url,
            is_min_tariff=fare.is_min_tariff,
        )

        thread_tariff.set_cleaned_order_data(fare.cleaned_order_data)

        shift = rts_from.calc_days_shift(event='departure', start_date=first_run, event_tz=rts_from.pytz)
        thread_tariff.year_days_from = str(tariff_mask.shifted(shift))
        naive_start_dt = datetime.combine(first_run, thread.tz_start_time)
        departure_dt = rts_from.get_departure_dt(naive_start_dt, rts_from.pytz)
        arrival_dt = rts_to.get_arrival_dt(naive_start_dt, rts_to.pytz)
        thread_tariff.time_from = departure_dt.time()
        thread_tariff.time_to = arrival_dt.time()
        thread_tariff.duration = int(timedelta2minutes(arrival_dt - departure_dt))

        return thread_tariff

    class Fare(namedtuple('Fare', [
        'tariff',
        'currency',
        'url',
        'cleaned_order_data',
        'is_min_tariff'
    ])):

        __slots__ = ()

        @classmethod
        def create(cls, tariff, currency=None, url=u'', order_data=None, is_min_tariff=False):
            from common.models.currency import Currency

            currency = currency or Currency.BASE_CURRENCY

            cleaned_order_data = ThreadTariff.clean_order_data(order_data)

            return cls(tariff, currency, url, cleaned_order_data, is_min_tariff)

        def __repr__(self):
            return (u'<Fare: %s %s>' % (self.tariff, self.currency)).encode('utf8')
