from datetime import date, timedelta
from StringIO import StringIO

import pandas as pd
import requests
from openpyxl import load_workbook, Workbook


def get_clickhouse_df(query, click_config, connection_timeout=1500):
    result = requests.post(
        click_config['host'],
        auth=(
            click_config['user'],
            click_config['password'],
        ),
        params={'query': query},
        timeout=connection_timeout,
    )
    result.raise_for_status()
    return pd.read_csv(StringIO(result.text), sep='\t')


def get_blablacar_data(start_date, end_date, click_config):
    query = '''
    SELECT
        toDate(EventTime) as day,
        countIf(WatchID, (CounterID = 99704)) as desktop,
        countIf(WatchID, (CounterID = 22352497)) as touch,
        countIf(WatchID, match(URL,
            '.*comuto_cmkt=[^&$]+(2550|5075).*')) as range_up_75,
        count(WatchID) - range_up_75 as range_over_75,
        range_up_75 + range_over_75 as total_clicks,
        range_up_75 * 0.055 as Euro_up_75,
        range_over_75 * 0.09 as Euro_over_75,
        Euro_up_75 + Euro_over_75 as Euro
    FROM hits_all

    WHERE
        (EventDate >= '{}') AND (EventDate < '{}')
        AND (CounterID IN (99704,22352497))
        AND URL like '%blablacar%search%'
        AND DontCountHits = 1

    GROUP BY day
    ORDER BY day

    FORMAT TabSeparatedWithNames'''.format(start_date, end_date)

    return get_clickhouse_df(query, click_config)


def append_df_to_excel(filename, df, sheet_name='Sheet1', startrow=None,
                       truncate_sheet=False,
                       **to_excel_kwargs):

    # ignore [engine] parameter if it was passed
    if 'engine' in to_excel_kwargs:
        to_excel_kwargs.pop('engine')

    writer = pd.ExcelWriter(filename, engine='openpyxl')

    # Python 2.x: define [FileNotFoundError] exception if it doesn't exist
    try:
        FileNotFoundError
    except NameError:
        FileNotFoundError = IOError

    try:
        # try to open an existing workbook
        writer.book = load_workbook(filename)

        # get the last row in the existing Excel sheet
        # if it was not specified explicitly
        if startrow is None and sheet_name in writer.book.sheetnames:
            startrow = writer.book[sheet_name].max_row

        # truncate sheet
        if truncate_sheet and sheet_name in writer.book.sheetnames:
            # index of [sheet_name] sheet
            idx = writer.book.sheetnames.index(sheet_name)
            # remove [sheet_name]
            writer.book.remove(writer.book.worksheets[idx])
            # create an empty sheet [sheet_name] using old index
            writer.book.create_sheet(sheet_name, idx)

        # copy existing sheets
        writer.sheets = {ws.title: ws for ws in writer.book.worksheets}
    except FileNotFoundError:
        # file does not exist yet, we will create it
        pass

    if startrow is None:
        startrow = 0

    # write out the new sheet
    df.to_excel(writer, sheet_name, startrow=startrow, **to_excel_kwargs)

    # save the workbook
    writer.save()


def send_email_with_attachment(subject, body_text,
                               to_emails, cc_emails, file_to_attach, human_file_name,
                               email_host, from_email):
    """
    Send an email with an attachment
    """
    import smtplib
    from email import encoders
    from email.mime.text import MIMEText
    from email.mime.base import MIMEBase
    from email.mime.multipart import MIMEMultipart
    from email.utils import formatdate
    header = 'Content-Disposition', 'attachment; filename="{}"'.format(human_file_name)

    # create the message
    msg = MIMEMultipart()
    msg["From"] = from_email
    msg["Subject"] = subject
    msg["Date"] = formatdate(localtime=True)

    if body_text:
        msg.attach(MIMEText(body_text, 'html', _charset="utf-8"))

    msg["To"] = ', '.join(to_emails)
    msg["cc"] = ', '.join(cc_emails)

    attachment = MIMEBase('application', "octet-stream")

    try:
        with open(file_to_attach, "rb") as fh:
            data = fh.read()

        attachment.set_payload(data)
        encoders.encode_base64(attachment)
        attachment.add_header(*header)
        msg.attach(attachment)
    except IOError:
        msg = "Error opening attachment file %s" % file_to_attach
        print(msg)

    emails = to_emails + cc_emails
    server = smtplib.SMTP(email_host)
    server.sendmail(from_email, emails, msg.as_string())
    server.quit()


def do_report_send(tmp_file_name, list_to_emails, list_cc_emails,
                   click_user, click_pass, click_host='http://clickhouse.metrika.yandex.net:8123/', today=date.today()):

    if today.day > 7:
        month_start = today.replace(day=1)
    else:
        month_start = (today-timedelta(days=9)).replace(day=1)
    human_file_name = 'BlaBlaCar_report_{}.xlsx'.format(today)

    click_config = {
        'user': click_user,
        'password': click_pass,
        'host': click_host,
    }

    click_df = get_blablacar_data(start_date=month_start, end_date=today, click_config=click_config)
    summed_click = click_df.sum()
    summed_click['day'] = 'SUM'
    df = click_df.append(summed_click, ignore_index=True)

    wb = Workbook()
    ws = wb.active
    ws.title = "Report"
    ws['A1'] = 'The report for the period from {} for {}'.format(month_start, today-timedelta(days=1))
    ws['A2'] = 'For partner: blablacar'
    ws['A3'] = 'Report title: chart for rasp.yandex - external transitions'
    ws['A4'] = 'Group by day'
    ws['A5'] = 'Attribution: the Last click'
    wb.save(tmp_file_name)

    append_df_to_excel(tmp_file_name, df, sheet_name='Report', startrow=7, index=False)

    subject = 'Yandex.Raspisanie: BlaBlaCar Report_{}'.format(today)
    body_text = '''Добрый день <br>
Во вложении еженедельный отчет по внешним переходам на сайт blablacar* с сайтов
rasp.yandex.* и t.rasp.yandex.* за период с {} по {}
    '''.format(month_start, today)

    email_host = 'outbound-relay.yandex.net'
    from_email = 'noreply@yandex-team.ru'

    send_email_with_attachment(
        subject, body_text, list_to_emails, list_cc_emails,
        tmp_file_name, human_file_name, email_host, from_email,
    )
