# -*- coding: utf-8 -*-

from itertools import chain

from django.conf import settings
from django.http import HttpResponseRedirect
from django.utils import translation

from common.xgettext.i18n import gettext

import travel.rasp.api_public.api_public.v3.core.apikeys as apikeys
from travel.rasp.api_public.api_public.v3.core.api_errors import ApiError
from travel.rasp.api_public.api_public.v3.views.error_view import error_view


def is_new_path(path):
    return path.startswith('/v3') or path.startswith('/partners')


def error_view_on_failure(func):
    def wrapper(*args, **kwargs):
        try:
            func(*args, **kwargs)
        except ApiError as ex:
            response = error_view(args[1], ex)
            response.render()
            return response
    return wrapper


class ApiLanguage(object):

    @error_view_on_failure
    def process_request(self, request):
        """
        Язык в v1 и v2 указывается в формате lang=ru.
        В v3 и partners в формате lang=ru_RU,
        где ru определяет язык выдачи,
        а RU - код страны, который определяет региональные особенности.
        """
        lang_info_str = request.GET.get('lang', None)

        if not is_new_path(request.path):
            lang = lang_info_str
            if not lang or (lang not in settings.API_LANGUAGES):
                lang = 'ru'
        else:
            if not lang_info_str:
                lang, lang_region = 'ru', 'RU'
            else:
                try:
                    lang, lang_region = lang_info_str.split('_')
                except ValueError:
                    raise ApiError(gettext(u'Указанный формат параметра lang - {} не поддерживается. '
                                           u'Поддерживаемый формат lang=language_region, '
                                           u'language — код языка в формате ISO 639, '
                                           u'region — код страны в формате ISO 3166. '
                                           u'Пример: ru_RU.'.format(lang_info_str)), 400)

            lang, lang_region = lang.lower(), lang_region.upper()

            if lang not in settings.API_LANGUAGES:
                raise ApiError(gettext(u'Указанный язык {} не поддерживается.'.format(lang)), 400)

            try:
                request.tld = settings.COUNTRY_ISO_TO_TLD[lang_region]
            except KeyError:
                raise ApiError(gettext(u'Указанный код страны {} не поддерживается.'.format(lang_region)), 400)

            country_to_nv = settings.COUNTRY_ISO_TO_NATIONAL_VERSION
            request.national_version = country_to_nv.get(lang_region, country_to_nv['RU'])

        translation.activate(lang)


class TuneDummy(object):
    """tune_host нужен геотергетингу, но функционал
    locale_request.Locale здесь не нужен"""

    def process_request(self, request):
        request.tune_host = ''


class RedirectToTerminalSlash(object):
    def process_request(self, request):
        if (request.method == 'GET' and is_new_path(request.path)) and not request.path.endswith('/'):
            urlencode = request.GET.urlencode()
            full_path = request.path + '/?' + urlencode if urlencode else request.path + '/'
            return HttpResponseRedirect(full_path)


class CheckKeyValid(object):
    ignore_paths = ['/version', '/ping', '/ping/']

    @error_view_on_failure
    def process_request(self, request):
        if not any(request.path == path for path in self.ignore_paths):
            request.valid_key = False
            apikeys.check_key(request)

            if request.path.startswith('/partners'):
                self.check_partners_key(request)

            request.valid_key = True

    def process_response(self, request, response):
        if hasattr(request, 'valid_key') and request.valid_key:
            size_bytes = len(response.content)
            apikeys.update_counter(request, hits=1, bytes=size_bytes)
        return response

    def check_partners_key(self, request):
        key = apikeys.get_key(request)
        if key in chain(VALID_KEYS_BY_URL.get(request.path, []), VALID_KEYS_BY_URL['testing_keys']):
            return
        else:
            raise ApiError(gettext(u'Указан неверный ключ "{}"').format(key), http_code=400)


VALID_KEYS_BY_URL = {
    '/partners/bzd/basic_threads_uids/': [

    ],
    '/partners/avia/groups_search': [
        'b9c7281a-6ce2-4806-ad61-22acb27ee58a',
    ],
    '/partners/autobus/points_info/': [
        '75d70b90-8d11-4f1e-b0d1-f6d62f16c38f',
    ],
    '/partners/autobus/geo_id_to_rasp_id/': [
        '75d70b90-8d11-4f1e-b0d1-f6d62f16c38f',
    ],
    'testing_keys': [
        '8eb91582-941a-4ac8-a618-040c314495ab',
        '559cdea5-2b77-4152-adf5-f6c600f2ebe2',
        'd1ded6a3-11cc-4247-99dd-5b445831e28e',  # avia testing
    ]
}
